# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.common.nanny import nanny

TEAM = ["dieash"]


class TrustedNetworksSerpResource(sdk2.Resource):
    """File with trusted networks (CIDR notation) and hosts"""
    any_arch = True
    executable = False
    releasable = True
    releasers = TEAM


RESULT_FILE = 'TrustedNet.txt'
ARCADIA_GENACCESSIP_SCRIPT_PATH = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/antirobot/scripts/support/genaccessip.py'
ARCADIA_GENACCESSIP_SCRIPT_NAME = ARCADIA_GENACCESSIP_SCRIPT_PATH.rsplit('/', 1)[-1]
ARCADIA_TRUSTEDNET_M4_PATH = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/web/report/data/util/trusted_net/server/servernets-serp.m4'
ARCADIA_TRUSTEDNET_M4_NAME = ARCADIA_TRUSTEDNET_M4_PATH.rsplit('/', 1)[-1]


class TrustedNetworksCollectSerpTask(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Task to collect m4 file with TRUSTEDNET macros and resolve them into hosts and networks"""

    class Requirements(sdk2.Task.Requirements):
        # They say default DNS servers are able to resolve IPv4
        # But we can ensure that by the following lines
        # import sandbox.common.types.client as ctc
        # client_tags = ctc.Tag.IPV4
        pass

    def parse_macro(self, line, idx):
        if (line[:3] != 'or ') or (line[-4:] != ' dnl'):
            raise Exception("Incorrect format of macro line %d: '%s'" % (idx, line))
        return line[3:-4]

    def clean_m4_file(self, in_file_name, out_file_name):
        """
            Check m4 file format and remove m4 constructions
            It will fail if format of servernets-serp.m4 changed
        """
        trustednet_first_line = "define(`_SERP_TRUSTEDNETS_', `dnl"
        trustednet_second_line = "localhost dnl"
        trustednet_last_line = "')dnl"
        trustednet_after_last_line = "TRUSTEDNETS: _SERP_TRUSTEDNETS_"

        with open(in_file_name, 'r') as file:
            lines = [line.strip() for line in file.readlines()]

        try:
            idx = lines.index(trustednet_first_line) + 1
        except ValueError:
            raise Exception("Line '%s' is not found in %s. Format was probably changed." % (trustednet_first_line, ARCADIA_GENACCESSIP_SCRIPT_PATH))

        if idx >= len(lines):
            raise Exception("There is nothing after first trustednet line in %s. Format was probably changed." % ARCADIA_GENACCESSIP_SCRIPT_PATH)

        if lines[idx] != trustednet_second_line:
            raise Exception("Second line of %s was changed. Expected '%s'. Got: '%s'. Format was probably changed." % (ARCADIA_GENACCESSIP_SCRIPT_PATH, trustednet_second_line, lines[idx]))

        # Parse all macros in section
        idx += 1
        macros = []
        while idx < len(lines) and lines[idx] != trustednet_last_line:
            macro = self.parse_macro(lines[idx], idx)
            macros.append(macro)
            idx += 1
        idx += 1

        if idx >= len(lines):
            raise Exception("Last line '%s' is not found in %s. Format was probably changed." % (trustednet_last_line, ARCADIA_GENACCESSIP_SCRIPT_PATH))

        if lines[idx] != trustednet_after_last_line:
            raise Exception("Line after last trustednet line in %s was changed. Expected '%s'. Got: '%s'. Format was probably changed."
                            % (ARCADIA_GENACCESSIP_SCRIPT_PATH, trustednet_after_last_line, lines[idx]))

        with open(out_file_name, "w") as out_file:
            out_file.write("\n".join(macros))

    def check_output(self, file_name):
        with open(file_name, 'r') as file:
            lines = file.readlines()

        non_empty_lines_count = 0

        for line in lines:
            line = line.split('#', 1)[0].strip()
            if len(line) == 0:
                continue

            non_empty_lines_count += 1

        if non_empty_lines_count < 500:
            raise Exception("Too few non-empty lines in result: %d. Format was probably changed or something is broken." % non_empty_lines_count)

    def on_execute(self):
        sdk2.svn.Arcadia.export(ARCADIA_GENACCESSIP_SCRIPT_PATH, ARCADIA_GENACCESSIP_SCRIPT_NAME)
        sdk2.svn.Arcadia.export(ARCADIA_TRUSTEDNET_M4_PATH, ARCADIA_TRUSTEDNET_M4_NAME)

        self.clean_m4_file(in_file_name=ARCADIA_TRUSTEDNET_M4_NAME, out_file_name='tmp.txt')

        cmd = ['/usr/bin/env', 'python',
               ARCADIA_GENACCESSIP_SCRIPT_NAME,
               '--src-file', 'tmp.txt',
               '--out-file', RESULT_FILE
               ]
        subprocess.check_call(cmd)

        self.check_output(RESULT_FILE)

        resource = sdk2.ResourceData(TrustedNetworksSerpResource(
            self, "File with trusted networks (CIDR notation) and hosts", RESULT_FILE
        ))
        resource.ready()
