# coding: utf-8

import os
import __builtin__

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process

import sandbox.projects.lemur as resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.build.parameters as build_params


def target2field_name(target):
    return 'build_%s' % target


class LemurBinaries(parameters.SandboxInfoParameter):
    name = 'info_field'
    description = 'Lemur Binaries'


class CreateSymbolsArchive(parameters.SandboxBoolParameter):
    name = 'lemur_create_symbols_archive'
    description = 'Create separate tarball with debug symbols'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class StripBinaries(parameters.SandboxBoolParameter):
    name = 'lemur_strip_binaries'
    description = 'Strip result binaries'
    default_value = True
    group = build_params.BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuse(build_params.UseArcadiaApiFuse):
    default_value = True


class YtStore(build_params.YtStore):
    default_value = True


def init(target_resources, custom_input_parameters):
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        CreateSymbolsArchive,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
        build_params.DefinitionFlags
    ] + custom_input_parameters

    enabled_builds = {}
    for target_resource in target_resources:
        target = os.path.basename(target_resource.arcadia_build_path)
        field_name = target2field_name(target)
        build = __builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )
        input_parameters.extend([build])
        enabled_builds[field_name] = target_resource

    return input_parameters, enabled_builds


class BaseBuildLemurBins(YaMakeTask, nanny.ReleaseToNannyTask):
    type = 'BASE_BUILD_LEMUR_BINS'

    enabled_builds = {}
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        StripBinaries,
        build_params.Sanitize,
        build_params.ArcadiaPatch
    ]
    TARGET_RESOURCES = ()

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM
        self.ctx[StripBinaries.name] = False
        #self.ctx[CreateSymbolsArchive.name] = True

    def get_targets(self):
        targets = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                targets.extend([os.path.dirname(resource_type.arcadia_build_path)])
        return targets

    def get_resources(self):
        resources = {}
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                resource = {
                    'description': resource_type.name,
                    'resource_type': resource_type,
                    'resource_path': os.path.basename(resource_type.arcadia_build_path),
                }
                resources[resource_type.name] = resource

        return resources

    def get_arts(self):
        arts = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                arts.extend([{'path': resource_type.arcadia_build_path}])
        return arts

    def __copy_symbols(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--only-keep-debug', object_file, symbols_file],
            log_prefix='copy_symbols'
        )

    def __link_symbols_with_executable(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--add-gnu-debuglink', symbols_file, object_file],
            log_prefix='debuglink'
        )

    def pre_build(self, source_dir):
        if self.ctx[StripBinaries.name]:
            self.ctx[consts.STRIP_BINARIES] = True

        YaMakeTask.pre_build(self, source_dir)

    def on_execute(self):
        self.ctx['ya_yt_token_vault_owner'] = 'UKROP-ROBOT'
        self.ctx['ya_yt_token_vault_name'] = 'sandbox-build-yt-token'
        YaMakeTask.on_execute(self)

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

        if self.ctx.get(StripBinaries.name) or self.ctx.get(CreateSymbolsArchive.name):
            self.__create_symbols_resource(pack_dir)

    def __create_symbols_resource(self, pack_dir):
        create_symbols_archive = self.ctx[CreateSymbolsArchive.name]

        symbols_dir_name = 'symbols'
        symbols_dir = paths.make_folder(symbols_dir_name)

        for build_name, resource_type in self.enabled_builds.items():
            if not self.ctx.get(build_name):
                continue

            binary_name = os.path.basename(resource_type.arcadia_build_path)
            binary_path = os.path.join(pack_dir, binary_name)
            symbols_file = os.path.join(symbols_dir, binary_name + '.sym')

            if create_symbols_archive:
                self.__copy_symbols(binary_path, symbols_file)

            if create_symbols_archive:
                self.__link_symbols_with_executable(binary_path, symbols_file)

        if not create_symbols_archive:
            return

        symbols_archive_file = os.path.abspath('symbols.tar.gz')

        process.run_process(
            ['tar', 'czf', symbols_archive_file, symbols_dir_name],
            log_prefix='tar_symbols'
        )

        resource = self.create_resource(
            description='Build symbols',
            resource_path=symbols_archive_file,
            resource_type=resource_types.LEMUR_SYMBOLS_ARCHIVE,
            attributes={
                'ttl': 4
            }
        )

        self.mark_resource_ready(resource.id)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

    # Use name option to rename dir
    def _create_svn_resource(self, pathes, resource_name, resource_type, source_dir):
        print pathes, resource_name, resource_type

        resource_root = repr(resource_type)
        os.mkdir(resource_root)
        for path, prefix in pathes.items():
            source_path = os.path.join(source_dir, path)
            source_target = os.path.join(resource_root, prefix)

            try:
                os.makedirs(source_target)
            except Exception:
                pass
            run_process(['cp', '-r', source_path, source_target])
            # copy to source_dir

        run_process(['tar', '-C', resource_root, '-zcf', resource_name, '.'])
        self.create_resource(
            '',
            resource_name,
            resource_type)
