# coding: utf-8

import logging
import sandbox.projects.limbo as limbo_resources
import sandbox.projects.lemur as lemur_resources
import sandbox.projects.samovar as samovar_resources

from sandbox import sdk2
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import parameters
from sandbox.projects.limbo.common.base_bin_build import BaseBuildLimboBins
from sandbox.projects.limbo.common.base_bin_build import LimboBinaries
from sandbox.projects.limbo.common.base_bin_build import init


class BuildLimboPackage(BaseBuildLimboBins):
    '''
    Binaries, configs and scripts for Limbo.
    '''

    type = 'BUILD_LIMBO_PACKAGE'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 40 * 1024

    class IncludeSvnConfigs(parameters.SandboxBoolParameter):
        name = 'include_svn_configs'
        description = 'configs'
        default_value = True
        group = 'Limbo Files'

    class IncludeSvnScripts(parameters.SandboxBoolParameter):
        name = 'include_svn_scripts'
        description = 'scripts'
        default_value = True
        group = 'Limbo Files'

    class ReleaseToNanny(parameters.SandboxRadioParameter):
        choices = [(_, _) for _ in ('none', sandboxapi.RELEASE_STABLE)]
        description = "Release to Nanny."
        default_value = 'none'
        name = 'release_to_nanny'
        group = 'After Build Actions'

    TARGET_RESOURCES = (
        limbo_resources.LIMBO_LIMBO,
        limbo_resources.LIMBO_SAMOVAR_PREPARE_IMAGES_FACES_PUSHES,
        limbo_resources.LIMBO_SAMOVAR_PREPARE_USERDATA,
        limbo_resources.LIMBO_SAMOVAR_PREPARE_UBLP,
        limbo_resources.LIMBO_SAMOVAR_PARSE_SEARCH_LOG_IMAGES,
        limbo_resources.LIMBO_SAMOVAR_WATCH_LOG_MANAGER,
        limbo_resources.LIMBO_ZORACL,
        lemur_resources.LEMUR_PREPARE_REGEX_TEST_DATA,
        lemur_resources.LEMUR_GEMINICL,
        samovar_resources.SAMOVAR_PUSH_WRITER,
        samovar_resources.SAMOVAR_CONSUMER_TABLES_MERGER,
        samovar_resources.SAMOVAR_WIKIMEDIA_UPDATES_READER,
    )

    input_parameters, enabled_builds = init(TARGET_RESOURCES, [LimboBinaries])
    input_parameters = input_parameters + [
                                           IncludeSvnConfigs,
                                           IncludeSvnScripts,
                                           ReleaseToNanny
                                          ]

    def on_execute(self):
        BaseBuildLimboBins.on_execute(self)

        if self.ctx['release_to_nanny'] != 'none':

            check_already_released = False

            releaser_task = self.create_subtask(
                task_type='RELEASE_ANY',
                input_parameters={
                    'check_already_released': check_already_released,
                    'release_task_id': self.id,
                    'release_status': self.ctx['release_to_nanny'],
                },
                description="Limbo binaries (task id: {}) auto release. Revision: {}".format(str(self.id), self.ctx.get('arcadia_revision'))
            )
            logging.debug("Subtask {} runned, waiting for it's result.\n\n".format(releaser_task.id))

    def post_build(self, source_dir, output_dir, pack_dir):
        BaseBuildLimboBins.post_build(self, source_dir, output_dir, pack_dir)

        if self.ctx['include_svn_configs']:
            self._create_svn_resource({
                'yweb/robot/limbo/conf': ''
            }, 'svn_configs.tar.gz', limbo_resources.LIMBO_SVN_CONFIGS, source_dir)

        if self.ctx['include_svn_scripts']:
            self._create_svn_resource({
                'yweb/robot/limbo/scripts': '',
                'yweb/robot/limbo/libs': 'limbo',
                'yweb/robot/limbo/imports': 'limbo',
            }, 'scripts.tar.gz', limbo_resources.LIMBO_SVN_SCRIPTS, source_dir)


__Task__ = BuildLimboPackage
