import re
import logging
from datetime import datetime
from sandbox import sdk2
import sandbox.common.errors as ce

from sandbox.projects.logbroker import resources


logger = logging.getLogger(__name__)


def get_arcadia_url(path, revision=None):
    if path.startswith('/'):
        path = path[1:]
    if revision is None:
        return 'arcadia:/arc/trunk/arcadia/%s' % path
    return 'arcadia:/arc/trunk/arcadia/%s@%s' % (path, revision)


def get_unified_agent_version(ua_version_file, revision):
    url = get_arcadia_url(ua_version_file, revision)
    version = str(sdk2.svn.Arcadia.cat(url)).strip()
    logger.debug('got version string "%s"' % version)
    match = re.match(r'^(?P<year>\d+)\.(?P<month>\d+)\.(?P<tag>\d+)$', version)
    if not match:
        raise ce.TaskError('invalid version string "%s" in file %s, should be YY.M.tag' % (version, ua_version_file))
    version_data = match.groupdict()
    now = datetime.now()
    if not int(version_data['year']) == now.year - 2000:
        raise ce.TaskError('incorrect year specified in %s, should be %s' % (ua_version_file, now.year - 2000))
    if not int(version_data['month']) == now.month:
        raise ce.TaskError('incorrect month specified in %s, should be %s' % (ua_version_file, now.month))
    if not 1 <= int(version_data['tag']):
        raise ce.TaskError('incorrect tag specified in %s, should be greater than 0' % ua_version_file)
    unified_agent_bin = resources.UnifiedAgentBin.find(state='READY',
                                                       attrs={
                                                           'released': 'stable',
                                                           'unified_agent_version': version
                                                       }).first()
    if unified_agent_bin:
        raise ce.TaskError('version %s has been already released, update %s' % (version, ua_version_file))
    return version


class ResourceTypeId(sdk2.parameters.Dict, sdk2.parameters.String):
    name = 'resource_name_ver'


class ScriptEnv(sdk2.parameters.Dict, sdk2.parameters.String):
    name = 'env_dict'
