# -*- coding: utf-8 -*-

import logging
import math
import os
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.projects.common.constants as consts
# from sandbox.sandboxsdk.errors import SandboxTaskFailureError#, SandboxEnvironmentError

logger = logging.getLogger(__name__)

logbroker_releasers = [
    'artgromov',
    'alexnick',
    'komels',
    'gusev-p',
    'alexbogo',
    'robot-logbroker-ci',
]


class LOGBROKER_CLI_LINUX(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = logbroker_releasers
    arcadia_build_path = 'logbroker/public/client/bin'


class LOGBROKER_CLI_DARWIN(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = logbroker_releasers
    arcadia_build_path = 'logbroker/public/client/bin'


class LOGBROKER_CLI_WINDOWS(sdk2.resource.AbstractResource):
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = logbroker_releasers
    arcadia_build_path = 'logbroker/public/client/bin'


class BuildLogbrokerCliTask(sdk2.Task):

    PUBLISH_REPOSITORY = "search"

    class Context(sdk2.Task.Context):
        build_tasks = dict()

    class Parameters(sdk2.Parameters):
        arcadia_path = sdk2.parameters.String("arcadia path")

    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('boto'),
                        PipEnvironment('filechunkio')]

    def upload_binary(self, bucket, binary_path, store_path):
        from filechunkio import FileChunkIO
        source_size = os.stat(binary_path).st_size
        mp = bucket.initiate_multipart_upload(store_path)
        chunk_size = 52428800
        chunk_count = int(math.ceil(source_size / float(chunk_size)))
        for i in range(chunk_count):
            offset = chunk_size * i
            bytes_count = min(chunk_size, source_size - offset)
            with FileChunkIO(binary_path, 'r', offset=offset, bytes=bytes_count) as fp:
                mp.upload_part_from_file(fp, part_num=i + 1)
        mp.complete_upload()

    def on_execute(self):
        from boto.s3.connection import S3Connection
        ACCESS_KEY_ID = "5c3z52L6KIfcMugdW6cn"
        SECRET_ACCESS_KEY = sdk2.Vault.data("s3_logbroker_secret_key_upload")
        s3_path = {}
        s3_path["LOGBROKER_CLI_LINUX"] = "/cli/linux/logbroker"
        s3_path["LOGBROKER_CLI_DARWIN"] = "/cli/macos/logbroker"
        s3_path["LOGBROKER_CLI_WINDOWS"] = "/cli/windows/logbroker.exe"
        logger.info("Start task. Arcadia path is: {}".format(self.Parameters.arcadia_path))
        arcadia_package_path = "logbroker/public/client/pkg.json"
        key_ya_package = "_ya_package_task_id"
        key_ya_make_linux = "_ya_make_linux_task_id"
        key_ya_make_darwin = "_ya_make_darwin_task_id"
        tasks_started = False
        task_ids = {}
        tasks = {}
        task_ids["ya_package"] = self.Context.build_tasks.get(key_ya_package)
        if task_ids["ya_package"] is None:
            logger.info("Creating task to build deb package")
            tasks['ya_package'] = sdk2.Task["YA_PACKAGE"](
                self,
                description="Ya package created by {0}".format(self.id),
                checkout_arcadia_from_url=self.Parameters.arcadia_path,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                use_new_format=True,
                packages=arcadia_package_path,
                host_platform="linux",
                package_type="debian",
                publish_package=True,
                publish_to=self.PUBLISH_REPOSITORY,
                checkout_mode="auto",
                run_tests=True,
                run_long_tests=True
            )
            tasks['ya_package'].save().enqueue()
            self.Context.build_tasks[key_ya_package] = tasks['ya_package'].id
            self.Context.save()
            tasks_started = True
        task_ids["ya_make_linux"] = self.Context.build_tasks.get(key_ya_make_linux)
        if task_ids["ya_make_linux"] is None:
            logger.info("Creating task to build linux binary")
            tasks['ya_make_linux'] = sdk2.Task["YA_MAKE"](
                self,
                build_system="ya_force",
                description="Ya make linux util created by {0}".format(self.id),
                checkout_arcadia_from_url=self.Parameters.arcadia_path,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                host_platform="linux",
                targets="logbroker/public/client/bin",
                result_rt="LOGBROKER_CLI_LINUX",
                result_dt="Arcadia Project",
                result_single_file=True,
                build_type="release",
                target_platform_flags="--target-platform=linux",
                arts="logbroker/public/client/bin/logbroker",
                checkout_mode="manual",
                checkout=True,
                test_parameter=True
            )
            tasks['ya_make_linux'].save().enqueue()
            self.Context.build_tasks[key_ya_make_linux] = tasks['ya_make_linux'].id
            self.Context.save()
            tasks_started = True

        task_ids["ya_make_darwin"] = self.Context.build_tasks.get(key_ya_make_darwin)
        if task_ids["ya_make_darwin"] is None:
            logger.info("Creating task to build darwin binary")
            tasks['ya_make_darwin'] = sdk2.Task["YA_MAKE"](
                self,
                build_system="ya_force",
                description="Ya make darwin util created by {0}".format(self.id),
                checkout_arcadia_from_url=self.Parameters.arcadia_path,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                host_platform="linux",
                targets="logbroker/public/client/bin",
                result_rt="LOGBROKER_CLI_DARWIN",
                result_dt="Arcadia Project",
                result_single_file=True,
                build_type="release",
                target_platform="DEFAULT-DARWIN-X86_64",
                arts="logbroker/public/client/bin/logbroker",
                checkout_mode="manual",
                checkoout=True,
                test_parameter=True
            )
            tasks['ya_make_darwin'].save().enqueue()
            self.Context.build_tasks[key_ya_make_darwin] = tasks['ya_make_darwin'].id
            self.Context.save()
            tasks_started = True

        if tasks_started:
            logger.info("Tasks started. Waiting untill they finished")
            raise sdk2.WaitTask(
                tasks.values(),
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        conn = S3Connection(
            host='s3.mds.yandex.net',
            aws_access_key_id=ACCESS_KEY_ID,
            aws_secret_access_key=SECRET_ACCESS_KEY
            )
        bucket = conn.get_bucket("logbroker")
        for task_name in task_ids.keys():
            logger.info("Start working on task with name {}".format(task_name))
            resource_type = ""
            if task_name == "ya_package":
                continue
            if task_name == "ya_make_linux":
                resource_type = "LOGBROKER_CLI_LINUX"
            if task_name == "ya_make_darwin":
                resource_type = "LOGBROKER_CLI_DARWIN"
            child = sdk2.Task[task_ids[task_name]]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Child task was finished with status {}".format(child.status))
            resource = sdk2.Resource[resource_type].find(task=child).first()
            resource_data = sdk2.ResourceData(resource)
            resource_path = resource_data.path
            logger.info("Resource {} dumps to {}".format(resource_type, resource_path))
            self.upload_binary(bucket, str(resource_path), s3_path[resource_type])

    def on_release(self):
        pass
