# coding: utf-8
import os
import json
import time

from sandbox import sdk2
from sandbox import common
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.channel as sdk_channel
import sandbox.sandboxsdk.process as sdk_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.build import parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.juggler.check_bundles import BaseBuildJugglerChecksBundleTask, ForceResourceCreationParameter
from sandbox.projects import resource_types
from sandbox.common.types.task import Status


class PackagePathParameter(sdk_parameters.SandboxStringParameter):
    name = 'package_path'
    description = 'Path to package file in Arcadia'
    default_value = "juggler/examples/check_bundles/test/bundle.json"
    required = True


class ResourceTypeParameter(sdk_parameters.SandboxSelectParameter):
    name = 'resource_type'
    description = 'Resource type to build'
    default_value = 'JUGGLER_CHECKS_BUNDLE'
    choices = [(r.name, r.name) for r in sdk2.Resource]
    required = True


class BundleNameParameter(sdk_parameters.SandboxStringParameter):
    name = 'bundle_name'
    description = 'Name of bundle'
    default_value = ""


class UploadToS3VaultOwner(sdk_parameters.SandboxStringParameter):
    name = 's3_vault_owner'
    description = 'S3 Auth vault owner'
    default_value = None


class UploadToS3VaultName(sdk_parameters.SandboxStringParameter):
    name = 's3_vault_name'
    description = 'S3 Auth vault name'
    default_value = None


class S3EndpointParameter(sdk_parameters.SandboxRadioParameter):
    name = 's3_endpoint'
    description = 'S3 Endpoint'
    default_value = "storage.yandexcloud.net"
    choices = [
        ('storage.yandexcloud.net', 'storage.yandexcloud.net'),
        ('s3.mds.yandex.net', 's3.mds.yandex.net'),
    ]


class CustomVersion(sdk_parameters.SandboxStringParameter):
    name = 'custom_version'
    description = 'Package custom version'
    default_value = None
    required = True


class BuildLogbrokerJugglerWrapperBundle(nanny.ReleaseToNannyTask, BaseBuildJugglerChecksBundleTask):
    input_parameters = BaseBuildJugglerChecksBundleTask.input_parameters + [
        parameters.ArcadiaUrl,
        PackagePathParameter,
        ResourceTypeParameter,
        ForceResourceCreationParameter,
        BundleNameParameter,
        UploadToS3VaultOwner,
        UploadToS3VaultName,
        S3EndpointParameter,
        CustomVersion,
    ]

    def on_execute(self):
        bundle_name = self.ctx.get(BundleNameParameter.name)
        if not bundle_name:
            raise common.errors.SandboxEnvironmentError("No bundle name provided")

        local_root = self.path("build")
        common.fs.make_folder(local_root, delete_content=True)
        sdk_process.run_process(["tar", "--directory", local_root, "-xf", self._get_package_path()])

        manifest_path = os.path.join(local_root, "MANIFEST.json")
        if not os.path.exists(manifest_path):
            raise SandboxTaskFailureError('MANIFEST.json file is not found')

        with open(manifest_path) as stream:
            manifest = json.load(stream)

        if "version" not in manifest:
            raise SandboxTaskFailureError("Manifest must have an integer field named 'version', set to 1")

        version = manifest["version"]
        if version != 1:
            raise SandboxTaskFailureError("Cannot work with versions other than 1")

        for executable in manifest.pop("discover_from_executable", ()):
            manifest = self._create_checks_from_executable(local_root, executable, manifest)

        if not manifest.get("checks"):
            raise SandboxTaskFailureError("Manifest must have non-empty checks list")

        env = manifest.get("env", {})
        if not isinstance(env, dict):
            raise SandboxTaskFailureError("Environment should be defined as a dictionary")

        checks_meta = list(self._iter_checks_meta(local_root, manifest["checks"]))
        bundle_meta = {
            "timestamp": int(time.time()),
            "bundle_name": bundle_name,
            "checks": checks_meta,
            "version": 1,
            "build_number": self.id,
            "env": env
        }
        files_list = [
            (os.path.join(local_root, file_name), file_name)
            for file_name in os.listdir(local_root)
            # ignore private files like .svn and manifest
            if not file_name.startswith(".") and file_name != "MANIFEST.json"
        ]

        resource_type = self.ctx.get(ResourceTypeParameter.name)

        s3_owner = self.ctx.get(UploadToS3VaultOwner.name)
        s3_name = self.ctx.get(UploadToS3VaultName.name)
        s3_endpoint = self.ctx.get(S3EndpointParameter.name)
        s3_user = None

        # Adding value only if s3_vault passed
        if s3_name and s3_owner:
            s3_user = {'vault_owner': s3_owner,
                       'vault_name': s3_name}

        self._make_bundle_resource(bundle_meta, files_list, resource_type=resource_type,
                                   s3_user=s3_user, s3_endpoint=s3_endpoint)

    @staticmethod
    def _create_checks_from_executable(bundle_root, executable, manifest):
        stdout, _ = sdk_process.run_process(
            [os.path.join(bundle_root, executable), "--show-manifest"],
            outs_to_pipe=True
        ).communicate()
        os.mkdir(os.path.join(bundle_root, 'links'))
        for check in json.loads(stdout)["checks"]:
            os.symlink(
                os.path.join("..", os.path.basename(executable)),
                os.path.join(bundle_root, check["check_script"]))
            manifest.setdefault("checks", []).append(check)
        return manifest

    @classmethod
    def _iter_checks_meta(cls, base_dir, checks_manifest):
        check_hash_map = {}
        for check_meta in checks_manifest:
            string_def = json.dumps(check_meta)

            if "check_script" not in check_meta:
                raise SandboxTaskFailureError(
                    "check_script option not present in check definition: {0!r}".format(string_def)
                )
            file_path = os.path.join(base_dir, check_meta["check_script"])
            if not os.path.exists(file_path):
                raise SandboxTaskFailureError("file for check definition not found: {0!r}".format(string_def))
            if not os.access(file_path, os.X_OK):
                raise SandboxTaskFailureError("check file is not marked as executable: {0!r}".format(string_def))

            services = []
            if "services" in check_meta:
                services = check_meta["services"]
            if not services:
                services = cls._get_provided_services(file_path)
            if not services:
                services = [os.path.splitext(os.path.basename(file_path))[0]]

            output_format = check_meta.get("format", "nagios")
            if output_format not in ("json", "nagios"):
                raise SandboxTaskFailureError("format should be equal to 'json' or 'nagios': {0!r}".format(string_def))

            real_file_path = os.path.realpath(file_path)
            check_hash = check_hash_map.get(real_file_path)
            if check_hash is None:
                check_hash = check_hash_map[real_file_path] = cls._get_file_hash(file_path)

            yield {
                "check_name": os.path.basename(file_path),
                "check_script": check_meta["check_script"],
                "run_always":  bool(check_meta.get("run_always", False)),
                "interval": int(check_meta.get("interval", 300)),
                "timeout": int(check_meta.get("timeout", 60)),
                "args": check_meta.get("args", []),
                "check_hash": check_hash,
                "services": services,
                "format": output_format
            }

    def _get_package_path(self):
        if not self.ctx.get(parameters.ArcadiaUrl.name):
            self.ctx[parameters.ArcadiaUrl.name] = "arcadia:/arc/trunk/arcadia"
        if not self.ctx.get(PackagePathParameter.name):
            raise SandboxTaskFailureError("Specify resource with package or path to build it")
        resource_id = self._build_tar_package(self.ctx[parameters.ArcadiaUrl.name], self.ctx[PackagePathParameter.name], self.ctx[CustomVersion.name])
        return self.sync_resource(resource_id)

    def _build_tar_package(self, arcadia_url, arcadia_package_path, custom_version):
        key = "_ya_package_task_id"
        task_id = self.ctx.get(key)
        if task_id is None:
            input_parameters = {
                # TODO: cleanup
                "checkout": True,
                "checkout_arcadia_from_url": arcadia_url,
                "checkout_mode": "manual",
                "packages": arcadia_package_path,
                "package_type": "tarball",
                "resource_type": resource_types.YA_PACKAGE.name,
                "build_type": "release",
                "build_system": "ya",
                'notify_via': '',
                'custom_version': custom_version,
            }
            task = self.create_subtask(
                task_type="YA_PACKAGE",
                description="Created by {0}".format(self.type),
                input_parameters=input_parameters
            )
            task_id = self.ctx[key] = task.id
            self.wait_tasks(
                tasks=[task],
                statuses=tuple(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True)

        task = sdk_channel.channel.sandbox.get_task(task_id)
        if not task.is_done():
            raise SandboxTaskFailureError("Package build failed, see logs for details")

        resources = sdk_channel.channel.sandbox.list_resources(
            resource_type=resource_types.YA_PACKAGE,
            task_id=task_id)

        if not resources or not all(x.is_ready() for x in resources):
            raise SandboxTaskFailureError("{0} not ready or not found".format(resource_types.YA_PACKAGE))

        return resources[0].id
