import os
import shutil
import logging
import sandbox.common.types.task as ctt
import sandbox.common.errors as ce
import sandbox.projects.common.constants as consts
from sandbox import sdk2
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.logbroker import common
from sandbox.projects.logbroker import resources
from sandbox.sandboxsdk.process import run_process


logger = logging.getLogger(__name__)


class BuildUnifiedAgent(ReleaseToNannyTask2, sdk2.Task):
    class Context(sdk2.Task.Context):
        child_tasks = dict()

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        with sdk2.parameters.Group("Binary build parameters"):
            ua_version_file = sdk2.parameters.String(
                'Unified Agent version file',
                required=True,
                default_value='logbroker/unified_agent/version.txt'
            )
            ua_target = sdk2.parameters.String(
                'Unified Agent binary build target',
                required=True,
                default_value='logbroker/unified_agent/bin'
            )
            ua_path = sdk2.parameters.String(
                'Unified Agent binary artifact path',
                required=True,
                default_value='logbroker/unified_agent/bin/unified_agent'
            )
            ua_win_path = sdk2.parameters.String(
                'Unified Agent windows binary artifact path',
                required=True,
                default_value='logbroker/unified_agent/bin/unified_agent.exe'
            )

        with sdk2.parameters.Group("Debian package build parameters"):
            trusty_pkg_target = sdk2.parameters.String(
                'Ubuntu Trusty package path',
                required=True,
                default_value='logbroker/unified_agent/pkg/yandex-unified-agent/ubuntu-14.04-trusty/pkg.json',
            )
            xenial_pkg_target = sdk2.parameters.String(
                'Ubuntu Xenial package path',
                required=True,
                default_value='logbroker/unified_agent/pkg/yandex-unified-agent/ubuntu-16.04-xenial/pkg.json',
            )
            bionic_pkg_target = sdk2.parameters.String(
                'Ubuntu Bionic package path',
                required=True,
                default_value='logbroker/unified_agent/pkg/yandex-unified-agent/ubuntu-18.04-bionic/pkg.json',
            )
            focal_pkg_target = sdk2.parameters.String(
                'Ubuntu Focal package path',
                required=True,
                default_value='logbroker/unified_agent/pkg/yandex-unified-agent/ubuntu-20.04-focal/pkg.json',
            )

        with sdk2.parameters.Group("Layer build parameters"):
            lt_target = sdk2.parameters.String(
                'Logbroker tools build target',
                required=True,
                default_value='logbroker/unified_agent/tools/deploy_logbroker_tools/bin',
            )
            lt_path = sdk2.parameters.String(
                'Logbroker tools artifact path',
                required=True,
                default_value='logbroker/unified_agent/tools/deploy_logbroker_tools/bin/deploy_logbroker_tools',
            )
            script = sdk2.parameters.String(
                'Layer setup shell script',
                required=True,
                default_value='logbroker/unified_agent/tools/deploy_logbroker_tools/porto/setup.sh',
            )
            parent_layer_type = sdk2.parameters.String(
                'Parent layer type',
                required=True,
                default_value='PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP',
            )
            parent_layer = sdk2.parameters.Resource(
                'Parent layer (default: latest stable release)',
                required=False,
                default_value=904130395,
            )
            merge_layers = sdk2.parameters.Bool(
                'Merge all layers',
                required=False,
                default_value=False,
            )

    def on_execute(self):
        with self.memoize_stage.prepare:
            self.Context.revision = sdk2.svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url).revision
            if self.Context.revision is None:
                self.set_info('Arcadia revision: not specified')
            else:
                self.set_info('Arcadia revision: %s' % self.Context.revision)
            self.Context.version = common.get_unified_agent_version(self.Parameters.ua_version_file, self.Context.revision)
            self.set_info('Unified Agent version: %s' % self.Context.version)

        with self.memoize_stage.build:
            task_ya_make_ua = YaMake2(
                self,
                description='unified_agent binary, version {}'.format(self.Context.version),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                build_type='release',
                check_return_code=True,
                targets=self.Parameters.ua_target,
                arts=self.Parameters.ua_path,
                result_single_file=True,
                result_rd='unified_agent binary, version {}'.format(self.Context.version),
                result_rt='UNIFIED_AGENT_BIN'
            )
            task_ya_make_ua.save().enqueue()
            self.Context.child_tasks['task_ya_make_ua'] = task_ya_make_ua.id
            self.set_info('unified_agent binary build task %s started' % task_ya_make_ua.id)

            task_ya_make_windows_ua = YaMake2(
                self,
                description='unified_agent windows binary, version {}'.format(self.Context.version),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                build_system=consts.YMAKE_BUILD_SYSTEM,
                build_type='release',
                check_return_code=True,
                targets=self.Parameters.ua_target,
                arts=self.Parameters.ua_win_path,
                result_single_file=True,
                result_rd='unified_agent windows binary, version {}'.format(self.Context.version),
                result_rt='UNIFIED_AGENT_WINDOWS_BIN',
                target_platform='windows'
            )
            task_ya_make_windows_ua.save().enqueue()
            self.Context.child_tasks['task_ya_make_windows_ua'] = task_ya_make_windows_ua.id
            self.set_info('unified_agent windows binary build task %s started' % task_ya_make_windows_ua.id)

            task_ya_make_lt = YaMake2(
                self,
                description='deploy_logbroker_tools binary',
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                build_type='release',
                check_return_code=True,
                strip_binaries=True,
                targets=self.Parameters.lt_target,
                arts=self.Parameters.lt_path,
                result_single_file=True,
                result_rd='deploy_logbroker_tools binary',
                result_rt='LOGBROKER_TOOLS_BIN',
            )
            task_ya_make_lt.save().enqueue()
            self.Context.child_tasks['task_ya_make_lt'] = task_ya_make_lt.id
            self.set_info('deploy_logbroker_tools binary build task %s started' % task_ya_make_lt.id)

            for distro in ['trusty', 'xenial', 'bionic', 'focal']:
                logger.info('start YA_PACKAGE %s' % distro)
                task_ya_package = sdk2.Task['YA_PACKAGE'](
                    self,
                    description='yandex-unified-agent debian package for ubuntu-{}, version {}'.format(distro, self.Context.version),
                    checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                    use_aapi_fuse=True,
                    use_arc_instead_of_aapi=False,
                    aapi_fallback=True,
                    build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                    build_type='release',
                    packages=getattr(self.Parameters, '{}_pkg_target'.format(distro)),
                    package_type='debian',
                    strip_binaries=True,
                    full_strip_binaries=True,
                    create_debug_packages=True,
                    compress_package_archive=True,
                    dupload_max_attempts=2,
                    publish_package=True,
                    key_user='robot-logbroker',
                    publish_to='yandex-{}'.format(distro),
                    custom_version=self.Context.version,
                )
                task_ya_package.save().enqueue()
                self.Context.child_tasks['task_ya_package_{}'.format(distro)] = task_ya_package.id
                self.set_info('yandex-unified-agent debian package for ubuntu-%s build task %s started' % (distro, task_ya_package.id))

            raise sdk2.WaitTask(
                [
                    task_ya_make_ua,
                    task_ya_make_windows_ua,
                    task_ya_make_lt,
                ],
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        with self.memoize_stage.build_ua_bin_resource:
            resources_per_tasks = {
                'task_ya_make_ua': [resources.UnifiedAgentBin],
                'task_ya_make_windows_ua': [resources.UnifiedAgentWindowsBin]
            }
            self.build_resources(resources_per_tasks)

        with self.memoize_stage.build_layer:
            task_ya_make_ua = sdk2.Task[self.Context.child_tasks['task_ya_make_ua']]
            task_ya_make_lt = sdk2.Task[self.Context.child_tasks['task_ya_make_lt']]

            for task in [task_ya_make_ua, task_ya_make_lt]:
                if task.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))

            ua_bin = resources.UnifiedAgentBin.find(task=self).first()
            lt_bin = resources.LogbrokerToolsBin.find(task=task_ya_make_lt).first()

            task_build_porto_layer = sdk2.Task['BUILD_PORTO_LAYER'](
                self,
                description='logbroker_tools_layer, version {}'.format(self.Context.version),
                parent_layer_type=self.Parameters.parent_layer_type,
                parent_layer=self.Parameters.parent_layer,
                layer_type='LOGBROKER_TOOLS_PORTO_LAYER',
                layer_name='logbroker_tools_layer',
                compress='tar.xz',
                script=common.get_arcadia_url(self.Parameters.script, self.Context.revision),
                script_env={
                    'UNIFIED_AGENT_BIN': 'https://proxy.sandbox.yandex-team.ru/%s' % ua_bin.id,
                    'DEPLOY_LOGBROKER_TOOLS_BIN': 'https://proxy.sandbox.yandex-team.ru/%s' % lt_bin.id,
                    'LOGBROKER_BUILD_TASK_ID': str(self.id),
                },
                start_os=True,
                merge_layers=self.Parameters.merge_layers,
            )
            task_build_porto_layer.save().enqueue()
            self.Context.child_tasks['task_build_layer'] = task_build_porto_layer.id
            self.set_info('logbroker_tools_layer build task %s started' % task_build_porto_layer.id)

            wait_tasks = [task_build_porto_layer]
            for distro in ['trusty', 'xenial', 'bionic', 'focal']:
                wait_tasks.append(sdk2.Task[self.Context.child_tasks['task_ya_package_{}'.format(distro)]])

            raise sdk2.WaitTask(
                wait_tasks,
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        with self.memoize_stage.finalize:
            for task_key, task_id in self.Context.child_tasks.items():
                task = sdk2.Task[task_id]
                if task.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))

            resources_per_tasks = {
                'task_ya_make_lt': [resources.LogbrokerToolsBin],
                'task_build_layer': [resources.LogbrokerToolsPortoLayer]
            }
            self.build_resources(resources_per_tasks)

            self.set_info('done')

    def build_resources(self, resources_per_tasks):
        for task_key in resources_per_tasks:
            task = sdk2.Task[self.Context.child_tasks[task_key]]
            if task.status != ctt.Status.SUCCESS:
                raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))
            for resource_cls in resources_per_tasks[task_key]:
                resource = resource_cls.find(task=task).first()
                resource.ttl = 1
                if task_key in ['task_ya_make_ua', 'task_ya_make_windows_ua', 'task_build_layer']:
                    resource_data = sdk2.ResourceData(resource)
                    shutil.copy(str(resource_data.path), resource_data.path.name)
                    if resource_cls == resources.UnifiedAgentBin:
                        symbols_path = self.strip_and_create_symbols(resource_data.path.name)
                        self.create_resource(resources.UnifiedAgentDebugSymbols, symbols_path, resource.description)
                    self.create_resource(resource_cls, resource_data.path.name, resource.description)

    def create_resource(self, resource_class, path, description):
        resource = resource_class(
            self,
            description,
            path,
            unified_agent_version=self.Context.version,
            build_task_id=self.id,
        )
        sdk2.ResourceData(resource).ready()

    def strip_and_create_symbols(self, bin_path):
        symbols_path = 'unified_agent.debug'
        os.chmod(bin_path, 0755)
        commands = [
            ['objcopy', '--only-keep-debug', bin_path, symbols_path],
            ['objcopy', '--strip-debug', '--strip-unneeded', '--remove-section=.gnu_debuglink', bin_path],
            ['objcopy', '--add-gnu-debuglink=' + symbols_path, '--remove-section=.gdb_index', bin_path]
        ]
        for command in commands:
            run_process(command, log_prefix='objcopy')
        return symbols_path

    def on_release(self, additional_parameters):
        logger.debug("Release parameters: %r", additional_parameters)
        ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

        logbroker_tools_layer = resources.LogbrokerToolsPortoLayer.find(state='READY', attrs={'unified_agent_version': self.Context.version}).first()
        unified_agent_bin = resources.UnifiedAgentBin.find(state='READY', attrs={'unified_agent_version': self.Context.version}).first()
        unified_agent_debug_symbols = resources.UnifiedAgentDebugSymbols.find(state='READY', attrs={'unified_agent_version': self.Context.version}).first()

        if additional_parameters['release_status'] == ctt.ReleaseStatus.STABLE:
            pass
            # logbroker_tools_layer.released_sas_test = True
            # logbroker_tools_layer.released_man_pre = True
            # logbroker_tools_layer.released_xdc_acceptance = True
            # logbroker_tools_layer.released_xdc = True

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.PRESTABLE:
            pass
            # logbroker_tools_layer.released_sas_test = True
            # logbroker_tools_layer.released_man_pre = True
            # logbroker_tools_layer.released_xdc_acceptance = True
            # logbroker_tools_layer.released_xdc = False

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.TESTING:
            pass
            # logbroker_tools_layer.released_sas_test = True
            # logbroker_tools_layer.released_man_pre = True
            # logbroker_tools_layer.released_xdc_acceptance = False
            # logbroker_tools_layer.released_xdc = False

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.UNSTABLE:
            pass
            # logbroker_tools_layer.released_sas_test = True
            # logbroker_tools_layer.released_man_pre = False
            # logbroker_tools_layer.released_xdc_acceptance = False
            # logbroker_tools_layer.released_xdc = False

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.CANCELLED:
            logbroker_tools_layer.released_sas_test = False
            logbroker_tools_layer.released_man_pre = False
            logbroker_tools_layer.released_xdc_acceptance = False
            logbroker_tools_layer.released_xdc = False
            logbroker_tools_layer.ttl = 14
            unified_agent_bin.ttl = 14
            unified_agent_debug_symbols.ttl = 14

        else:
            raise ce.TaskFailure('Unexpected release_status')
