import shutil
import logging
import sandbox.common.types.task as ctt
import sandbox.common.errors as ce
import sandbox.projects.common.constants as consts
from sandbox import sdk2
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.logbroker import resources


logger = logging.getLogger(__name__)


class BuildYtsender(ReleaseToNannyTask2, sdk2.Task):
    class Context(sdk2.Task.Context):
        child_tasks = dict()

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        with sdk2.parameters.Group("Binary build parameters"):
            target = sdk2.parameters.String(
                'YtSender binary build target',
                required=True,
                default_value='logbroker/pipe-parser/ytsender-ng/bin'
            )
            path = sdk2.parameters.String(
                'YtSender artifact path',
                required=True,
                default_value='logbroker/pipe-parser/ytsender-ng/bin/ytsender-ng'
            )

        with sdk2.parameters.Group("Debian package build parameters"):
            sender_pkg_target = sdk2.parameters.String(
                'YtSender sender package path',
                required=True,
                default_value='logbroker/pipe-parser/pkg/yandex-logbroker-ytsender-ng/pkg.json',
            )
            master_pkg_target = sdk2.parameters.String(
                'YtSender master package path',
                required=True,
                default_value='logbroker/pipe-parser/pkg/yandex-logbroker-ytsender-ng-master/pkg.json',
            )
            publish_to = sdk2.parameters.String(
                'Publish to',
                required=True,
                default_value='statbox-common',
            )

    def on_execute(self):
        with self.memoize_stage.prepare:
            self.Context.revision = sdk2.svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url).revision
            if self.Context.revision is None:
                self.set_info('Arcadia revision: not specified')
            else:
                self.set_info('Arcadia revision: %s' % self.Context.revision)

        with self.memoize_stage.build:
            wait_tasks = []
            task_ya_make = YaMake2(
                self,
                description='ytsender binary, revision {}'.format(self.Context.revision),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True,
                use_arc_instead_of_aapi=False,
                aapi_fallback=True,
                build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                build_type='release',
                check_return_code=True,
                strip_binaries=False,
                targets=self.Parameters.target,
                arts=self.Parameters.path,
                result_single_file=True,
                result_rd='ytsender binary, revision {}'.format(self.Context.revision),
                result_rt='YTSENDER_BIN',
            )
            task_ya_make.save().enqueue()
            self.Context.child_tasks['task_ya_make'] = task_ya_make.id
            self.set_info('ytsender binary build task %s started' % task_ya_make.id)
            wait_tasks.append(task_ya_make)

            for pkg_name, pkg_target in zip(['yandex-logbroker-ytsender-ng', 'yandex-logbroker-ytsender-ng-master'], [self.Parameters.sender_pkg_target, self.Parameters.master_pkg_target]):
                task_ya_package = sdk2.Task['YA_PACKAGE'](
                    self,
                    description='{} debian package, revision {}'.format(pkg_name, self.Context.revision),
                    checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                    use_aapi_fuse=True,
                    use_arc_instead_of_aapi=False,
                    aapi_fallback=True,
                    build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                    build_type='release',
                    packages=pkg_target,
                    package_type='debian',
                    strip_binaries=False,
                    create_debug_packages=True,
                    compress_package_archive=True,
                    dupload_max_attempts=2,
                    publish_package=True,
                    key_user='robot-logbroker',
                    publish_to=self.Parameters.publish_to,
                )
                task_ya_package.save().enqueue()
                self.Context.child_tasks['task_ya_package_{}'.format(pkg_name)] = task_ya_package.id
                self.set_info('%s debian package build task %s started' % (pkg_name, task_ya_package.id))
                wait_tasks.append(task_ya_package)

            raise sdk2.WaitTask(
                wait_tasks,
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        with self.memoize_stage.finalize:
            for task_key, task_id in self.Context.child_tasks.items():
                task = sdk2.Task[task_id]
                if task.status != ctt.Status.SUCCESS:
                    raise ce.TaskFailure('Child task was finished with status {}'.format(task.status))

                if task_key == 'task_ya_make':
                    resource = resources.YtsenderBin.find(task=task).first()
                    resource.ttl = 1
                    resource_data = sdk2.ResourceData(resource)
                    resource_new = resources.YtsenderBin(
                        self,
                        resource.description,
                        resource_data.path.name,
                        build_task_id=self.id,
                    )
                    shutil.copy(str(resource_data.path), resource_data.path.name)
                    sdk2.ResourceData(resource_new).ready()

            self.set_info('done')

    def on_release(self, additional_parameters):
        logger.debug("Release parameters: %r", additional_parameters)
        ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

        ytsender_bin = resources.UnifiedAgentBin.find(state='READY', attrs={'unified_agent_version': self.Context.version}).first()

        if additional_parameters['release_status'] == ctt.ReleaseStatus.STABLE:
            pass

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.PRESTABLE:
            pass

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.TESTING:
            pass

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.UNSTABLE:
            pass

        elif additional_parameters['release_status'] == ctt.ReleaseStatus.CANCELLED:
            ytsender_bin.ttl = 14

        else:
            raise ce.TaskFailure('Unexpected release_status')
