import os
import re
import logging

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.logbroker.common import ResourceTypeId, ScriptEnv
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog
import sandbox.common.errors as ce
from sandbox.projects.logbroker.resources import AnsibleBin


logger = logging.getLogger('sandbox-task')
logger.setLevel(logging.DEBUG)


logger_sp = logging.getLogger('sandbox-subprocess')
logger_sp.setLevel(logging.DEBUG)


class LogbrokerAnsibleJuggler(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 512

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        ansible_bin = sdk2.parameters.Resource(
            'ANSIBLE_BIN resource id to use',
            resource_type=AnsibleBin,
            required=True,
        )
        ansible_juggler_url = sdk2.parameters.ArcadiaUrl('ansbile-juggler directory arcadia url', required=True)
        playbook_name = sdk2.parameters.String('Name of playbook to run', default_value='juggler.yml')
        resources = ResourceTypeId('Resources to load (key=resource_type, value=resource_id)')
        secrets = ScriptEnv('Environment variables from vault (key=env_var, value=vault_owner,vault_key)')
        verbosity = sdk2.parameters.Integer('Verbosity level (0 - inf)', default_value=0)

    def get_resource(self, resource_type, resource_id):
        logger.debug('searching for resource_type: %s, resource_id: %s', resource_type, resource_id)
        resource = sdk2.Resource[resource_type].find(id=resource_id).first()

        if resource is None:
            msg = 'Cannot find resource_type: %s, resource_id: %s' % (resource_type, resource_id)
            logger.error(msg)
            self.set_info(msg)
            raise ce.TaskError(msg)

        symlink = resource_type.lower()

        msg = 'Using resource_type: %s, resource_id: %s, link: %s' % (resource_type, resource.id, symlink)
        logger.debug(msg)
        self.set_info(msg)

        os.symlink(str(sdk2.ResourceData(resource).path), os.path.abspath(symlink))

        return resource

    def on_execute(self):
        logger.debug('executing')

        logger.debug('getting ANSIBLE_BIN resource')
        ansible_bin = str(sdk2.ResourceData(self.Parameters.ansible_bin).path)
        os.chmod(ansible_bin, 0o755)
        os.symlink(ansible_bin, 'ansible')
        os.symlink(ansible_bin, 'ansible-playbook')

        sdk2.svn.Arcadia.checkout(self.Parameters.ansible_juggler_url, '.', sdk2.svn.Arcadia.Depth.INFINITY)

        rev_info = sdk2.svn.Arcadia.info('.')
        logger.debug('arcadia revision: %s', rev_info)
        msg_lines = [
            'ARCADIA REVISION:',
            'author: %s' % rev_info['author'],
            'commit revision: %s' % rev_info['commit_revision'],
            'link: https://a.yandex-team.ru/arc/commit/%s' % rev_info['commit_revision'],
        ]
        self.set_info('\n'.join(msg_lines))

        for resource_type in self.Parameters.resources:
            resource_id = self.Parameters.resources[resource_type]
            self.get_resource(resource_type, resource_id)

        for var in self.Parameters.secrets:
            value = self.Parameters.secrets[var]
            vault_owner, vault_key = value.split(',')
            logger.debug('loading vault data with owner %s, key %s', vault_owner, vault_key)
            os.environ[var] = sdk2.Vault.data(vault_owner, vault_key)

        logger.debug('workdir contents:\n%s', '\n'.join(os.listdir('.')))

        os.environ['ANSIBLE_CONFIG'] = os.path.join(os.path.abspath('.'), 'ansible.cfg')
        os.environ['ANSIBLE_YANDEX_INSTALL_PATH'] = 'ansible-yandex'
        os.environ['ANSIBLE_YANDEX_DEBUG'] = 'true'

        output = sp.check_output(['./ansible', '--version']).strip()
        self.set_info('ANSIBLE VERSION:\n%s' % output)
        logger.debug('ansible version: %s', ', '.join(re.split(r'\s\s+', output)))

        cmd = ['./ansible-playbook']
        if self.Parameters.verbosity > 0:
            cmd.append('-{}'.format('v' * self.Parameters.verbosity))
        cmd.append(self.Parameters.playbook_name)

        self.set_info('COMMAND:\n%s' % ' '.join(cmd))
        logger.info('cmd: %s', cmd)

        try:
            with ProcessLog(self, logger='ansible-playbook') as pl:
                logger.debug('playbook execution starting')
                sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)
                logger.debug('playbook execution finished')

        except sp.CalledProcessError:
            ansible_log = []

            for line in reversed(fu.read_lines(str(self.log_path('common.log')))):
                if '(ansible-playbook)' in line:
                    ansible_log.insert(0, line)
                    if 'TASK: [' in line:
                        break

            if ansible_log:
                eh.fail('\n'.join(ansible_log))
            else:
                raise
