#!/usr/bin/env python
# -*- coding: UTF-8 -*-
from __future__ import print_function
import os
import json
import urllib2
import logging
import sandbox.common.errors as ce
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog
from sandbox.projects.logbroker.common import ScriptEnv
from sandbox.projects.logbroker.resources import LogbrokerJugglerStatsBinary


logger = logging.getLogger('sandbox-task')
logger.setLevel(logging.DEBUG)


logger_sp = logging.getLogger('sandbox-subprocess')
logger_sp.setLevel(logging.DEBUG)


RESOURCE_NAME = 'LOGBROKER_JUGGLER_STATS_BINARY'


class LogbrokerJugglerStats(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 2 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        description = 'Juggler stats solomon uploader.'
        resource = sdk2.parameters.Resource(
            'LOGBROKER_JUGGLER_STATS_BINARY resource id to use',
            resource_type=LogbrokerJugglerStatsBinary,
            default=996933820,
        )

        secrets = ScriptEnv('Environment variables from vault (key=env_var, value=vault_owner,vault_key)')

        juggler_namespace = sdk2.parameters.String('Name of juggler namespace to filter checks.', required=True)
        juggler_hostname = sdk2.parameters.String('Name of juggler hostname to filter checks.')
        juggler_service = sdk2.parameters.String('Name of juggler service to filter checks.')

        solomon_project = sdk2.parameters.String('Name of solomon project to push sensors to.', required=True)
        solomon_cluster = sdk2.parameters.String('Name of solomon cluster to push sensors to.', required=True)
        solomon_service = sdk2.parameters.String('Name of solomon service to push sensors to.', required=True)

        backfill_days = sdk2.parameters.Integer('How many days to fetch from juggler history if nothing was found in solomon.', default=90)

        notifications_stats = sdk2.parameters.Bool('Collect notifications statistics.', default=False)
        with notifications_stats.value[True]:
            notifications_logins = sdk2.parameters.String('Comma-separated list of logins.')
            notifications_backfill_hours = sdk2.parameters.String('How many hours to fetch from juggler notification history and push to solomon.', default=25)

        workers = sdk2.parameters.Integer('How many worker processes to use.', default=8)
        dry_run = sdk2.parameters.Bool('Dry run (do not push data to solomon).', default=False)

        juggler_event_send = sdk2.parameters.Bool('Send result of this task execution to juggler for monitoring purposes.', default=False)
        with juggler_event_send.value[True]:
            juggler_event_service = sdk2.parameters.String('Juggler event service.')
            juggler_event_host = sdk2.parameters.String('Juggler event host.')

    def send_juggler_event(self, status, description):
        if self.Parameters.juggler_event_send:
            if status != 'OK':
                description += ' https://sandbox.yandex-team.ru/task/%s' % self.id

            event = {
                'service': self.Parameters.juggler_event_service,
                'host': self.Parameters.juggler_event_host,
                'status': status.upper(),
                'description': description,
            }
            data = {
                'source': 'sandbox.projects.LogbrokerJugglerStats',
                'events': [event],
            }
            logger.debug('sending event to juggler: %s', data)
            req = urllib2.Request('http://juggler-push.search.yandex.net/events')
            req.add_header('Content-Type', 'application/json')
            retries = 3
            while retries > 0:
                retries -= 1
                response = urllib2.urlopen(req, json.dumps(data))
                if response.getcode() == 200:
                    break
                else:
                    logger.debug('got http error: %s, retries left: %s', response.getcode(), retries)
                    if retries == 0:
                        logger.error('cannot push event to juggler')
                        return

            try:
                response_text = response.read()
                response_data = json.loads(response_text)
                logger.debug('got response data: %s', response_data)
            except ValueError:
                logger.error('value error while parsing response text: %s', response_text)

    def on_execute(self):
        logger.debug('calling on_execute')

        script_path = str(sdk2.ResourceData(self.Parameters.resource).path)

        for var in self.Parameters.secrets:
            value = self.Parameters.secrets[var]
            vault_owner, vault_key = value.split(',')
            logger.debug('loading vault data with owner %s, key %s', vault_owner, vault_key)
            os.environ[var] = sdk2.Vault.data(vault_owner, vault_key)

        cmd = [script_path, '--juggler-namespace', self.Parameters.juggler_namespace]

        if self.Parameters.juggler_hostname != '':
            cmd += ['--juggler-hostname', self.Parameters.juggler_hostname]

        if self.Parameters.juggler_service != '':
            cmd += ['--juggler-service', self.Parameters.juggler_service]

        cmd += [
            '--solomon-project', self.Parameters.solomon_project,
            '--solomon-cluster', self.Parameters.solomon_cluster,
            '--solomon-service', self.Parameters.solomon_service,
            '--backfill-days', str(self.Parameters.backfill_days),
            '--workers', str(self.Parameters.workers),
        ]

        if self.Parameters.notifications_stats:
            cmd += [
                '--notifications-logins', self.Parameters.notifications_logins,
                '--notifications-backfill-hours', self.Parameters.notifications_backfill_hours,
            ]

        if self.Parameters.dry_run:
            cmd += ['--dry-run']

        msg = 'Run command: %s.' % cmd
        logger.debug(msg)
        self.set_info(msg)

        with ProcessLog(self, logger=logger_sp) as pl:
            proc = sp.Popen(cmd, shell=False, stdout=pl.stdout, stderr=pl.stdout)
            proc.communicate()
            if proc.returncode != 0:
                msg = 'Command failed with returncode: %s.' % proc.returncode
                logger.debug(msg)
                self.send_juggler_event('CRIT', msg)
                raise ce.TaskError(msg)

        composed_sensors = 0
        warnings_count = 0
        errors_count = 0
        juggler_status = 'OK'
        juggler_description = 'OK'

        try:
            with open('log1/common.log') as file:
                for line in file:
                    if 'sandbox-subprocess' in line:
                        if 'COMPOSED SENSOR' in line:
                            composed_sensors += 1
                        if 'WARNING' in line:
                            warnings_count += 1
                        if 'ERROR' in line:
                            errors_count += 1

        except OSError:
            msg = 'Cannot read log1/common.log.'
            logger.exception(msg)
            self.set_info(msg)
            juggler_status = 'WARN'
            juggler_description = msg

        msg = 'Composed sensors: %s' % composed_sensors
        logger.debug(msg)
        self.set_info(msg)

        if warnings_count:
            msg = 'Warnings count: %s.' % warnings_count
            logger.debug(msg)
            self.set_info(msg)
            juggler_status = 'WARN'
            juggler_description = msg

        if errors_count:
            msg = 'Errors count: %s.' % errors_count
            logger.debug(msg)
            self.set_info(msg)
            juggler_status = 'CRIT'
            juggler_description = msg

        self.send_juggler_event(juggler_status, juggler_description)
        self.set_info('Done')
