#!/usr/bin/env python
# -*- coding: UTF-8 -*-
from __future__ import print_function
import logging
import time
from subprocess import Popen, PIPE

from sandbox import sdk2
from sandbox.common import errors


logger = logging.getLogger('LogbrokerMirrorRepair.sandbox')
logger.setLevel(logging.DEBUG)


class LogbrokerMirrorRepair(sdk2.Task):

    ROBOT_KEY = 'ROBOT_LOGBROKER_OAUTH'
    ROBOT_OWNER = 'MR-USERS'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 128

    class Parameters(sdk2.Task.Parameters):
        description = 'Task for automatic repair of legacy logbroker mirror process.\n' \
                      'Designed for unattended scheduled execution.\n' \
                      'Will be decommissioned with logbroker mirror process.\n'

        kill_timeout = 60 * 60

        verbose = sdk2.parameters.Integer(
            'verbosity level (0 - inf)',
            default_value=0,
        )
        login_user = sdk2.parameters.String(
            'ssh remote command user and owner of sandbox vault entry for robot ssh key',
            required=True,
            default_value='robot-logbroker',
        )
        ssh_key_vault_entry = sdk2.parameters.String(
            'name of sandbox vault entry for robot ssh key',
            required=True,
            default_value='ROBOT_LOGBROKER_SSH_PRIVATE_KEY',
        )
        restart_period = sdk2.parameters.Integer(
            'restart period in seconds',
            default_value=60 * 60,
        )
        reassign_period = sdk2.parameters.Integer(
            'reassign period in seconds',
            default_value=60 * 60,
        )
        main_threshold = sdk2.parameters.Integer(
            'main threshold for automated acitons in seconds',
            default_value=10 * 60,
        )
        mirror_dc = sdk2.parameters.String(
            'list of DC to launch check (sas, man, vla, myt, iva, all), default: all',
            required=False,
            default_value='all',
        )
        actiontype = sdk2.parameters.String(
            'perform only specific type of action (only reassign, only restart or all of them), default: all',
            required=False,
            default_value='all',
        )
        stattype = sdk2.parameters.String(
            'how much data report to solomon (detail or aggregated), aggregated data reports always, detail means report reason for each partition, default: detail',
            required=False,
            default_value='detail',
        )
        dry_run = sdk2.parameters.Bool(
            'dry run (dont do tablet reassign or node restart)',
            default_value=False,
        )
        unstable = sdk2.parameters.Bool(
            'run unstable code',
            default_value=False,
        )

    def on_execute(self):
        logger.debug('calling on_execute')

        resource = sdk2.Resource['ARCADIA_PROJECT'].find(
            attrs=dict(logbroker_name='mirror_repair')
        ).first()
        if resource is None:
            raise errors.TaskError('cannot find resource')
        msg = 'using resource id %s' % resource.id
        self.set_info(msg)
        logger.debug(msg)

        script_path = str(sdk2.ResourceData(resource).path)

        # get secret key for script
        try:
            oauth_secret = sdk2.Vault.data(self.ROBOT_OWNER, self.ROBOT_KEY)
        except errors.VaultError:
            raise errors.TaskFailure("Cannot get key for %s" % self.ROBOT_KEY)

        cmd = [
            script_path,
            '-l', self.Parameters.login_user,
            '-s',
            '--restart-period', str(self.Parameters.restart_period),
            '--reassign-period', str(self.Parameters.reassign_period),
            '--main-threshold', str(self.Parameters.main_threshold),
            '--dc', str(self.Parameters.mirror_dc),
            '--actiontype', str(self.Parameters.actiontype),
            '--stattype', str(self.Parameters.stattype),
            '--token', oauth_secret,
        ]

        if self.Parameters.verbose > 0:
            cmd.append('-{}'.format('v' * self.Parameters.verbose))

        if self.Parameters.dry_run:
            cmd.append('--dry-run')

        logger.debug('prepared cmd: %s', cmd)

        if self.Parameters.unstable:
            logger.debug('entering unstable zone')

            logger_sp_formatter = logging.Formatter('%(message)s')

            logger_sp_out_filename = 'subprocess.out'
            logger_sp_out_handler = logging.FileHandler(logger_sp_out_filename)
            logger_sp_out_handler.setFormatter(logger_sp_formatter)
            logger_sp_out = logging.getLogger('LogbrokerMirrorRepair.subprocess.out')
            logger_sp_out.addHandler(logger_sp_out_handler)
            logger_sp_out.setLevel(logging.DEBUG)

            logger_sp_err_filename = 'subprocess.err'
            logger_sp_err_handler = logging.FileHandler(logger_sp_err_filename)
            logger_sp_err_handler.setFormatter(logger_sp_formatter)
            logger_sp_err = logging.getLogger('LogbrokerMirrorRepair.subprocess.err')
            logger_sp_err.addHandler(logger_sp_err_handler)
            logger_sp_err.setLevel(logging.DEBUG)

            with sdk2.ssh.Key(self, self.Parameters.login_user, self.Parameters.ssh_key_vault_entry):
                with sdk2.helpers.ProcessLog(self, logger=logger_sp_out) as pl_out:
                    with sdk2.helpers.ProcessLog(self, logger=logger_sp_err) as pl_err:
                        proc = Popen(cmd, shell=False, stdout=pl_out.stdout, stderr=pl_err.stdout)
                        stdout_file = open(logger_sp_out_filename)
                        stderr_file = open(logger_sp_err_filename)
                        while proc.poll() is None:
                            for line in stdout_file:
                                self.set_info(line.strip())

                            for line in stderr_file:
                                logger.debug(line.strip())

                            time.sleep(1)

                        for line in stdout_file:
                            self.set_info(line.strip())

                        for line in stderr_file:
                            logger.debug(line.strip())

                        if proc.returncode != 0:
                            raise errors.TaskError('command %s failed' % cmd)

        else:
            logger.debug('wait for finish process and write output')
            with sdk2.ssh.Key(self, self.Parameters.login_user, self.Parameters.ssh_key_vault_entry):
                proc = Popen(cmd, shell=False, stdout=PIPE, stderr=PIPE)
                stdout, stderr = proc.communicate()
                self.set_info(stdout.strip())
                for line in stderr.strip().split('\n'):
                    logger.debug(line)

                if proc.returncode != 0:
                    raise errors.TaskError('command %s failed' % cmd)
