import os
import yaml
import logging

import sandbox.common.errors as ce
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog
from sandbox.projects.logbroker.resources import LogbrokerTestData
from sandbox.projects.logbroker.common import ResourceTypeId, ScriptEnv


logger = logging.getLogger('task')
logger.setLevel(logging.DEBUG)


class TestParamsBlock(sdk2.parameters.String):
    multiline = True
    required = True
    description = 'Test params in yaml format'


class LogbrokerRunTest(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024 * 8

    class Parameters(sdk2.Parameters):
        params = TestParamsBlock()
        resources = ResourceTypeId('Resources to load (key=resource_type, value=resource_id)')
        secrets = ScriptEnv('Environment variables from vault (key=env_var, value=vault_owner,vault_key)')
        cmdline = sdk2.parameters.String('Cmd line to run test', required=True)

    def get_resource(self, resource_type, resource_id):
        logger.debug('searching for resource_type: %s, resource_id: %s', resource_type, resource_id)
        resource = sdk2.Resource[resource_type].find(id=resource_id).first()

        if resource is None:
            msg = 'Cannot find resource_type: %s, resource_id: %s' % (resource_type, resource_id)
            logger.error(msg)
            self.set_info(msg)
            raise ce.TaskError(msg)

        symlink = resource_type.lower()

        msg = 'Using resource_type: %s, resource_id: %s, link: %s' % (resource_type, resource.id, symlink)
        logger.debug(msg)
        self.set_info(msg)

        os.symlink(str(sdk2.ResourceData(resource).path), os.path.abspath(symlink))

        return resource

    def collect_test_data(self):
        logger.debug('archiving test data')
        with sdk2.helpers.ProcessLog(self, logger='tar') as pl:
            sp.Popen("tar -cf LogbrokerTestData.tgz --exclude synchrophazotron --exclude arcaphazotron ./*", shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()

        logger.debug('creating resource')
        test_data = sdk2.ResourceData(LogbrokerTestData(self, "Test data", "LogbrokerTestData.tgz"))
        test_data.ready()

    def on_execute(self):
        logger.debug('executing')
        try:
            params = yaml.safe_dump(yaml.safe_load(self.Parameters.params), indent=2)
            with open('params.yml', 'w') as file:
                file.write(params)

            resources = dict()
            for resource_type in self.Parameters.resources:
                resource_id = self.Parameters.resources[resource_type]
                resources[resource_type] = self.get_resource(resource_type, resource_id)

            for var in self.Parameters.secrets:
                value = self.Parameters.secrets[var]
                vault_owner, vault_key = value.split(',')
                logger.debug('loading vault data with owner %s, key %s', vault_owner, vault_key)
                os.environ[var] = sdk2.Vault.data(vault_owner, vault_key)

            logger.debug('test execution started')
            with ProcessLog(self, logger='test') as pl:
                proc = sp.Popen(self.Parameters.cmdline, shell=True, stdout=pl.stdout, stderr=sp.STDOUT)
                proc_rc = proc.wait()
            logger.debug('test execution finished with rc %s', proc_rc)

            result_files = [n for n in os.listdir('.') if n.startswith('result') and n.endswith('.yml')]
            logger.debug('result_files: %s', result_files)
            if len(result_files) != 1:
                raise ce.TaskError('number of result files is %s', len(result_files))

            with open(result_files[0]) as file:
                result_text = file.read()
                result_data = yaml.safe_load(result_text)

            logger.info(result_text)
            self.set_info(result_text)

            if result_data['status'] == 'failed':
                raise ce.TaskFailure('test failed')

            elif result_data['status'] == 'exception':
                raise ce.TaskError('test exception')

            elif result_data['status'] == 'success' and proc_rc > 0:
                raise ce.TaskError('test rc > 0')

        except ce.SandboxException:
            self.collect_test_data()
            raise

        except Exception as e:
            self.collect_test_data()
            logger.exception('unexpected exception')
            raise ce.TaskError(e.__class__.__name__)
