import os
import logging

import sandbox.common.errors as ce
from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog
from sandbox.projects.logbroker.resources import (
    LegacyReadDataInfoBinary,
    LegacyWriteDataInfoBinary,
    GRPCWriteDataInfoBinary,
    RetentionTimeInfoBinary
)


logger = logging.getLogger('sandbox-task')
logger.setLevel(logging.DEBUG)


logger_sp = logging.getLogger('sandbox-subprocess')
logger_sp.setLevel(logging.DEBUG)


resource_dict = {
    'legacy-http-read': LegacyReadDataInfoBinary,
    'legacy-http-write': LegacyWriteDataInfoBinary,
    'grpc-write': GRPCWriteDataInfoBinary,
    'retention-time': RetentionTimeInfoBinary,
}


class CollectUserInfo(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 2 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        description = 'Collect userinfo from logs and solomon graphics'

        with sdk2.parameters.String('Utility to get data') as utility:
            utility.values['legacy-http-read'] = utility.Value('legacy-http-read')
            utility.values['legacy-http-write'] = utility.Value('legacy-http-write')
            utility.values['grpc-write'] = utility.Value('grpc-write')
            utility.values['retention-time'] = utility.Value('retention-time')

        send_to_solomon = sdk2.parameters.Bool('Send data to solomon', default=True)
        num_days = sdk2.parameters.Integer('The number of dayly log files to analyse, start with yesterday file', default=14)

        token_yav_id = sdk2.parameters.String('Name of yav secret id', default='sec-01cyp205c3f4nc2mvfww09d1bz')
        token_yav_ver = sdk2.parameters.String('Name of yav secret id version', default='ver-01g4zccmzemyy7tr1nqa0msew1')
        token_logbroker = sdk2.parameters.String('Name of logbroker oauth token secret', default='robot-logbroker_lb_token')
        token_solomon = sdk2.parameters.String('Name of solomon oauth token secret', default='robot-logbroker_solomon_token')
        token_yql = sdk2.parameters.String('Name of yql oauth token secret', default='robot-logbroker_yql_token')

    def on_execute(self):
        logger.debug('calling on_execute')
        resource_binary = resource_dict[self.Parameters.utility]

        resource = resource_binary.find(
            attrs=dict(released='stable')
        ).first()

        if resource is None:
            msg = 'Cannot find %s resource' % resource_binary.name
            logger.debug(msg)
            raise ce.TaskError(msg)

        msg = 'Using resource id %s.' % resource.id
        logger.debug(msg)
        self.set_info(msg)

        script_path = str(sdk2.ResourceData(resource).path)

        secret = yav.Secret(self.Parameters.token_yav_id, self.Parameters.token_yav_ver)
        os.environ['LOGBROKER_TOKEN'] = secret.data()[self.Parameters.token_logbroker]
        os.environ['SOLOMON_TOKEN'] = secret.data()[self.Parameters.token_solomon]
        os.environ['YT_TOKEN'] = secret.data()[self.Parameters.token_yql]

        cmd = [script_path]

        if self.Parameters.send_to_solomon:
            cmd += ['--send-to-solomon']

        msg = 'Run %s: %s.' % (self.Parameters.utility, cmd)
        logger.debug(msg)
        self.set_info(msg)

        with ProcessLog(self, logger=logger_sp) as pl:
            proc = sp.Popen(cmd, shell=False, stdout=pl.stdout, stderr=pl.stdout)
            proc.communicate()
            if proc.returncode != 0:
                msg = 'Collectinfo failed with returncode %s.' % proc.returncode
                logger.debug(msg)
                raise ce.TaskError(msg)

        self.set_info('Done')
