from sandbox import sandboxsdk
from sandbox import sdk2

import sandbox.common.types.notification as ctn
import sandbox.common.types.task as ctt
from sandbox.projects.logfeller.common.deploy import DeployConfig, TestRunner
import sandbox.projects.logfeller.common.misc as misc

from release_task import ClusterReleaseTask, ReleaseRunner, ReleaseToNirvactorRunner, RELEASE_ENTITY_LOGS, RELEASE_ENTITY_PARSERS
from schedule import TaskSchedule

import datetime
import logging


ARCADIA_PATH_TO_DEPLOY_LOGS = "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/logs.json"
# ARCADIA_PATH_TO_DEPLOY_PARSERS = "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/parsers_exclude_robot_resources.json"
ARCADIA_PATH_TO_DEPLOY_PARSERS = "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/parsers.json"


class DeployLogfellerConfigs(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Settings") as settings_block:
            with sdk2.parameters.CheckGroup("YT clusters") as yt_clusters:
                yt_clusters.values.hahn = yt_clusters.Value("hahn", checked=True)
                yt_clusters.values.arnold = yt_clusters.Value("arnold", checked=True)
                yt_clusters.values.bohr = yt_clusters.Value("bohr", checked=True)
                yt_clusters.values.landau = yt_clusters.Value("landau", checked=True)
                yt_clusters.values.freud = yt_clusters.Value("freud", checked=True)
                yt_clusters.values.hume = yt_clusters.Value("hume", checked=True)

            with sdk2.parameters.CheckGroup("Release entity (choose only one)", required=True) as release_entity:
                release_entity.values.logs = yt_clusters.Value(RELEASE_ENTITY_LOGS, checked=False)
                release_entity.values.parsers = yt_clusters.Value(RELEASE_ENTITY_PARSERS, checked=False)

            # ---

            revision = sdk2.parameters.Integer(
                "Revision number",
                required=False
            )

            run_tests = sdk2.parameters.Bool(
                "Run tests",
                required=False,
                default=True
            )

            ignore_schedule_limitations = sdk2.parameters.Bool(
                "Ignore schedule limitations: only weekdays, from 10:00 till 18:00",
                required=False,
                default=False
            )

            validate = sdk2.parameters.Bool(
                "Validate (checking for: need_release, consistent_release)",
                required=False,
                default=True
            )

            allow_downgrade = sdk2.parameters.Bool(
                "Allow downgrade",
                required=False,
                default=False
            )

        with sdk2.parameters.Group("Build settings") as build_settings_block:
            use_last_binary = sdk2.parameters.Bool(
                "Use last binary archive",
                default=True
            )

            custom_tasks_archive_resource = sdk2.parameters.Resource(
                "task archive resource",
                default=None,
            )

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'target': 'logfeller/DeployLogfellerConfigs'}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    ###################################################################################################################
    # release

    def _is_release_logs(self):
        return RELEASE_ENTITY_LOGS in self.Parameters.release_entity

    def _is_release_parsers(self):
        return RELEASE_ENTITY_PARSERS in self.Parameters.release_entity

    def _raise_if_empty_release(self):
        if (not self._is_release_logs()) and (not self._is_release_parsers()):
            raise Exception("You started empty release - you should release logs or/and parsers")

    def _get_deploying_entities(self):
        if self._is_release_logs() and self._is_release_parsers():
            return "{} & {}".format(RELEASE_ENTITY_LOGS, RELEASE_ENTITY_PARSERS)
        elif self._is_release_logs():
            return RELEASE_ENTITY_LOGS
        elif self._is_release_logs():
            return RELEASE_ENTITY_PARSERS
        else:
            return ""

    ###################################################################################################################
    # vault

    @staticmethod
    def _get_yt_token():
        return sdk2.Vault.data("LOGFELLER_YT_TOKEN")

    @staticmethod
    def _get_nirvactor_token():
        return sdk2.Vault.data("LOGFELLER_STEP_TOKEN")

    ###################################################################################################################
    # context

    def set_deploy_time(self):
        self.Context.deploy_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M")

    ###################################################################################################################
    # revision

    def set_revision_to_release(self, deploy_source_path_logs, deploy_source_path_parsers):
        if deploy_source_path_logs:
            self.Context.revision_logs = int(self.get_revision_to_release(deploy_source_path_logs))
        if deploy_source_path_parsers:
            self.Context.revision_parsers = int(self.get_revision_to_release(deploy_source_path_parsers))

    def get_revision_to_release(self, source_path):
        if self.Parameters.revision:
            return self.Parameters.revision
        else:
            return self.get_revision_from_arcadia(source_path)

    def get_revision_from_arcadia(self, path):
        arcadia_info = sandboxsdk.svn.Arcadia.info(misc.make_arcadia_path(path))
        return arcadia_info["commit_revision"]

    ###################################################################################################################
    # utils

    def create_yt_client(self, proxy):
        import yt.wrapper as yt
        return yt.YtClient(
            proxy=proxy,
            token=self.get_yt_token()
        )

    def get_yt_token(self):
        return sdk2.Vault.data("LOGFELLER_YT_TOKEN")

    def raise_if_configs_tests_failed(self, deploy_config, revision):
        TestRunner(self, deploy_config.tests, revision).run_tests()

    def run_release_tasks(self):
        cluster_release_tasks = self._load_tasks_info_from_context()
        for task in cluster_release_tasks:
            if task.valid:
                if task.is_logs():
                    ReleaseToNirvactorRunner(self, task.yt_cluster, task.release_revision, self.Parameters.allow_downgrade).release_files_to_nirvactor()
                if task.is_parsers():
                    ReleaseRunner(self, task, self.Parameters.allow_downgrade).release_files_to_yt()

    def _should_exit_by_schedule_limitations(self):
        if self.Parameters.ignore_schedule_limitations:
            return False
        else:
            deploy_schedule = TaskSchedule(from_hour=10, to_hour=18)
            return not deploy_schedule.avail_now()

    def _get_logs_deployed_revision(self, yt_cluster):
        from logfeller.python.logfeller.infra.release.nirvactor.connectors.reactor_client_wrapper import ReactorClientWrapper
        from logfeller.python.logfeller.infra.release.nirvactor.utils import utils
        from logfeller.python.logfeller.infra.release.nirvactor.navigator.navigator_nirvactor import NavigatorNirvactorClusterLevel

        cfg_reactor_address, _ = utils.get_reactor_and_nirvana_servers(is_prod=True)
        reactor_client = ReactorClientWrapper(self._get_nirvactor_token(), cfg_reactor_address)
        return reactor_client.artifact_instance_last_value(
            NavigatorNirvactorClusterLevel(yt_cluster).get_reactor_path_to_public_logs_revision()
        )

    def init_release_tasks(self, deploy_config, deploy_config_path, release_entity_type, revision):
        release_tasks = []
        for yt_cluster in self.Parameters.yt_clusters:

            current_revision = None
            if self._is_release_logs():
                current_revision = self._get_logs_deployed_revision(yt_cluster)

            task = ClusterReleaseTask(
                yt_cluster_name=yt_cluster,
                yt_client=self.create_yt_client(yt_cluster),
                revision=revision,
                deploy_config=deploy_config,
                deploy_config_path=deploy_config_path,
                validate=self.Parameters.validate,
                allow_downgrade=self.Parameters.allow_downgrade,
                release_entity_type=release_entity_type,
                current_revision=current_revision,
            )
            release_tasks.append(task)
            self.Context.release_tasks.append(task.to_json())
        return release_tasks

    def has_pending_tasks(self, cluster_release_tasks):
        for task in cluster_release_tasks:
            logging.info("has_pending_tasks::task={}".format(task.to_json()))
            if task.valid:
                return True
        return False

    def _load_tasks_info_from_context(self):
        release_tasks = []
        if self.Context.release_tasks:
            for dumped_task in self.Context.release_tasks:
                release_task = ClusterReleaseTask(from_json=dumped_task)
                if release_task.yt_cluster in self.Context.release_tasks_ids:
                    release_task.id = self.Context.release_tasks_ids[release_task.yt_cluster]
                release_tasks.append(release_task)
        return release_tasks

    def get_finished_release_tasks(self):
        release_tasks = self._load_tasks_info_from_context()
        for child_task in self.find():
            for task in release_tasks:
                if child_task.id == task.id:
                    task.sandbox_task = child_task
        return release_tasks

    def send_email(self, cluster_release_tasks):
        def _build_email_body():
            return \
                """
 {task_link}
 {release_summary}
                """.format(
                    task_link="https://sandbox.yandex-team.ru/task/{}/view".format(self.id),
                    release_summary="\n".join([release_task.get_summary() for release_task in cluster_release_tasks])
                )

        def _get_total_status():
            release_count = 0
            for release in cluster_release_tasks:
                if release.sandbox_task or release.current_revision != release.release_revision:
                    release_count += 1
                    if not release.valid or release.sandbox_task.status != ctt.Status.SUCCESS:
                        return "FAIL"
            return "SUCCESS" if release_count else "NO UPDATES"

        self.server.notification(
            subject="{}: Deploy LogFeller configs {} at {}".format(_get_total_status(), self._get_deploying_entities(), self.Context.deploy_time),
            body=_build_email_body(),
            recipients=['logfeller-dev@yandex-team.ru'],
            transport=ctn.Transport.EMAIL,
        )

    ###################################################################################################################

    def on_execute(self):
        if self._should_exit_by_schedule_limitations():
            logging.warning("cannot execute task at time out of schedule, exiting...")
            return

        if self._is_release_logs() and self._is_release_parsers():
            logging.warning("you can choose only one: logs | parsers, exiting...")
            return

        self._raise_if_empty_release()

        deploy_configs_logs = DeployConfig.read_from_arcadia(ARCADIA_PATH_TO_DEPLOY_LOGS) if self._is_release_logs() else None
        deploy_configs_parsers = DeployConfig.read_from_arcadia(ARCADIA_PATH_TO_DEPLOY_PARSERS) if self._is_release_parsers() else None

        with self.memoize_stage.init:
            self.set_deploy_time()
            self.set_revision_to_release(
                deploy_configs_logs.source.path if deploy_configs_logs else None,
                deploy_configs_parsers.source.path if deploy_configs_parsers else None
            )

            self.Context.release_tasks = []
            release_tasks = []
            if deploy_configs_logs:
                release_tasks.extend(self.init_release_tasks(deploy_configs_logs, ARCADIA_PATH_TO_DEPLOY_LOGS, RELEASE_ENTITY_LOGS, self.Context.revision_logs))
            if deploy_configs_parsers:
                release_tasks.extend(self.init_release_tasks(deploy_configs_parsers, ARCADIA_PATH_TO_DEPLOY_PARSERS, RELEASE_ENTITY_PARSERS, self.Context.revision_parsers))

            logging.info("deploy_configs_logs={}".format(deploy_configs_logs))
            logging.info("deploy_configs_parsers={}".format(deploy_configs_parsers))

            if not self.has_pending_tasks(release_tasks):
                logging.info("nothing to release")
                self.send_email(release_tasks)
                return

        if self.Parameters.run_tests:
            if deploy_configs_logs:
                self.raise_if_configs_tests_failed(deploy_configs_logs, self.Context.revision_logs)
            if deploy_configs_parsers:
                self.raise_if_configs_tests_failed(deploy_configs_parsers, self.Context.revision_parsers)

        with self.memoize_stage.run_release_tasks:
            self.run_release_tasks()
            wait_statuses = [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK]
            release_tasks_ids = self.Context.release_tasks_ids.values() if self.Context.release_tasks_ids else []

            logging.info("release_tasks_ids: {}".format(release_tasks_ids))
            raise sdk2.WaitTask(release_tasks_ids, wait_statuses, wait_all=True)

        release_tasks = self.get_finished_release_tasks()
        self.send_email(release_tasks)
