from release_task import ClusterReleaseTask, ReleaseRunner, ReleaseToNirvactorRunner
from schedule import TaskSchedule

from sandbox.projects.logfeller.common.deploy import DeployConfig, TestRunner
import sandbox.projects.logfeller.common.misc as misc

from sandbox import sandboxsdk
from sandbox import sdk2
import sandbox.common.types.notification as ctn
import sandbox.common.types.task as ctt

import datetime
import logging


############################################################################


class DeployLogfellerConfigsToYT(sdk2.Task):
    """Release files to YT cluster"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup("YT clusters") as yt_clusters:
            yt_clusters.values.arnold = yt_clusters.Value("arnold", checked=True)
            yt_clusters.values.bohr = yt_clusters.Value("bohr", checked=True)
            yt_clusters.values.landau = yt_clusters.Value("landau", checked=True)
            yt_clusters.values.freud = yt_clusters.Value("freud", checked=True)
            yt_clusters.values.hahn = yt_clusters.Value("hahn", checked=True)
            yt_clusters.values.hume = yt_clusters.Value("hume", checked=True)

        deploy_config_path = sdk2.parameters.String(
            "Arcadia path to deploy config",
            default="arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/except_market_logs.json",
            required=True,
            choices=(
                ("all logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/logs.json"),
                ("except market logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/except_market_logs.json"),
                ("drug logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/drug_logs.json"),
                ("market logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/market_logs.json"),
                ("taxi logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/taxi_logs.json"),
                ("prod logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/prod_logs.json"),
                ("test logs", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/test_logs.json"),
                ("parsers", "arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/parsers.json")
            ),
        )

        revision = sdk2.parameters.Integer(
            "Revision number",
            required=False
        )

        deploy_to_nirvactor = sdk2.parameters.Bool(
            "Run child tasks DEPLOY_LOGFELLER_TO_NIRVACTOR",
            default=True
        )

    class Requirements(sdk2.Requirements):
        environments = [sandboxsdk.environments.PipEnvironment('yandex-yt')]

    def on_execute(self):
        deploy_schedule = TaskSchedule(from_hour=10, to_hour=18)
        if deploy_schedule.avail_now():
            deploy_config = DeployConfig.read_from_arcadia(self.Parameters.deploy_config_path)

            with self.memoize_stage.init:
                self.set_deploy_time()
                self.set_revision_to_release(deploy_config.source.path)
                release_tasks = self.init_release_tasks(deploy_config)
                if not self.has_pending_tasks(release_tasks):
                    logging.info("nothing to release")
                    self.send_email(release_tasks)
                    return

            self.raise_if_configs_tests_failed(deploy_config)

            with self.memoize_stage.run_release_tasks:
                self.run_release_tasks()
                wait_statuses = [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK]
                release_tasks_ids = self.Context.release_tasks_ids.values() if self.Context.release_tasks_ids else []

                raise sdk2.WaitTask(release_tasks_ids, wait_statuses, wait_all=True)

            release_tasks = self.get_finished_release_tasks()
            self.send_email(release_tasks)
        else:
            logging.warning("cannot execute task at time out of schedule, exiting...")

    def set_deploy_time(self):
        self.Context.deploy_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M")

    def set_revision_to_release(self, deploy_source_path):
        self.Context.revision = int(self.get_revision_to_release(deploy_source_path))

    def get_revision_to_release(self, source_path):
        if self.Parameters.revision:
            return self.Parameters.revision
        else:
            return self.get_revision_from_arcadia(source_path)

    def get_revision_from_arcadia(self, path):
        arcadia_info = sandboxsdk.svn.Arcadia.info(misc.make_arcadia_path(path))
        return arcadia_info["commit_revision"]

    def init_release_tasks(self, deploy_config):
        release_tasks = []
        self.Context.release_tasks = []
        for yt_cluster in self.Parameters.yt_clusters:
            task = ClusterReleaseTask(yt_cluster, self.create_yt_client(yt_cluster), self.Context.revision, deploy_config)
            release_tasks.append(task)
            self.Context.release_tasks.append(task.to_json())
        return release_tasks

    def create_yt_client(self, proxy):
        import yt.wrapper as yt
        return yt.YtClient(
            proxy=proxy,
            token=self.get_yt_token()
        )

    def get_yt_token(self):
        return sdk2.Vault.data("LOGFELLER_YT_TOKEN")

    def has_pending_tasks(self, cluster_release_tasks):
        for task in cluster_release_tasks:
            if task.valid:
                return True
        return False

    def raise_if_configs_tests_failed(self, deploy_config):
        TestRunner(self, deploy_config.tests, self.Context.revision).run_tests()

    def run_release_tasks(self):
        cluster_release_tasks = self._load_tasks_info_from_context()
        for task in cluster_release_tasks:
            if task.valid:
                ReleaseRunner(self, task).release_files_to_yt()
                if self.Parameters.deploy_to_nirvactor and (self.Parameters.deploy_config_path[-9:] == "logs.json"):
                    ReleaseToNirvactorRunner(self, task.yt_cluster, task.release_revision).release_files_to_nirvactor()

    def _load_tasks_info_from_context(self):
        release_tasks = []
        if self.Context.release_tasks:
            for dumped_task in self.Context.release_tasks:
                release_task = ClusterReleaseTask(from_json=dumped_task)
                if release_task.yt_cluster in self.Context.release_tasks_ids:
                    release_task.id = self.Context.release_tasks_ids[release_task.yt_cluster]
                release_tasks.append(release_task)
        return release_tasks

    def get_finished_release_tasks(self):
        release_tasks = self._load_tasks_info_from_context()
        for child_task in self.find():
            for task in release_tasks:
                if child_task.id == task.id:
                    task.sandbox_task = child_task
        return release_tasks

    def send_email(self, cluster_release_tasks):
        def _build_email_body():
            return \
                """
 {task_link}
 {release_summary}
                """.format(
                    task_link="https://sandbox.yandex-team.ru/task/{}/view".format(self.id),
                    release_summary="\n".join([release_task.get_summary() for release_task in cluster_release_tasks])
                )

        def _get_total_status():
            release_count = 0
            for release in cluster_release_tasks:
                if release.sandbox_task or release.current_revision != release.release_revision:
                    release_count += 1
                    if not release.valid or release.sandbox_task.status != ctt.Status.SUCCESS:
                        return "FAIL"
            return "SUCCESS" if release_count else "NO UPDATES"

        self.server.notification(
            subject="{}: Deploy LogFeller configs {} at {}".format(_get_total_status(), self.Parameters.deploy_config_path, self.Context.deploy_time),
            body=_build_email_body(),
            recipients=['logfeller-dev@yandex-team.ru'],
            transport=ctn.Transport.EMAIL,
        )

############################################################################
