from sandbox.projects.logfeller.common.deploy import SvnFetcher, DeployConfig
import sandbox.projects.logfeller.common.misc as misc

from sandbox import sandboxsdk
from sandbox import sdk2

import copy
import logging
import os
import sys

############################################################################


class ClusterReleaseTask(object):
    def __init__(self, yt_cluster_name=None, yt_client=None, revision=None, deploy_config=None, from_json=None):
        if from_json:
            self.init_from_json(from_json)
        else:
            self.init_from_args(yt_cluster_name, yt_client, revision, deploy_config)

    def init_from_json(self, config_json):
        self.id = config_json["id"]
        self.valid = config_json["valid"]
        self.validation_message = config_json["validation_message"]
        self.release_message = config_json["release_message"]
        self.yt_cluster = config_json["yt_cluster"]
        self.release_revision = config_json["release_revision"]
        self.deploy_config = DeployConfig.load_from_json(config_json["deploy_config"])
        self.change_log = config_json["change_log"]
        self.current_revision = config_json["current_revision"]

        self.sandbox_task = None

    def init_from_args(self, yt_cluster_name, yt_client, revision, deploy_config):
        self.id = None
        self.sandbox_task = None
        self.valid = False
        self.validation_message = None
        self.release_message = None

        self.yt_cluster = yt_cluster_name
        self.yt_client = yt_client
        self.release_revision = revision
        self.deploy_config = deploy_config

        self._describe_release()
        self._validate()

################################################################

    def to_json(self):
        return {
            "id": self.id,
            "valid": self.valid,
            "validation_message": self.validation_message,
            "release_message": self.release_message,
            "yt_cluster": self.yt_cluster,
            "release_revision": self.release_revision,
            "deploy_config": self.deploy_config.to_json(),
            "change_log": self.change_log,
            "current_revision": self.current_revision,
        }

################################################################

    def _describe_release(self):
        self.files_to_release = self._get_files_to_release()
        logging.debug("files_to_release: {}".format(self.files_to_release))

        self.release_log = self._get_release_log()
        self.released_files_revisions = self._get_released_files_revisions()
        logging.debug("released_files_revision: {}".format(",".join(["{}: {}".format(file.name, file.revision) for file in self.released_files_revisions])))

        self.current_revision = self._get_current_revision()
        logging.debug("current_revision: {}".format(self.current_revision))
        self.change_log = self._get_change_log()
        logging.debug("change_log: {}".format(self.change_log))

    def _get_release_log(self):
        sys.path.append(sandboxsdk.svn.Arcadia.get_arcadia_src_dir(
            "arcadia:/arc/trunk/arcadia/logfeller/python/logfeller/infra/release/yt"
        ))
        from log import ReleaseLog
        return ReleaseLog(self.yt_client, self.deploy_config.destination)

    def _get_files_to_release(self):
        svn_fetcher = SvnFetcher(
            self.deploy_config.source.path,
            self.deploy_config.source.files,
            self.release_revision
        )

        return svn_fetcher.list_files()

    def _get_released_files_revisions(self):
        all_released_files = self.release_log.list_released_files_revision()
        return [released_file for released_file in all_released_files if released_file.name in self.files_to_release]

    def _get_current_revision(self):
        return min([released_file.revision for released_file in self.released_files_revisions])

    def _get_change_log(self):
        def _prepare_commit(commit):
            prepared = copy.deepcopy(commit)
            prepared["date"] = commit["date"].strftime("%Y-%m-%d %H:%M:%S")
            return prepared

        def _filter_by_files_to_release(change_log):
            filtered_commits = []
            for commit in change_log:
                change_files_to_release = False
                for _, changed_file_path in commit["paths"]:
                    changed_file_name = changed_file_path.rsplit("/")[-1]
                    if changed_file_name in self.files_to_release:
                        change_files_to_release = True
                if change_files_to_release:
                    filtered_commits.append(_prepare_commit(commit))
            return filtered_commits

        all_commits = sandboxsdk.svn.Arcadia.log(misc.make_arcadia_path(self.deploy_config.source.path), revision_from=self.current_revision, revision_to=self.release_revision)
        return _filter_by_files_to_release(all_commits)

################################################################

    def _validate(self):
        if not self._check_downgrade():
            return False
        elif not self._check_need_release():
            return False
        elif not self._check_reviewed_changes():
            return False
        elif not self._check_consistent_release():
            return False
        else:
            self.valid = True
            self.validation_message = None
            self.release_message = self._make_release_message()
            return True

    def _check_downgrade(self):
        def make_error_message(files):
            return "some files have greater revision than {} : ({})".format(
                self.release_revision,
                ", ".join([
                    "({}: {})".format(
                        greater_file.name,
                        greater_file.revision
                    ) for greater_file in files
                ])
            )

        released_upper_files = self._get_files_over_revision(self.release_revision)
        logging.info("check downgrade: found {} files over release revision: {} {}".format(len(released_upper_files), self.release_revision, [file.name for file in released_upper_files]))
        if released_upper_files:
            self.valid = False
            self.validation_message = make_error_message(released_upper_files)
            return False
        return True

    def _get_files_over_revision(self, revision):
        files = []
        for released_file in self.released_files_revisions:
            if released_file.revision > revision:
                files.append(released_file)
        return files

    def _check_need_release(self):
        released_files_at_need_revision = set(self._get_files_at_revision(self.release_revision))
        files_to_release_by_config = set(self.files_to_release)
        if released_files_at_need_revision == files_to_release_by_config:
            logging.info("all files aready at release revision")
            self.valid = False
            self.validation_message = "all files are already at revision {}".format(self.release_revision)
            return False
        return True

    def _get_files_at_revision(self, revision):
        files = []
        for released_file in self.released_files_revisions:
            if released_file.revision == revision:
                files.append(released_file.name)
        return files

    def _check_reviewed_changes(self):
        def make_error_message(commits):
            return "not reviewed commits: \n{}".format(
                "\n".join([
                    "{revision}: {message} by {author}".format(
                        revision=commit["revision"],
                        message=commit["msg"],
                        author=commit["author"]
                    ) for commit in commits
                ])
            )

        not_reviewed_commits = []
        for commit in self.change_log:
            if "REVIEW:" not in commit["msg"] and commit["author"] != "robot-deployer-logs":
                logging.info("found not reviewed commit: {} {}".format(commit["revision"], commit["msg"]))
                not_reviewed_commits.append(commit)

        if not_reviewed_commits:
            self.valid = False
            self.validation_message = make_error_message(not_reviewed_commits)
            return False
        return True

    def _check_consistent_release(self):
        def path(changed_item):
            return changed_item[1]

        def status(changed_item):
            return changed_item[0]

        except_files = ["ya.make"]
        wont_be_consistently_released = []
        for commit in self.change_log:
            changed_files = [os.path.basename(path(changed)) for changed in commit["paths"] if self.deploy_config.source.path in path(changed) and status(changed) != "D"]
            logging.debug("commit {}, changed files from config: {}".format(commit["revision"], changed_files))
            wont_be_updated = []
            for changed_file in changed_files:
                if changed_file not in self.files_to_release and changed_file not in except_files:
                    wont_be_updated.append(changed_file)
            if wont_be_updated:
                logging.debug("commit {}, files are not going to be released: {}".format(commit["revision"], wont_be_updated))
                wont_be_consistently_released.append(commit["revision"])
        if wont_be_consistently_released:
            self.valid = False
            self.validation_message = "some commits require another deploy config that would cover all changes: {}".format(wont_be_consistently_released)
            return False
        return True

    def get_change_log_summary(self):
        return "\n".join([
            "{revision}: {message} by {author}".format(
                revision=commit["revision"],
                message=commit["msg"].encode("utf-8").strip().replace("\n", " "),
                author=commit["author"]
            ) for commit in self.change_log
        ])

    def _make_release_message(self):
        return "release files by config: {}".format(self.deploy_config.source.path)

################################################################

    def get_summary(self):
        def _compare_revisions_link():
            return "https://a.yandex-team.ru/arc/diff/trunk/arcadia/{deploy_source_path}?prevRev={current_revision}&rev={release_revision}"\
                .format(
                    deploy_source_path=self.deploy_config.source.path,
                    current_revision=self.current_revision,
                    release_revision=self.release_revision
                )

        def _sandbox_task_link():
            return "https://sandbox.yandex-team.ru/task/{id}/view".format(id=self.id)

        return \
            """
 Release configs {source_path} to {yt_cluster}:
   release_revision: {release_revision}
   prerelease_validation_status: {validation_status}
   prerelease_validation_message: {validation_message}
   sandbox_task: {sandbox_task_link}
   sandbox_task_status: {sandbox_task_status}
   release_commits: {arcanum_diff_link}
   release_change_log:
     {change_log_summary}
            """.format(
                source_path=self.deploy_config.source.path,
                yt_cluster=self.yt_cluster,
                release_revision=self.release_revision,
                validation_status="OK" if self.valid else "EXCEPTION",
                validation_message=self.validation_message,
                arcanum_diff_link=_compare_revisions_link() if self.valid else None,
                change_log_summary=self.get_change_log_summary() if self.valid else None,
                sandbox_task_link=_sandbox_task_link() if self.sandbox_task else None,
                sandbox_task_status=self.sandbox_task.status if self.sandbox_task else None
            )

############################################################################


class ReleaseRunner(object):
    def __init__(self, parent_task, task):
        self.parent_task = parent_task
        self.task_options = task

    def put_id_into_parent_context(self, id):
        if not self.parent_task.Context.release_tasks_ids:
            self.parent_task.Context.release_tasks_ids = dict()
        self.parent_task.Context.release_tasks_ids[self.task_options.yt_cluster] = id

    def release_files_to_yt(self):
        logging.info("release files from {source_path}(filter by {files}) at revision {revision} to yt-{yt_cluster}".format(
            source_path=self.task_options.deploy_config.source.path,
            files=", ".join(self.task_options.deploy_config.source.files),
            revision=self.task_options.release_revision,
            yt_cluster=self.task_options.yt_cluster
        ))

        release_parameters = {
            "yt_cluster": self.task_options.yt_cluster,
            "yt_token_vault_name": "LOGFELLER_YT_TOKEN",
            "deploy_config_path": self.parent_task.Parameters.deploy_config_path,
            "clean_up": False,
            "revision": self.task_options.release_revision,
            "message": self.task_options.release_message,
            "force": False,
            "run_tests": False
        }

        task_class = sdk2.Task["RELEASE_FILES_TO_YT"]
        release_task = task_class(
            self.parent_task,
            description="Release Logfeller configs to YT. Sub task of {}".format(self.parent_task.id),
            inherit_notifications=True,
            **release_parameters
        )

        release_task.enqueue()
        logging.info("enqueue release task with id {}".format(release_task.id))
        self.task_options.id = release_task.id
        self.put_id_into_parent_context(release_task.id)

############################################################################


class ReleaseToNirvactorRunner(object):
    def __init__(self, parent_task, yt_cluster, revision):
        self.parent_task = parent_task
        self.yt_cluster = str(yt_cluster)
        self.revision = revision

    def put_id_into_parent_context(self, id):
        if not self.parent_task.Context.release_tasks_ids:
            self.parent_task.Context.release_tasks_ids = dict()
        self.parent_task.Context.release_tasks_ids[self.yt_cluster] = id

    def release_files_to_nirvactor(self):
        logging.info("release files at revision {revision} to nirvactor-{yt_cluster}".format(
            revision=self.revision,
            yt_cluster=self.yt_cluster
        ))

        release_parameters = {
            "yt_cluster": self.yt_cluster,
            "revision": self.revision,
            "use_last_binary": True
        }

        task_class = sdk2.Task["DEPLOY_LOGFELLER_CONFIGS_TO_NIRVACTOR"]
        release_task = task_class(
            self.parent_task,
            description="Release Logfeller configs to Nirvactor. Sub task of {}".format(self.parent_task.id),
            inherit_notifications=True,
            **release_parameters
        )

        release_task.enqueue()
        logging.info("enqueue release task with id {}".format(release_task.id))
        self.put_id_into_parent_context(release_task.id)

############################################################################
