from configs import ConfigManager
from permissions import AllowedActions, AllAllowedPermissionsManager, SandboxPermissionsManager

from sandbox.projects.logfeller.common.deploy import run_release_files_to_yt_task

from sandbox import sdk2
import sandbox.common.types.task as ctt

import logging


class ModifyLogfellerConfigs(sdk2.Task):
    """Modify LogFeller configs (logs/streams) and release its to YT cluster"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Config changes') as config_chnages_block:
            logs_config_filename = sdk2.parameters.String(
                "logs config filename",
                required=True
            )
            streams_config_filename = sdk2.parameters.String(
                "streams config filename",
                required=True
            )
            log_tmpl_filename = sdk2.parameters.String(
                "log config template filename",
                default=None
            )
            stream_tmpl_filename = sdk2.parameters.String(
                "stream config template filename",
                default=None
            )
            logs_config_changes = sdk2.parameters.JSON('logs configs changes', required=True, default={})
            streams_config_changes = sdk2.parameters.JSON('streams configs changes', required=True, default={})
            logs_to_remove = sdk2.parameters.List('Logs list to remove')

        with sdk2.parameters.Group('Parsers changes') as parsers_changes_block:
            robot_parsers_config_changes = sdk2.parameters.JSON('parsers configs changes', default=None)
            robot_parsers_resource_file_name = sdk2.parameters.String('parsers resource name', default=None)
            robot_parsers_resource_content = sdk2.parameters.String('parsers resource content', default=None, multiline=True)

        with sdk2.parameters.Group('Commit information') as commit_block:
            committer = sdk2.parameters.String('committer login', required=True)
            ssh_key_vault_owner = sdk2.parameters.String('SSH key vault owner (task owner by default)', default=None)
            ssh_key_vault_name = sdk2.parameters.String('SSH key vault name', required=True)
            create_review = sdk2.parameters.Bool('Create review, otherwise commit', default=False)

        with create_review.value[False]:
            with sdk2.parameters.Group('Release configs') as release_configs_block:
                release_configs = sdk2.parameters.Bool('Release configs', default=False)
                with release_configs.value[True]:
                    yt_cluster = sdk2.parameters.String('YT cluster')
                    yt_token_vault_owner = sdk2.parameters.String('YT token vault owner')
                    yt_token_vault_name = sdk2.parameters.String('YT token vault name')
                    release_config_filename = sdk2.parameters.String('Release config filename')

        with sdk2.parameters.Group('Additional settings') as additional_settings_block:
            dry_run = sdk2.parameters.Bool("Dry run mode", default=False)

    def on_enqueue(self):
        if not self.Parameters.ssh_key_vault_owner:
            self.Parameters.ssh_key_vault_owner = self.Parameters.owner

        if self.Parameters.create_review:
            self.Parameters.release_configs = False
        else:
            self.Context.semaphore_name = 'LOGFELLER_COMMIT_CONFIGS_' + self.Parameters.owner
            semaphores = [ctt.Semaphores.Acquire(name=self.Context.semaphore_name, weight=1, capacity=1)]

            if self.change_parsers():
                semaphores.append(ctt.Semaphores.Acquire(name='LOGFELLER_COMMIT_ROBOT_PARSERS', weight=1, capacity=1))

            self.Requirements.semaphores = ctt.Semaphores(
                acquires=semaphores,
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            )

    def validate_filename(self, filename):
        prefix = 'autoupdate.{}.'.format(self.Parameters.owner.lower())
        if not filename.startswith(prefix):
            raise Exception('You can commit only to files with prefix: "{}".'.format(prefix))

    def commit_changes(self):
        configs = ConfigManager(
            logs_filename=self.Parameters.logs_config_filename,
            streams_filename=self.Parameters.streams_config_filename,
            log_tmpl_filename=self.Parameters.log_tmpl_filename,
            stream_tmpl_filename=self.Parameters.stream_tmpl_filename,
            dry_run=self.Parameters.dry_run,
            modify_permissions=self.modify_permissions
        )

        configs.modify_logs(self.Parameters.logs_config_changes, self.Parameters.streams_config_changes)
        configs.remove_logs(self.Parameters.logs_to_remove)

        if self.change_parsers():
            configs.modify_robot_parsers(
                parsers_diff=self.Parameters.robot_parsers_config_changes,
                resource_name=self.Parameters.robot_parsers_resource_file_name,
                resource_content=self.Parameters.robot_parsers_resource_content,
            )

        prefix_commit_msg = 'SB task #{} - logs configs changes for {}'.format(self.id, self.Parameters.owner)

        return configs.save_and_commit(
            self.Parameters.create_review,
            self.Parameters.committer,
            prefix_commit_msg,
            self.Parameters.ssh_key_vault_owner,
            self.Parameters.ssh_key_vault_name
        )

    def release_configs(self):
        return run_release_files_to_yt_task(
            self.Parameters.yt_cluster,
            self.Parameters.yt_token_vault_owner,
            self.Parameters.yt_token_vault_name,
            self.Parameters.release_config_filename,
            self.Context.committed_revision,
            'Auto release logs configs to YT  cluster {} for {} with revision {} from task #{}'.format(
                self.Parameters.yt_cluster,
                self.Parameters.owner,
                self.Context.committed_revision,
                self.id
            ),
            self
        )

    def validate_permissions(self):
        if not self.Parameters.create_review:
            self.modify_permissions.verify(AllowedActions.COMMIT)
        if self.Parameters.release_configs:
            if self.modify_permissions.check(AllowedActions.COMMIT):
                raise Exception('You must have permission COMMIT for release configs')
            if self.modify_permissions.check(AllowedActions.RELEASE):
                raise Exception('You must have permission RELEASE for release configs')

    def change_parsers(self):
        return self.Parameters.robot_parsers_config_changes is not None

    def validate_parsers_parameters(self):
        if self.Parameters.robot_parsers_resource_file_name is not None:
            resource_name = self.Parameters.robot_parsers_resource_file_name
            if not resource_name.startswith("robot_resources/"):
                raise Exception("parsers resource name must starts with robot_resources/")
            name_tokens = resource_name.split("/")
            if len(name_tokens) > 2:
                raise Exception("parsers resource name should have only one nested path - robot_resources")

            if self.Parameters.robot_parsers_resource_content is None:
                raise Exception("parsers resource content should be defined")

        if self.Parameters.robot_parsers_resource_content is not None and self.Parameters.robot_parsers_resource_file_name is None:
            raise Exception("both parameters: parsers resource name and resource content should be defined or not at the same time")

    def on_execute(self):
        self.modify_permissions = None
        if self.Parameters.create_review:
            self.modify_permissions = AllAllowedPermissionsManager()
        else:
            self.validate_filename(self.Parameters.logs_config_filename)
            self.validate_filename(self.Parameters.streams_config_filename)
            self.modify_permissions = SandboxPermissionsManager(self.Parameters.owner)

        if self.change_parsers():
            self.validate_parsers_parameters()

        self.validate_permissions()

        with self.memoize_stage.commit:
            self.Context.committed_revision = self.commit_changes()

            logging.info('Release semaphore: {}'.format(self.Context.semaphore_name))
            sdk2.Requirements.semaphores.release()

        if self.Parameters.release_configs:
            if self.Context.committed_revision is None:
                logging.warning('Nothing to release')
            else:
                with self.memoize_stage.release_configs:
                    self.modify_permissions.verify(AllowedActions.RELEASE)
                    self.Context.release_task_id = self.release_configs()

                    wait_statuses = [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK]
                    raise sdk2.WaitTask(self.Context.release_task_id, wait_statuses)

                release_task = self.find(id=self.Context.release_task_id).first()
                release_task_result = 'Release task #{} finished with status: {}'.format(self.Context.release_task_id, release_task.status)
                logging.info(release_task_result)
                if release_task.status != ctt.Status.SUCCESS:
                    raise Exception(release_task_result)
