from permissions import AllowedActions

from sandbox.projects.common.arcadia import sdk as arcadiasdk
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.svn import Arcadia
from sandbox import sdk2

import logging
import enum
import os
import re
import json
from collections import OrderedDict


class ConfigType(enum.Enum):
    log = 0
    stream = 1
    robot_parser = 2
    robot_parser_resource = 3


class ConfigManager(object):
    def __init__(
        self,
        logs_filename,
        streams_filename,
        log_tmpl_filename=None,
        stream_tmpl_filename=None,
        dry_run=False,
        modify_permissions=None
    ):
        self.__changes = []
        self._modify_parsers = False

        self._dry_run = dry_run
        self._modify_permissions = modify_permissions

        self._checkout_configs(logs_filename, streams_filename, log_tmpl_filename, stream_tmpl_filename)
        self._load_configs()

    def _logging_changes(self, action, type, subject):
        event = '{} {} {}'.format(action.name, type.name, subject)
        logging.info(event)
        self.__changes.append(event)

    def _checkout_configs(self, logs_filename, streams_filename, log_tmpl_filename=None, stream_tmpl_filename=None):
        self._arcadia_path = Arcadia.get_arcadia_src_dir(Arcadia.trunk_url())

        self._configs_path = os.path.join(self._arcadia_path, "logfeller/configs")

        self._parsers_path = os.path.join(self._configs_path, "parsers")
        self._robot_parsers_resources_path = os.path.join(self._parsers_path, "robot_resources")
        self._robot_parsers_path = os.path.join(self._parsers_path, "parsers.robot.json")

        self._all_logs_configs_path = os.path.join(self._configs_path, "logs")
        self._logs_path = os.path.join(self._all_logs_configs_path, logs_filename)
        self._streams_path = os.path.join(self._all_logs_configs_path, streams_filename)

        self._templates_path = os.path.join(self._arcadia_path, "logfeller/deploy/sandbox")
        self._log_tmpl_path = None
        self._stream_tmpl_path = None
        if log_tmpl_filename:
            self._log_tmpl_path = os.path.join(self._templates_path, log_tmpl_filename)
        if stream_tmpl_filename:
            self._stream_tmpl_path = os.path.join(self._templates_path, stream_tmpl_filename)

        self._exists_directory(self._configs_path)
        self._exists_directory(self._parsers_path)
        self._exists_file(self._robot_parsers_path)
        self._exists_directory(self._robot_parsers_resources_path)
        self._exists_directory(self._all_logs_configs_path)
        self._exists_directory(self._templates_path)
        self._exists_file(self._logs_path)
        self._exists_file(self._streams_path)
        self._exists_file(self._log_tmpl_path)
        self._exists_file(self._stream_tmpl_path)

    def _exist_item(self, path):
        if os.path.exists(path):
            return True
        raise Exception('Item not exist: {}'.format(path))

    def _exists_directory(self, path):
        if self._exist_item(path) and os.path.isdir(path):
            logging.info('Directory {}: {}'.format(path, ''.join(['\n - ' + x for x in os.listdir(path)])))
        else:
            raise Exception('It is not a directory: {}'.format(path))

    def _exists_file(self, path):
        if path is None:
            return
        if self._exist_item(path) and os.path.isfile(path):
            logging.info('File exist {}.'.format(path))
        else:
            raise Exception('It is not a file: {}'.format(path))

    def _get_config_name(self, type):
        return '_{}_items_config'.format(type.name)

    def _get_template_name(self, type):
        return '_{}_item_template'.format(type.name)

    def _set_attr_from_file(self, attr_name, path, default=None):
        data = default
        if path:
            data = self._load_from_file(path)
        setattr(self, attr_name, data)

    def _set_template(self, type, path):
        self._set_attr_from_file(self._get_template_name(type), path)

    def _set_config(self, type, path):
        self._set_attr_from_file(self._get_config_name(type), path)

    def _get_config(self, type):
        return getattr(self, self._get_config_name(type))

    def _get_template(self, type):
        return getattr(self, self._get_template_name(type))

    def _load_configs(self):
        self._set_config(ConfigType.log, self._logs_path)
        self._set_config(ConfigType.stream, self._streams_path)
        self._set_config(ConfigType.robot_parser, self._robot_parsers_path)

        self._set_template(ConfigType.log, self._log_tmpl_path)
        self._set_template(ConfigType.stream, self._stream_tmpl_path)

    def _load_from_file(self, file_path):
        with open(file_path, "r") as f:
            return json.load(f, object_pairs_hook=OrderedDict)

    def _dump_configs(self):
        logs = self._get_config(ConfigType.log)
        streams = self._get_config(ConfigType.stream)
        robot_parsers = self._get_config(ConfigType.robot_parser)

        self._dump_to_file(self._logs_path, logs)
        self._dump_to_file(self._streams_path, streams)
        self._dump_to_file(self._robot_parsers_path, robot_parsers)

    def _dump_to_file(self, file_path, data):
        with open(file_path, "w") as f:
            f.write(json.dumps(data, indent=4, separators=(',', ': ')))

    def _get_topic_from_stream(self, stream_str):
        # @streams:cluster:topic
        parts = stream_str.split(':')
        if len(parts) != 3:
            raise Exception('Cannot parse stream string to get topic name: {}'.format(stream_str))
        return parts[2]

    def _create_item(self, conf_type, name, data):
        if conf_type == ConfigType.robot_parser:
            return data

        if type(data) is dict:
            return data

        template = self._get_template(conf_type)
        if template is None:
            raise Exception('Cannot create {} with name "{}": no template to complite it'.format(conf_type.name, name))

        item = template.copy()
        if conf_type is ConfigType.log:
            if type(data) is not list:
                raise Exception('Cannot create {} with name "{}": incorrect format for template'.format(conf_type.name, name))
            item['name'] = name
            stream_prefix_label = '__stream_prefix__'
            stream_prefix = template[stream_prefix_label]
            item.pop(stream_prefix_label)
            item['streams'] = [stream_prefix + topic for topic in data]
        elif conf_type is ConfigType.stream:
            if data is not None:
                raise Exception('Cannot create {} with name "{}": incorrect format for template'.format(conf_type.name, name))
            item['topic_path'] = name
        else:
            raise Exception('Unknown config type: {}'.format(type))
        return item

    def _replace_item(self, type, configs, name, data):
        data = self._create_item(type, name, data)
        if configs[name] == data:
            logging.warning('Do not replace {} with name={}, because the data is the same'.format(type.name, name))
            return
        self._modify_permissions.verify(AllowedActions.REPLACE)
        configs[name] = data

        self._logging_changes(AllowedActions.REPLACE, type, name)

    def _add_item(self, type, configs, name, data):
        self._modify_permissions.verify(AllowedActions.ADD)

        data = self._create_item(type, name, data)
        configs[name] = data

        self._logging_changes(AllowedActions.ADD, type, name)

    def _remove_log(self, log):
        self._modify_permissions.verify(AllowedActions.REMOVE)
        logs = self._get_config(ConfigType.log)
        streams = self._get_config(ConfigType.stream)
        if log not in logs:
            logging.warning('No exist log to remove: {}'.format(log))
            return
        for stream in logs[log]['streams']:
            topic = self._get_topic_from_stream(stream)
            if topic not in streams:
                raise Exception('Incorrect configuration: no topic={} (stream={}) for log={} in current configs'.format(topic, stream, log))
            streams.pop(topic)
            self._logging_changes(AllowedActions.REMOVE, ConfigType.stream, topic)
        logs.pop(log)
        self._logging_changes(AllowedActions.REMOVE, ConfigType.log, log)

    def _apply_diff(self, type, configs_diff):
        config = self._get_config(type)
        for name, data in configs_diff.items():
            if name in config:
                self._replace_item(type, config, name, data)
            else:
                self._add_item(type, config, name, data)

    def modify_logs(self, logs_diff, streams_diff):
        self._apply_diff(ConfigType.log, logs_diff)
        self._apply_diff(ConfigType.stream, streams_diff)

    def remove_logs(self, logs):
        for log in logs:
            self._remove_log(log)

    def _save_robot_parser_resource(self, resource_name, resource_content):
        def write_resource_content_to_file():
            resource_basename = os.path.basename(resource_name)
            resource_path = os.path.join(self._robot_parsers_resources_path, resource_basename)

            is_new_resource = not os.path.exists(resource_path)

            with open(resource_path, 'w') as f:
                f.write(resource_content)

            if is_new_resource:
                self._modify_permissions.verify(AllowedActions.ADD)
                self._logging_changes(AllowedActions.ADD, ConfigType.robot_parser_resource, resource_name)
                sdk2.svn.Arcadia.add(resource_path)
            else:
                self._modify_permissions.verify(AllowedActions.REPLACE)
                self._logging_changes(AllowedActions.REPLACE, ConfigType.robot_parser_resource, resource_name)

        def update_robot_resources_makefile():
            import subprocess

            command = "cd {robot_resources_dir} && ./update_makefile.sh && cd {current_dir}".format(
                robot_resources_dir=self._robot_parsers_resources_path,
                current_dir=os.getcwd()
            )
            subprocess.call(command, shell=True)

        self._modify_parsers = True
        write_resource_content_to_file()
        update_robot_resources_makefile()

    def modify_robot_parsers(self, parsers_diff, resource_name, resource_content):
        self._apply_diff(ConfigType.robot_parser, parsers_diff)

        if resource_name:
            self._save_robot_parser_resource(resource_name, resource_content)

############################################################################

    def _run_configs_tests(self):
        result_code = arcadiasdk.do_build(
            consts.YMAKE_BUILD_SYSTEM,
            self._arcadia_path,
            ["logfeller/configs/tests"],
            clear_build=False,
            test=True
        )
        logging.info('Configs tests finished with code={}'.format(result_code))

        return False if result_code else True

    def _run_parsers_tests(self):
        result_code = arcadiasdk.do_build(
            consts.YMAKE_BUILD_SYSTEM,
            self._arcadia_path,
            ["logfeller/lib/log_parser/tests"],
            clear_build=False,
            test=True
        )
        logging.info('Parsers tests finished with code={}'.format(result_code))

        return False if result_code else True

    def _run_tests(self):
        configs_test_passed = self._run_configs_tests()
        parsers_test_passed = True if not self._modify_parsers or not configs_test_passed else self.run_parsers_test()

        return configs_test_passed and parsers_test_passed

    def __extract_committed_revision(self, output):
        committed_revision_re = re.compile("(.|\n)*Committed revision (?P<revision>[0-9]+)(.|\n)*")
        match = committed_revision_re.match(output)
        if not match:
            return None
        else:
            return match.group("revision")

    def __extract_created_review(self, output):
        committed_revision_re = re.compile("(.|\n)*Commit is uploaded to Arcanum( |\n)*https://a.yandex-team.ru/review/(?P<review>[0-9]+)(.|\n)*")
        match = committed_revision_re.match(output)
        if not match:
            return None
        else:
            return match.group("review")

    def _commit(self, committer, commit_message, ssh_key_vault_owner, ssh_key_vault_name):
        with sdk2.ssh.Key(self, ssh_key_vault_owner, ssh_key_vault_name):
            commit_output = sdk2.svn.Arcadia.commit(
                self._configs_path,
                commit_message,
                user=committer
            )
            revision = self.__extract_committed_revision(commit_output)
            if revision:
                return revision
            raise Exception(commit_output)

    def _create_review(self, committer, commit_message, ssh_key_vault_owner, ssh_key_vault_name):
        with sdk2.ssh.Key(self, ssh_key_vault_owner, ssh_key_vault_name):
            try:
                sdk2.svn.Arcadia.commit(
                    self._configs_path,
                    commit_message,
                    user=committer
                )
            except Exception as e:
                review = self.__extract_created_review(str(e))

                if review:
                    return review
                else:
                    raise e

            raise Exception("unexpected behavior when creating a review")

    def run_tests(self):
        if len(self.__changes) == 0:
            logging.info('No changes in logs. We didn\'t run tests.')
            return None

    def save_and_commit(self, create_review, committer, prefix_commit_msg, ssh_key_vault_owner, ssh_key_vault_name):
        if len(self.__changes) == 0:
            logging.info('No changes in logs.')
            return None

        self._dump_configs()

        commit_hook = 'REVIEW:NEW' if create_review else 'SKIP_CHECK'
        commit_message = '{message} {hook}'.format(
            message=prefix_commit_msg,
            hook=commit_hook
        )

        if self._dry_run:
            logging.info('Do not commit because of the dry-run mode')
            return None

        if create_review:
            return self._create_review(committer, commit_message, ssh_key_vault_owner, ssh_key_vault_name)
        else:
            if not self._run_tests():
                raise Exception("Failed tests")

            return self._commit(committer, commit_message, ssh_key_vault_owner, ssh_key_vault_name)
