from sandbox.projects.logfeller.common.deploy import DeployConfig, TestRunner, SvnFetcher

from sandbox import sandboxsdk
from sandbox import sdk2

import sys

############################################################################


class ReleaseFilesToYT(sdk2.Task):
    """Release files to YT cluster"""

    class Parameters(sdk2.Task.Parameters):
        yt_cluster = sdk2.parameters.String(
            "YT cluster",
            default="hume",
            required=True,
            choices=(("hume", "hume"), ("hahn", "hahn"), ("arnold", "arnold"), ("freud", "freud"), ("bohr", "bohr"), ("landau", "landau"))
        )

        yt_token_vault_owner = sdk2.parameters.String(
            "YT tokent vault owner",
            default=None,
        )

        yt_token_vault_name = sdk2.parameters.String(
            "YT tokent vault name",
            required=True,
        )

        deploy_config_path = sdk2.parameters.String(
            "Arcadia path to deploy config",
            required=True
        )

        clean_up = sdk2.parameters.Bool(
            "Remove all existing files before release",
            required=True
        )

        incremental = sdk2.parameters.Bool(
            "Incremental release (do not upload unchanged files to YT)",
            default=True,
            required=False
        )

        revision = sdk2.parameters.Integer(
            "Revision number",
            required=True
        )
        message = sdk2.parameters.String(
            "Release message, say something to your colleagues",
        )
        force = sdk2.parameters.Bool(
            "Force",
            default=False
        )
        run_tests = sdk2.parameters.Bool(
            "Run tests",
            default=True
        )

    class Requirements(sdk2.Requirements):
        environments = [sandboxsdk.environments.PipEnvironment('yandex-yt')]

    def on_enqueue(self):
        if not self.Parameters.yt_token_vault_owner:
            self.Parameters.yt_token_vault_owner = self.Parameters.owner

    def get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)

    def create_yt_client(self):
        import yt.wrapper as yt
        return yt.YtClient(
            proxy=self.Parameters.yt_cluster,
            token=self.get_yt_token()
        )

    def get_release_manager_class(self, yt_client, destination):
        sys.path.append(sandboxsdk.svn.Arcadia.get_arcadia_src_dir(
            "arcadia:/arc/trunk/arcadia/logfeller/python/logfeller/infra/release/yt"
        ))
        from manager import ReleaseManager
        return ReleaseManager(yt_client, destination)

    def on_execute(self):
        deploy_config = DeployConfig.read_from_arcadia(self.Parameters.deploy_config_path)

        if self.Parameters.run_tests:
            TestRunner(self, deploy_config.tests, self.Parameters.revision).run_tests()

        svn_fetcher = SvnFetcher(
            deploy_config.source.path,
            deploy_config.source.files,
            self.Parameters.revision
        )
        fetched_dir, fetched_files = svn_fetcher.fetch_files()

        release_manager = self.get_release_manager_class(self.create_yt_client(), deploy_config.destination)
        release_manager.release(
            fetched_dir,
            fetched_files,
            self.Parameters.revision,
            self.author,
            self.Parameters.message,
            force=self.Parameters.force,
            clean_up=self.Parameters.clean_up,
            incremental=self.Parameters.incremental
        )

############################################################################
