from sandbox.projects.logfeller.common.deploy import DeployConfig, TestRunner, SvnFetcher

from sandbox import sandboxsdk
from sandbox import sdk2

############################################################################


class ReleaseParsersToYT(sdk2.Task):
    """Release files to YT cluster"""

    class Parameters(sdk2.Task.Parameters):
        yt_cluster = sdk2.parameters.String(
            "YT cluster",
            default="hume",
            required=True,
            choices=(("hahn", "hahn"), ("arnold", "arnold"), ("bohr", "bohr"), ("landau", "landau"), ("freud", "freud"), ("hume", "hume"))
        )

        revision = sdk2.parameters.Integer(
            "Revision number",
            required=True
        )

        # ---

        yt_token_vault_owner = sdk2.parameters.String(
            "YT token vault owner",
            default="LOGFELLER",
        )

        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            default="LOGFELLER_YT_TOKEN",
        )

        deploy_config_path = sdk2.parameters.String(
            "Arcadia path to deploy config",
            default="arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/parsers.json"
        )

        clean_up = sdk2.parameters.Bool(
            "Remove all existing files before release",
            default=False,
            required=False
        )

        incremental = sdk2.parameters.Bool(
            "Incremental release (do not upload unchanged files to YT)",
            default=True,
            required=False
        )

        message = sdk2.parameters.String(
            "Release message, say something to your colleagues",
        )
        allow_downgrade = sdk2.parameters.Bool(
            "Allow downgrade",
            default=False
        )
        run_tests = sdk2.parameters.Bool(
            "Run tests",
            default=True
        )

        with sdk2.parameters.Group("Build settings") as build_settings_block:
            use_last_binary = sdk2.parameters.Bool(
                "Use last binary archive",
                default=True
            )

            custom_tasks_archive_resource = sdk2.parameters.Resource(
                "task archive resource",
                default=None,
            )

    ########################################################################################

    class Requirements(sdk2.Requirements):
        environments = [sandboxsdk.environments.PipEnvironment('yandex-yt')]

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'target': 'logfeller/ReleaseParsersToYT'}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def on_enqueue(self):
        if not self.Parameters.yt_token_vault_owner:
            self.Parameters.yt_token_vault_owner = self.Parameters.owner

    def get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)

    def create_yt_client(self):
        import yt.wrapper as yt
        return yt.YtClient(
            proxy=self.Parameters.yt_cluster,
            token=self.get_yt_token()
        )

    def get_release_manager(self, yt_client, destination):
        from logfeller.infra.release.yt.manager import ReleaseManager
        return ReleaseManager(yt_client, destination)

    def on_execute(self):
        deploy_config = DeployConfig.read_from_arcadia(self.Parameters.deploy_config_path)

        if self.Parameters.run_tests:
            TestRunner(self, deploy_config.tests, self.Parameters.revision).run_tests()

        svn_fetcher = SvnFetcher(
            deploy_config.source.path,
            deploy_config.source.files,
            self.Parameters.revision
        )
        local_dir, local_files = svn_fetcher.fetch_files()

        release_manager = self.get_release_manager(self.create_yt_client(), deploy_config.destination)
        release_manager.release(
            local_dir,
            local_files,
            self.Parameters.revision,
            self.author,
            self.Parameters.message,
            force=self.Parameters.allow_downgrade,
            clean_up=self.Parameters.clean_up,
            incremental=self.Parameters.incremental
        )
