from sandbox.projects.common.build import parameters, YaPackage

from sandbox import sandboxsdk
from sandbox import sdk2
import sandbox.common.types.task as task_types

import logging
import json
import os
import re

############################################################################


class DeployConfig(object):
    class Source(object):
        def __init__(self, path, files):
            self.path = path
            self.files = files

    def __init__(self, source, destination, tests):
        self.source = source
        self.destination = destination
        self.tests = tests

    def to_json(self):
        return {
            "source": {
                "path": self.source.path,
                "files": self.source.files
            },
            "destination": self.destination,
            "tests": self.tests
        }

    @staticmethod
    def read_from_arcadia(path):
        logging.info("read deploy config from {}".format(path))
        config_json = json.loads(sandboxsdk.svn.Arcadia.cat(path))
        return DeployConfig.load_from_json(config_json)

    @staticmethod
    def load_from_json(config_json):
        return DeployConfig(
            source=DeployConfig.Source(
                path=config_json["source"]["path"],
                files=config_json["source"]["files"],
            ),
            destination=config_json["destination"],
            tests=config_json["tests"],
        )

############################################################################


# timmyb32r note: this class assume there are ONLY ONE child task YA_PACKAGE (for tests)
class TestRunner(object):
    def __init__(self, parent_task, targets, revision):
        self.parent_task = parent_task
        self.targets = targets
        self.revision = revision

    def generate_package_config(self):
        return json.dumps({
            "meta": {
                "name": "yandex-fake-package-sandbox-test",
                "version": "{revision}",
                "description": "fake package for sandbox testing",
                "maintainer": "Totally Notme <totallynotme@yandex-team.ru>"
            },
            "build": {
                "targets": self.targets
            },
            "data": []
        })

    def find_ya_package_task(self):
        for child_task in self.parent_task.find():
            if child_task.id == self.parent_task.Context.ya_package_task_id:
                return child_task

    def run_tests(self):
        logging.info("running tests for [{}], revision {}".format(
            ", ".join(self.targets), self.revision
        ))

        with self.parent_task.memoize_stage.create_children:
            ya_package_parameters = {
                "acceptance_testing": True,
                parameters.ArcadiaUrl.name: "arcadia:/arc/trunk/arcadia@{}".format(self.revision),
                YaPackage.AdhocPackagesParameter.name: [self.generate_package_config()],
                YaPackage.RunTestsParameter.name: True,
                YaPackage.RunLongTestsParameter.name: True,
                YaPackage.parameters.CheckoutParameter.name: True
            }

            task_class = sdk2.Task["YA_PACKAGE"]
            ya_package_task = task_class(
                self.parent_task,
                description="YA_PACKAGE for check tests. Sub task of {}".format(self.parent_task.id),
                inherit_notifications=True,
                **ya_package_parameters
            )

            ya_package_task.enqueue()

            self.parent_task.Context.ya_package_task_id = ya_package_task.id

            raise sdk2.WaitTask(ya_package_task, [task_types.Status.Group.FINISH, task_types.Status.Group.BREAK], wait_all=True)

        ya_package_task = self.find_ya_package_task()

        logging.info("task with configs tests finished with status {}".format(ya_package_task.status))
        if ya_package_task.status != task_types.Status.SUCCESS:
            raise Exception("failed to run tests in subtask {}".format(ya_package_task.id))

############################################################################


class SvnFetcher(object):
    def __init__(self, path, file_patterns, revision):
        self.path = path
        self.file_patterns = list(map(re.compile, file_patterns))
        self.revision = revision

    def is_file_matches(self, name):
        for pattern in self.file_patterns:
            if pattern.match(name):
                return True
        return False

    def make_arcadia_path(self, nested_dir=None):
        path = self.path if not nested_dir else os.path.join(self.path, nested_dir)

        if path.startswith("arcadia:/arc/trunk/arcadia"):
            return sandboxsdk.svn.Arcadia.replace(path, revision=self.revision)
        else:
            return sandboxsdk.svn.Arcadia.replace(
                os.path.join("arcadia:/arc/trunk/arcadia", path),
                revision=self.revision
            )

    def checkout_to_local_path(self, nested_dir=None):
        arcadia_configs_path = self.make_arcadia_path(nested_dir)
        logging.info("checkout arcadia path {}".format(arcadia_configs_path))
        local_path = sandboxsdk.svn.Arcadia.get_arcadia_src_dir(arcadia_configs_path)

        if local_path is None:
            raise Exception("failed to checkout {} on revision {}".format(self.path, self.revision))

        logging.info("local configs path {}".format(local_path))
        return local_path

    def _list_local_files(self, local_path):
        paths = []
        for item in os.listdir(local_path):
            item_path = os.path.join(local_path, item)
            if os.path.isfile(item_path):
                if self.is_file_matches(item_path):
                    paths.append(item_path)
            elif os.path.isdir(item_path):
                paths.extend(self._list_local_files(item_path))
        return paths

    def fetch_files(self):
        local_path = self.checkout_to_local_path()
        return local_path, self._list_local_files(local_path)

    def list_files(self):
        files = []
        for file_name in sandboxsdk.svn.Arcadia.list(self.make_arcadia_path(), depth="infinity", as_list=True):
            if self.is_file_matches(file_name):
                files.append(file_name)
        return files

############################################################################

def run_release_files_to_yt_task(yt_cluster, yt_token_vault_owner, yt_token_vault_name, deploy_config_filename, revision, message, parent_task=None):
    release_parameters = {
        'yt_cluster': yt_cluster,
        'yt_token_vault_owner': yt_token_vault_owner,
        'yt_token_vault_name': yt_token_vault_name,
        'deploy_config_path': 'arcadia:/arc/trunk/arcadia/logfeller/deploy/yt/{}'.format(deploy_config_filename),
        'clean_up': False,
        'revision': revision,
        'message': message,
        'force': False,
        'run_tests': False
    }

    task_class = sdk2.Task['RELEASE_FILES_TO_YT']
    release_task = task_class(
        parent_task,
        description='Release Logfeller configs to YT. {}'.format(message),
        inherit_notifications=True,
        **release_parameters
    )

    release_task.enqueue()
    logging.info('Enqueue release task with id {}'.format(release_task.id))
    return release_task.id

############################################################################
