import json
import logging
import os
import re
import textwrap
import time

import typing as tp  # noqa

import six

import sandbox.sandboxsdk.svn as sdk_svn
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.projects.common.vcs import arc
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.resource_types import TASK_LOGS
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.yav import Secret


DEFAULT_RESOURCE_SEARCH = {
    "type": "ARCADIA_PROJECT",
    "state": "READY",
    "attrs": {
        "released": "stable",
    },
}
TOKEN_PATTERN = re.compile(r"^\$\((?P<storage>vault|yav):(?P<owner_ver>[^:]+):(?P<name>[^:]+)\)$")


def raise_if(condition, msg):
    if condition:
        raise ValueError(msg)


def load_resource_conf(conf):
    try:
        return json.loads(conf)
    except Exception as exc:
        raise ValueError(
            "Error while loading resource config: {}, {}"
            .format(conf, exc),
        )


def get_proxy_log_link(resouce_id, log_name):
    # type: (int, str) -> str
    url = "https://proxy.sandbox.yandex-team.ru/{id}/{log}?force_text_mode=1".format(id=resouce_id, log=log_name)
    return '<a href="{url}" target="_blank">{log}</a> | <a href="{url}&tail=1" style="color:forestgreen;" target="_blank">tail</a>'.format(url=url, log=log_name)


def get_resource_link(resouce_id):
    # type: (int) -> str
    return '<a href="/resource/{id}/view" target="_blank">#{id}</a>'.format(id=resouce_id)


class RunScript2(sdk2.Task):
    """
    Run arbitrary script with specified resource as argument, sdk2
    """

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 4096
        client_tags = (Tag.MULTISLOT | Tag.GENERIC) & Tag.Group.LINUX & ~Tag.INTEL_E5645  # NOTE: exclude hosts without AVX support
        container_resource = 2796964077

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Run script"

        cmdline = sdk2.parameters.String(
            "Command",
            description="Script cmdline. Use {resource} for resource file name, or {resource_name} for named resources",
            required=True,
            multiline=True,
        )

        use_arc_fuse = sdk2.parameters.Bool("Use Arc VCS", description="Choose to use arc fuse api", default=True)
        with sdk2.parameters.Group("Arc params") as arc_params_group:
            with use_arc_fuse.value[True]:
                arc_revision = sdk2.parameters.String(
                    "Arcadia branch / commit hash / revision",
                    description="Ex: r7312366, users/dim-gonch/logos, trunk, fd6aa52239c",
                    default="trunk",
                )
                arc_token = sdk2.parameters.YavSecret("Arc token", description="Arc OAuth token, if empty get from vault ARC_TOKEN")
            with use_arc_fuse.value[False]:
                svn_revision = sdk2.parameters.String("Svn revision", description="Arcadia trunk revision")

        with sdk2.parameters.Group("Resources params") as resources_params_group:
            with sdk2.parameters.RadioGroup("Resource select", per_line=2, description="Use static resource or search by attributes, use as {resource}") as resource_select:
                resource_select.values["id"] = resource_select.Value(value="Static", default=True)
                resource_select.values["search"] = resource_select.Value(value="Search")

            with resource_select.value["id"]:
                resource = sdk2.parameters.Resource(
                    "Resource",
                    description="Resource",
                    required=False,
                )
            with resource_select.value["search"]:
                resource_search_conf = sdk2.parameters.JSON(
                    "Resource search config",
                    description="Search resource by attributes",
                    default=DEFAULT_RESOURCE_SEARCH,
                    required=False,
                )

            sb_resources = sdk2.parameters.Dict(
                "Sandbox Resources",
                description="Mapping resource_name => resource_id | search config: {!r}".format(json.dumps(DEFAULT_RESOURCE_SEARCH)),
            )

            arc_resources = sdk2.parameters.Dict(
                "Arc resources",
                description="Mapping resource_name => arcadia root relative path",
            )

        env_vars = sdk2.parameters.Dict(
            "Environment variables", description=textwrap.dedent("""
            Environment variables

            May be used with Vault or Yav:
                - $(vault:owner:name)
                - $(yav:version:name)

            Example:
                YT_TOKEN=$(vault:yazevnul:yazevnul-yt_token)
                NIRVANA_TOKEN=$(yav:sec-01daxhrvak3kbcd6fxhydp3esp:nirvana-secret)
            """),
        )

        with sdk2.parameters.Group("Run params") as run_params_group:
            retry_on_failure = sdk2.parameters.Bool(
                "Retry on failure",
                description="Retry task in case of task execution error",
                default=False,
            )

            with retry_on_failure.value[True]:
                retries = sdk2.parameters.Integer(
                    "Retries", description="Retry N times in case of task execution error",
                    default=1, required=True,
                )
                retry_interval = sdk2.parameters.Integer(
                    "Retry Interval", description="Interval between retries (secs)",
                    default=3600, required=True,
                )

        with sdk2.parameters.Group("Output params") as output_params_group:
            save_as_resource = sdk2.parameters.Dict(
                "Save as resource",
                description='Mapping local_path_to_result => RESOURCE_TYPE | {"type": "RESOURCE_TYPE", "attrs": {"name": "value", "ttl": 1}. Save these results as resources',
            )

            resources_ttl = sdk2.parameters.String(
                "Resources TTL",
                description="Saved resources TTL (days). Used for resources without an explicitly defined ttl",
                default="4",
            )
            task_logs_ttl = sdk2.parameters.String(
                "Task Logs TTL",
                description="Set TASK_LOGS resource TTL (days).",
                default="14"
            )

    def get_resource(self):
        if self.Parameters.resource_select == "id":
            resource = self.Parameters.resource
            if not resource:
                raise ValueError("Resource not provided")
        elif self.Parameters.resource_search_conf:
            resource = self.find_resource(self.Parameters.resource_search_conf)
            if resource is None:
                raise ValueError("Resource not found: {!r}".format(self.Parameters.resource_search_conf))
        else:
            raise ValueError("Resource Search config is empty")
        return sdk2.ResourceData(resource).path, resource

    @staticmethod
    def find_resource(search_conf):
        if isinstance(search_conf, six.string_types):
            search_conf_params = load_resource_conf(search_conf)
        else:
            search_conf_params = search_conf

        if not isinstance(search_conf_params, list):
            params_list = [search_conf_params]
        else:
            params_list = search_conf_params

        resources = []
        for params in params_list:
            logging.info("find_resource with params: {!r}".format(params))
            res = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()
            if res:
                resources.append(res)
        resources.sort(key=lambda r: r.id, reverse=True)
        eh.ensure(resources, "No resources found by {}".format(params_list))
        logging.info("Resource found: {}".format(resources[0]))
        return resources[0]

    @staticmethod
    def _checkout_svn(filepath, revision):
        logging.debug("Checkout SVN: {}".format(filepath))
        for i in range(10):
            try:
                svn_dir = sdk_svn.Arcadia.checkout(
                    sdk_svn.Arcadia.trunk_url(os.path.dirname(filepath), revision=revision),
                    "svn_dir",
                )
                break
            except Exception:
                logging.warning("Checkout failed: {}. Try: [{}/10]".format(filepath, i + 1))
                if i == 9:
                    raise
                time.sleep(5)
        return os.path.join(svn_dir, os.path.basename(filepath))

    @staticmethod
    def _mount_arc(arc_token, mount_point, changeset, fetch_all=False):
        arc_cli = arc.Arc(arc_oauth_token=arc_token)
        arcadia_src_dir = os.path.abspath(mount_point)
        for i in range(10):
            try:
                mp = arc_cli.mount_path(None, mount_point=mount_point, changeset=changeset, fetch_all=fetch_all)
                if mp.mounted:
                    logging.debug("Arc Mounted: {}".format(arcadia_src_dir))
                    break
                raise ValueError("not mounted")
            except Exception as e:
                logging.warning("Arc mount failed. Error: {} | Try: [{}/10]".format(e, i + 1))
                if i == 9:
                    raise TaskFailure("Can't mount arcadia: {}, rev={}".format(arcadia_src_dir, changeset))
                time.sleep(5)
        return arcadia_src_dir, mp

    def get_arc_resources(self, resource_map):
        arc_resources = {}
        mp = None
        if not self.Parameters.use_arc_fuse:  # use SVN
            for name, svn_file in six.iteritems(resource_map):
                arc_resources[name] = self._checkout_svn(svn_file, revision=self.Parameters.svn_revision)
        else:
            arc_token = None
            if self.Parameters.arc_token:
                default_key = self.Parameters.arc_token.default_key or "arc_token"
                yav_data = self.Parameters.arc_token.data()
                eh.ensure(default_key in yav_data, "No such key `{}` found in `arc_token`".format(default_key))
                arc_token = yav_data[default_key]
            arcadia_src_dir, mp = self._mount_arc(arc_token=arc_token, mount_point="arc_dir", changeset=self.Parameters.arc_revision)
            for name, arc_file in six.iteritems(resource_map):
                path = os.path.join(arcadia_src_dir, arc_file)
                if not os.path.exists(path):
                    raise ValueError("arc_resources | Path: {} not found".format(path))
                arc_resources[name] = path
        return arc_resources, mp

    def get_sb_resources(self, resource_map):
        sb_resources = {}
        for name, value in six.iteritems(resource_map):
            try:
                resource_id = int(value)
            except (ValueError, TypeError):
                resource = self.find_resource(value)
            else:
                resource = sdk2.Resource.find(id=resource_id).first()
            if resource is None:
                raise ValueError("Resource not found: {!r}".format(value))
            sb_resources[name] = sdk2.ResourceData(resource).path
        return sb_resources

    def _get_token_var(self, value):
        """ Returns {token} if TOKEN_PATTERN matches, otherwise returns value without changes """
        m = TOKEN_PATTERN.match(value.strip())
        if m is None:
            return value
        if m.group("storage") == "vault":
            value = sdk2.Vault.data(m.group("owner_ver"), m.group("name"))
        else:  # yav
            value = Secret(m.group("owner_ver")).data()[m.group("name")]
        return value

    @staticmethod
    def _convert_ttl(ttl):
        # type: (tp.Union[str, int]) -> tp.Union[int, str]
        if not isinstance(ttl, int):
            if ttl == "inf":
                return "inf"
            raise_if(not ttl.isdecimal(), "Resource ttl must contains only digits")
            ttl = int(ttl)
        return max(1, ttl)

    def _change_task_logs_ttl(self, ttl):
        # type: (tp.Union[str, int]) -> tp.Optional[int]
        res = TASK_LOGS.find(task=self).first()
        if res:
            res.ttl = self._convert_ttl(ttl)
            return res.id
        logging.error("Not found TASK_LOGS for task {}".format(self.id))
        return None

    def _create_resource(self, resource_type, resource_path, description="", resource_ttl="14", **attrs):

        resource_ttl = attrs.pop("ttl", resource_ttl)
        resource_ttl = self._convert_ttl(resource_ttl)

        resource = sdk2.Resource[resource_type](
            self,
            description,
            resource_path,
            ttl=resource_ttl,
            **attrs
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        return resource.id

    def _execute(self):
        script_dir = str(self.path("script").absolute())
        logging.info("Script dir: {}".format(script_dir))
        os.makedirs(script_dir)

        context = {}

        arc_mount = None
        if self.Parameters.arc_resources:
            arc_resources, arc_mount = self.get_arc_resources(self.Parameters.arc_resources)
            context.update(arc_resources)

        if self.Parameters.sb_resources:
            sb_resources = self.get_sb_resources(self.Parameters.sb_resources)
            context.update(sb_resources)

        if self.Parameters.resource_select == "id" and self.Parameters.resource or \
                self.Parameters.resource_select == "search" and self.Parameters.resource_search_conf:
            resource_path, resource = self.get_resource()
            self.set_info("Found resource: {} | {}".format(resource.type, get_resource_link(resource.id)), do_escape=False)
            context["resource"] = resource_path

        env = os.environ.copy()

        for key, placeholder in six.iteritems(self.Parameters.env_vars):
            env[key] = self._get_token_var(placeholder)

        with sdk2.helpers.ProcessLog(self, logger="script_run") as pl:

            task_logs_id = self._change_task_logs_ttl(ttl=self.Parameters.task_logs_ttl)
            if task_logs_id is None:
                start_info = "Run script"
            else:
                start_info = "Run script | {}".format(get_proxy_log_link(task_logs_id, "script_run.out.log"))

            self.set_info(start_info, do_escape=False)
            sp.check_call(
                self.Parameters.cmdline.format(**context),
                shell=True,
                stdout=pl.stdout, stderr=sp.STDOUT,
                cwd=script_dir,
                env=env,
            )
            self.set_info("Script completed")

        if self.Parameters.save_as_resource:
            for local_path, resource_info in six.iteritems(self.Parameters.save_as_resource):
                resource_path = os.path.join(script_dir, local_path)
                if os.path.exists(resource_path):
                    resource_info = resource_info.strip()
                    if resource_info.startswith("{"):
                        conf = load_resource_conf(resource_info)
                    else:
                        conf = {"type": resource_info}

                    resource_type = conf.pop("type", None)

                    if resource_type is None:
                        raise ValueError("save_as_resource | Can't fetch resource type: {!r}".format(resource_info))

                    resource_id = self._create_resource(
                        resource_type=resource_type,
                        resource_path=resource_path,
                        description="Ouput resource {} from task {}".format(local_path, self.id),
                        resource_ttl=self.Parameters.resources_ttl,
                        **conf.get("attrs", {})
                    )
                    self.set_info("Created resource: {} -> {} | {}".format(local_path, resource_type, get_resource_link(resource_id)), do_escape=False)

        if arc_mount and arc_mount.mounted:
            arc_mount.unmount()

    def _validate_params(self):
        arc_resources = self.Parameters.arc_resources or {}
        sb_resources = self.Parameters.sb_resources or {}
        if arc_resources and sb_resources:
            for name in arc_resources:
                raise_if(name in sb_resources, "Redefining named resource {}".format(name))
        raise_if(
            "resource" in arc_resources or "resource" in sb_resources,
            "Can't use {resource} in named resource",
        )
        if self.Parameters.retry_on_failure:
            raise_if(self.Parameters.retries < 1, "Retries less than 1: {}".format(self.Parameters.retries))
            raise_if(self.Parameters.retry_interval < 0, "Retry interval is negative: {}".format(self.Parameters.retry_interval))

    def on_execute(self):
        self._validate_params()
        while True:
            try:
                self._execute()
                return
            except Exception as e:
                if self.Parameters.retry_on_failure and self.Parameters.retries > 0:
                    self.Parameters.retries -= 1
                    logging.error("Exception: {}".format(e))
                    tts = self.Parameters.retry_interval
                    logging.info("Retry in {} seconds [remaining tries: {}]".format(tts, self.Parameters.retries))
                    time.sleep(tts)
                else:
                    raise
