# -*- coding: utf-8 -*-

import json
import logging

import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.file_utils as fu
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.sdk2 as sdk2


ICOOKIE_ARTIFACT_DESCRIPTION = "File with pairs (yuid, generated icookie) in JSON format"


class IcookieArtifact(sdk2.Resource):
    """
        JSON with pairs (yuid, generated icookie)
    """
    auto_backup = True


class GenerateICookie(sdk2.Task, binary_task.LastBinaryTaskRelease):
    """
        This task generates IcookieArtifact resource with pairs (yuid, generated icookie) by given yuids.
    """

    class Requirements(task_env.TinyRequirements):
        # RMDEV-1971
        pass

    class Parameters(binary_task.LastBinaryReleaseParameters):
        yandexuids = sdk2.parameters.List(
            "Yandex UIDs to generate ICookies from",
            sdk2.parameters.String,
            default=[],
        )

        with sdk2.parameters.Output():
            icookie_artifact = sdk2.parameters.Resource(
                ICOOKIE_ARTIFACT_DESCRIPTION,
                resource_type=IcookieArtifact,
            )

    def on_enqueue(self):
        with self.memoize_stage.icookie_artifact:
            self.Parameters.icookie_artifact = IcookieArtifact(
                task=self,
                description=ICOOKIE_ARTIFACT_DESCRIPTION,
                path="icookie.json",
            )

    def on_execute(self):
        import yweb.webdaemons.icookiedaemon.icookie_lib.utils_py as icookie

        yuid_icookie_pairs = {}

        for yuid in self.Parameters.yandexuids:
            yuid_icookie_pairs[yuid] = icookie.GetEncryptedIcookie(yuid).decode('utf8')
        logging.debug("Got yuid-icookie pairs %s", yuid_icookie_pairs)

        fu.write_file(
            self.Parameters.icookie_artifact.path,
            json.dumps(yuid_icookie_pairs, indent=4)
        )
        sdk2.ResourceData(self.Parameters.icookie_artifact).ready()
