#!/usr/bin/env python
# -*- coding: utf-8 -*-

__author__ = 'lexplua'
from jinja2 import Template
import os
import json
import copy


def update_flags(src_fields, dst_fields):
    """
    Update dst_fields dictionary from src_fields keys with False values
    :param src_fields: dict
    :param dst_fields: dict
    """
    if isinstance(src_fields, dict):
        dst_fields.update(dict(zip(src_fields.keys(), (False,) * len(src_fields))))


def dictyfy(obj):
    if isinstance(obj, basestring):
        return {
            "diagnosis": obj
        }
    if not obj:
        return {
            "diagnosis": ""
        }

    return obj


def fix_diagnosis(obj):
    _obj = copy.deepcopy(obj)
    _obj["rec1Info"] = dictyfy(_obj.get("rec1Info", ""))
    _obj["rec2Info"] = dictyfy(_obj.get("rec2Info", ""))
    _obj["changeInfo"] = (_obj.get("changeInfo", {}))
    return _obj


def make_diagnosis_class_header(diagnoses):
    support_fields1 = {}
    support_fields2 = {}
    changed_fields = {}
    for diagnosis in diagnoses:
        _support_fields1 = diagnosis.get("rec1Info", {})
        _support_fields2 = diagnosis.get("rec2Info", {})
        _changed_fields = diagnosis.get("changeInfo", {})
        update_flags(_support_fields1, support_fields1)
        update_flags(_support_fields2, support_fields2)
        update_flags(_changed_fields, changed_fields)
        diagnosis['intervals'] = [[int(diagnosis["pos"]), int(diagnosis["pos"]) + int(diagnosis["count"])], ]
    return changed_fields, support_fields1, support_fields2


def process_diagnosis_class(diagnoses):
    fixed_diagnoses = [fix_diagnosis(x) for x in diagnoses]
    changed_fields, support_fields1, support_fields2 = make_diagnosis_class_header(fixed_diagnoses)

    result = {
        'flags': {
            'support_fields1': support_fields1,
            'support_fields2': support_fields2,
            'changed_fields': changed_fields
        },
        'diagnoses': fixed_diagnoses
    }

    return result


def get_context(diff_content, meta_info, js_file_path):
    context = {}
    context['raw_content'] = diff_content
    with open(js_file_path) as js_file:
        context['custom_js'] = js_file.read()
    json_object = json.loads(context['raw_content'])
    precomp_data = {}
    descriptions = {}
    for class_name, diagnoses in json_object.iteritems():
        if '->' in class_name:
            json_class_name = class_name.replace(" -> ", "2")
            descriptions[json_class_name] = process_diagnosis_class(diagnoses)

    precomp_data['descriptions'] = descriptions
    context['precomp_data'] = json.dumps(descriptions, sort_keys=True, separators=(',', ':'), indent=4)
    context.update(meta_info)
    return context


def render_diff_to_html(diff_json_content, meta_info, out_file_name='index.html', js_file_path=None,
                        template_file_path=None):
    """

    :param meta_info_json: dict with meta information
    :param diff_json_content: json with diff for one log
    :param out_file_name: name of result file
    :param js_file_path: path to file with js code
    :param template_file_path: path to file with template
    """
    with open(template_file_path) as f:
        template_content = Template(f.read())
        rendered = template_content.render(**get_context(diff_json_content, meta_info, js_file_path))
        with open(out_file_name, 'w') as out:
            out.write(rendered)
    return out_file_name


def render_main_page(urls, template_path, out_filename='index.html'):
    """

    :param urls: dict logname -> diff url
    :param template_path: path to file with template
    :param out_filename: path to result file
    :return: path to result file
    """
    context = {
        "urls": urls
    }
    with open(template_path) as f:
        template_content = Template(f.read())
        rendered = template_content.render(**context)
        with open(out_filename, 'w') as out:
            out.write(rendered)
    return out_filename


def main():
    root = os.path.abspath(os.path.dirname(__file__))
    diff_file_path = os.path.join(root, 'diff.json')
    js_file_path = os.path.join(root, 'us_diff.js')
    template_file_path = os.path.join(root, 'template.html')

    meta_info = {
        'FetchedDataPath': 'path/to/fetched/data',
        'SessionsTable1': 'home/userdata-sessions-build-ci/2016-09-25/user_sessions/build/logs/reqans-log/1d/create_sessions_test-trunk/testenv_hashed_2480695/hashed',
        'SessionsTable2': 'home/userdata-sessions-build-ci/2016-09-25/user_sessions/build/logs/reqans-log/1d/create_sessions_test-trunk/testenv_hashed_2481226/hashed',
        'DiagHashTable': 'tmp/andreysdkh/test_res2/support',
        'ErrCount1': '10',
        'ErrCount2': '5',
    }

    with open(diff_file_path) as diff_file:
        render_diff_to_html(diff_file.read(), meta_info, "some_log.html", js_file_path, template_file_path)

    main_template_file_path = os.path.join(root, 'index_template.html')
    urls = {
        "Access log": "http://blahblah.yandex.ru/access_diff.html",
        "Redir log": "http://blahblah.yandex.ru/redir_diff.html",
    }
    render_main_page(urls, main_template_file_path)


if __name__ == '__main__':
    main()
