# -*- coding: utf-8 -*-


from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common import utils
import sandbox.sandboxsdk.parameters as sb_params

import itertools

from sandbox.projects.logs import UserSessionsProcessesTest as uspt

TIME_FORMAT = '%Y-%m-%d:%H:%M'
QUEUE_WORKING_LIMIT = 5
DEFAULT_CI_YT_POOL = 'userdata-sessions-build-ci'


DAILY_SESSIONS_PARAMETERS = 'Daily sessions parameters'
ACCESS_PARAMETERS = 'Yt-tokens, yt-pools, yt-paths. All this is really connected, yt-tokens, for example have to have write-rights to yt-paths'


class MRClusters(sb_params.SandboxStringParameter):
    name = 'mr_clusters'
    default_value = 'hahn'
    description = 'Clusters, \';\'-separated'


class ArcadiaUrl(sb_params.SandboxStringParameter):
    name = 'arcadia_url_with_revision'
    required = True
    description = ('Default svn url for arcadia (should be with revision). It is used for checkouting rem_processes and other components. '
                   'By default it is also used for building binaries, but you can specify other bins-settings in the following params')


class CustomBinariesArcadiaUrl(sb_params.SandboxStringParameter):
    name = 'custom_binaries_arcadia_url'
    required = True
    description = 'Custom svn url for binaries (should be with revision)'


class BinariesFetchMethod(sb_params.SandboxStringParameter):
    name = 'how_to_fetch_binaries'
    description = 'How to fetch binaries. Default method. Will be used for binaries which don\'t have specified custom fetching method'
    choices = [(p, p) for p in ['default_arcadia_url', 'custom_binaries_arcadia_url', 'rsync']]
    default = 'default_arcadia_url'
    sub_fields = {'custom_binaries_arcadia_url': [CustomBinariesArcadiaUrl.name]}


class CustomizeBinaries(sb_params.SandboxBoolParameter):
    name = 'customize_binaries'
    default = False
    description = 'Customize some binaries. If you want to use special fetching methods for some binaries'
    sub_fields = {'true': [binary_input_params[0].name for binary_input_params in uspt.BINS_CUSTOM_INPUT_PARAMS]}


class ArcadiaPatch(sp.SandboxStringParameter):
    name = 'arcadia_patch'
    description = 'Apply arcadia patch to working copy of arcadia. Important for testing from RB.'
    default_value = ''


class CheckProgressTries(sp.SandboxIntegerParameter):
    name = 'check_progress_tries'
    description = 'Tries to check progress (number)'
    default_value = 60


class NextCheckProgressAfter(sp.SandboxIntegerParameter):
    name = 'next_check_progress_after'
    description = 'Next check progress after (sec.)'
    default_value = 180  # 180s


class MonitorErrorStats(sp.SandboxBoolParameter):
    name = 'monitor_error_stats'
    description = 'Should we check and wait for error_stats_packet (depends from publish_packet).'
    default_value = True


class CustomYtPool(sp.SandboxStringParameter):
    name = 'custom_yt_pool'
    description = 'Optional! Custom yt pool. If empty, takes default one: {}. Ignored if ManualRecalcMode is turned on'.format(DEFAULT_CI_YT_POOL)
    default_value = DEFAULT_CI_YT_POOL
    group = ACCESS_PARAMETERS


class SubstituteYtPoolsWithCutomOne(sp.SandboxBoolParameter):
    name = 'substitute_yt_pools_with_custom_one'
    description = 'Should we substitute production yt pools from rem_processes\' configs with a custom one?'
    default_value = True
    sub_fields = {'true': [CustomYtPool.name]}
    group = ACCESS_PARAMETERS


class CustomYTToken(sp.SandboxStringParameter):
    name = 'custom-yt-token'
    description = 'Name of secret with YT token (from sb-vault)'
    group = ACCESS_PARAMETERS


class CustomYTTokenOwner(sp.SandboxStringParameter):
    name = 'yt_token_owner'
    description = 'Owner of secret with YT token (default: owner of task)'
    group = ACCESS_PARAMETERS


class YtToken(sp.SandboxStringParameter):
    name = 'yt-token'
    description = 'Токен доступа.'
    group = ACCESS_PARAMETERS
    choices = [(p, p) for p in ['CI-token', 'recalc-token', 'custom']]
    default = 'CI-token'
    sub_fields = {'custom': [CustomYTToken.name, CustomYTTokenOwner.name]}


NON_MANUAL_RECALC_MODE_SUB_FIELDS = (
    YtToken,
    SubstituteYtPoolsWithCutomOne,
)


class ManualRecalcMode(sp.SandboxBoolParameter):
    name = 'manual_recalc_mode'
    description = ('CAREFUL WITH THIS. Turns on manual-recalc-mode. If True: uses robot-make-sessions\'s token; doesn\'t substitute prod-yt-pools '
                   'with testing one; launches rem_processes with option --manual-recalc (to use special recalc-prod-yt-pools).')
    default_value = False
    group = ACCESS_PARAMETERS
    sub_fields = {'false': [param.name for param in NON_MANUAL_RECALC_MODE_SUB_FIELDS]}


class CustomWholePrefixPath(sp.SandboxStringParameter):
    name = 'custom_whole_prefix_path'
    description = ('Optional! Custom whole prefix-path. It has to be created in advance. And you also have to configure it by yourself (it'
                   ' has to contain "user_sessions/build/tmp", maybe "home" and links or folders to "logs", "cooked_logs" (depends on your task)')
    default_value = ''
    group = ACCESS_PARAMETERS


class DailyDatetimes(sp.SandboxStringParameter):
    name = 'daily_datetimes'
    description = 'Daily datetimes in format {}, \';\'-separated.'.format(TIME_FORMAT)
    required = True
    group = DAILY_SESSIONS_PARAMETERS


class ExtraLabelsForDaily(sp.SandboxStringParameter):
    name = 'extra_labels_for_daily'
    description = ('Optional! Extra labels for daily sessions, \',\'-separated. If you want the task to launch create_sessions.py several times'
                   ' with different sets of labels, specify here these sets \';\'-separated, every set \',\'-separated')
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class DailySessionsToCheck(sp.SandboxStringParameter):
    name = 'daily_sessions_to_check'
    description = 'Daily sessions to check, \',\'-separated. It is not required. If empty, checks ALL sessions. Recommended if extra labels are used.'
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class DailyQueueWorkingLimit(sp.SandboxIntegerParameter):
    name = 'daily_queue_working_limit'
    description = 'Daily queue working limit'
    default_value = QUEUE_WORKING_LIMIT
    group = DAILY_SESSIONS_PARAMETERS


class DailyScarabUserSessionsSourceLogsToSkip(sp.SandboxStringParameter):
    name = 'daily_scarab_user_sessions_logs_to_skip'
    description = 'Daily scarab logs, which will be ignored during building daily-search user_sessions, \',\'-separated. Use this as last resort'
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


DAILY_SUB_FIELDS = (
    DailyDatetimes,
    ExtraLabelsForDaily,
    DailySessionsToCheck,
    DailyQueueWorkingLimit,
    DailyScarabUserSessionsSourceLogsToSkip,
    uspt.LateRecalcMode,
)


class UserSessionsProcessesRecalc(SandboxTask):
    ''' Task for recalc of user_sessions'''

    type = 'USER_SESSIONS_PROCESSES_RECALC'

    input_parameters = (
        (
            MRClusters,
            ArcadiaUrl,
            BinariesFetchMethod,
            CustomBinariesArcadiaUrl,
            CustomizeBinaries,
        ) +
        tuple(itertools.chain.from_iterable(uspt.BINS_CUSTOM_INPUT_PARAMS)) +
        (
            ArcadiaPatch,
            CheckProgressTries,
            NextCheckProgressAfter,
            MonitorErrorStats,
            ManualRecalcMode,
            YtToken,
            CustomYTToken,
            CustomYTTokenOwner,
            SubstituteYtPoolsWithCutomOne,
            CustomYtPool,
            CustomWholePrefixPath,
            uspt.KeepBuildTables,
        ) +
        DAILY_SUB_FIELDS
    )

    def CreateRecalcSubtask(self, params, cluster, date_str, sessions_type):
        params[uspt.MRCluster.name] = cluster
        params[uspt.DailyDatetime.name] = date_str
        self.priority = self.Priority(self.Priority.Class.USER, self.Priority.Subclass.NORMAL)  # XXX hack to increase children priority
        subtask = SandboxTask.create_subtask(
            self,
            task_type='USER_SESSIONS_PROCESSES_TEST',
            description='Building {} sessions on {} for date {}'.format(sessions_type, cluster, date_str),
            input_parameters=params,
        )
        return subtask

    def on_execute(self):
        if 'subtasks_ids' not in self.ctx:
            self.ctx['subtasks_ids'] = []
            extra_labels_daily = self.ctx[ExtraLabelsForDaily.name]

            params = self.ctx.copy()
            params['use_obj_cache'] = True
            params[uspt.LaunchDaily.name] = True
            params[uspt.LaunchFast.name] = False
            params['kill_timeout'] = ((self.ctx[CheckProgressTries.name] * self.ctx[NextCheckProgressAfter.name] * 1.15 / 3600) + 1) * 3600
            params[ExtraLabelsForDaily.name] = extra_labels_daily
            params['exception_on_build_fail'] = False

            clusters = self.ctx[MRClusters.name].strip(';').split(';')
            date_strs = sorted(self.ctx[DailyDatetimes.name].strip(';').split(';'))
            all_labels = [l for b in uspt.extract_labels(extra_labels_daily) for l in b]
            has_late = 'late' in all_labels and not self.ctx[uspt.LateRecalcMode.name]

            for cluster in clusters:
                for date_idx, date_str in enumerate(date_strs):
                    if has_late:  # LOGSTAT-6549
                        params = params.copy()
                        params[uspt.LateRecalcMode.name] = 'all_backlog' if date_idx == 0 else 'current_only'

                    subtask = self.CreateRecalcSubtask(params, cluster, date_str, sessions_type='main')
                    self.ctx['subtasks_ids'].append(subtask.id)

            SandboxTask.wait_all_tasks_completed(self, self.ctx['subtasks_ids'])
        else:
            utils.check_subtasks_fails(self.ctx['subtasks_ids'])


__Task__ = UserSessionsProcessesRecalc
