# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects import resource_types as rst
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string as su
from sandbox.projects.common import utils
from sandbox.sandboxsdk import environments
from sandbox.projects.common import apihelpers
import sandbox.projects.common.rem as common_rem
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.utils import set_resource_attributes
import sandbox.sandboxsdk.parameters as sb_params

import datetime
import itertools
import logging
import time
import os
from os.path import join as pj
import sys
import random
import string
import traceback

from sandbox.projects.logs.common import GetRevision, MRPathJoin
from sandbox.projects.logs.common.binaries import get_binary_from_svn_path, REM_BINARIES

from sandbox.projects.logs import resources
from sandbox.projects.geobase.Geodata6BinXurmaStable import resource as geo_resource

def extract_labels(all_labels_string):
    result = []

    labels_bunches = all_labels_string.replace(' ', '').strip(';').strip(',').split(';')
    for labels_string in labels_bunches:
        labels = labels_string.strip(',').split(',')
        if labels:
            result.append(filter(lambda l: len(l), labels))

    return result


def create_custom_input_params_classes(binary_name):
    class CustomBinaryArcadiaUrl(sb_params.SandboxStringParameter):
        name = 'arcadia_url_for_binary_{}'.format(binary_name)
        required = True
        description = 'Custom svn url for binary {} (should be with revision)'.format(binary_name)

    class BinaryFetchMethod(sb_params.SandboxStringParameter):
        name = 'how_to_fetch_binary_{}'.format(binary_name)
        description = 'How to fetch binary {}'.format(binary_name)
        hidden = None
        choices = [(p, p) for p in ['use_default_binaries_fetching_method', 'custom_binary_arcadia_url', 'rsync', 'dont_fetch_this_binary']]
        default = 'use_default_binaries_fetching_method'
        sub_fields = {'custom_binary_arcadia_url': [CustomBinaryArcadiaUrl.name]}

    return [BinaryFetchMethod, CustomBinaryArcadiaUrl]


BINS_CUSTOM_INPUT_PARAMS = [create_custom_input_params_classes(binary_name) for binary_name in sorted(REM_BINARIES.keys())]

PRODUCTION_SERVER = 'veles02.search.yandex.net'
TIME_FORMAT = '%Y-%m-%d:%H:%M'
QUEUE_WORKING_LIMIT = 5
DEFAULT_DAYS_TO_STORE_OUTPUT = 3
DEFAULT_PACKET_LABELS = ['!calcerrors', '!archive', '!history', '!send_to_statface', '!send_to_reactor', '!scarab_razladki']
DISABLE_ANTIFRAUD_RULES_PACKET_LABEL = '!antifraud_rules'
DEFAULT_CI_YT_POOL = 'userdata-sessions-build-ci'


BIN_NAME_TO_TASK_ID_AND_SVN_URL = 'bin_name_to_task_id_and_svn_url'
BIN_NAME_TO_READY_RESOURCE_ID = 'bin_name_to_ready_resource_id'
BUILD_SVN_PATH_ATTR_NAME = 'build_from_svn_url'
ATTR_VALUE_FOR_NEW_BINS = 'build_from_svn_url_attr_value'

FILENAME = 'filename'
RESOURCE_TYPE = 'resource_type'
WORKING_FOLDER = 'FOLDER_WITH_SESSIONS'

FAST_SESSIONS_PARAMETERS = 'Fast sessions parameters'
DAILY_SESSIONS_PARAMETERS = 'Daily sessions parameters'
ACCESS_PARAMETERS = 'Yt-tokens, yt-pools, yt-paths. All this is really connected, yt-tokens, for example have to have write-rights to yt-paths'
ANTIFRAUD_PARAMETERS = 'Antifraud parameters'


class MRCluster(sb_params.SandboxStringParameter):
    name = 'mr_cluster'
    default_value = 'hahn'
    description = 'Cluster: hahn or banach'


class ArcadiaUrl(sb_params.SandboxStringParameter):
    name = 'arcadia_url_with_revision'
    required = True
    description = ('Default svn url for arcadia (should be with revision). It is used for checkouting rem_processes and other components. '
                   'By default it is also used for building binaries, but you can specify other bins-settings in the following params')


class CustomBinariesArcadiaUrl(sb_params.SandboxStringParameter):
    name = 'custom_binaries_arcadia_url'
    required = True
    description = 'Custom svn url for binaries (should be with revision)'


class BinariesFetchMethod(sb_params.SandboxStringParameter):
    name = 'how_to_fetch_binaries'
    description = 'How to fetch binaries. Default method. Will be used for binaries which don\'t have specified custom fetching method'
    choices = [(p, p) for p in ['default_arcadia_url', 'custom_binaries_arcadia_url', 'rsync']]
    default = 'default_arcadia_url'
    sub_fields = {'custom_binaries_arcadia_url': [CustomBinariesArcadiaUrl.name]}


class CustomizeBinaries(sb_params.SandboxBoolParameter):
    name = 'customize_binaries'
    default = False
    description = 'Customize some binaries. If you want to use special fetching methods for some binaries'
    sub_fields = {'true': [binary_input_params[0].name for binary_input_params in BINS_CUSTOM_INPUT_PARAMS]}


class ArcadiaPatch(sp.SandboxStringParameter):
    name = 'arcadia_patch'
    description = 'Apply arcadia patch to working copy of arcadia. Important for testing from RB.'
    default_value = ''


class CheckProgressTries(sp.SandboxIntegerParameter):
    name = 'check_progress_tries'
    description = 'Tries to check progress (number)'
    default_value = 60


class NextCheckProgressAfter(sp.SandboxIntegerParameter):
    name = 'next_check_progress_after'
    description = 'Next check progress after (sec.)'
    default_value = 180  # 180s


class MonitorErrorStats(sp.SandboxBoolParameter):
    name = 'monitor_error_stats'
    description = 'Should we check and wait for error_stats_packet (depends from publish_packet).'
    default_value = True


class CustomYtPool(sp.SandboxStringParameter):
    name = 'custom_yt_pool'
    description = 'Optional! Custom yt pool. If empty, takes default one: {}'.format(DEFAULT_CI_YT_POOL)
    default_value = DEFAULT_CI_YT_POOL
    group = ACCESS_PARAMETERS


class SubstituteYtPoolsWithCutomOne(sp.SandboxBoolParameter):
    name = 'substitute_yt_pools_with_custom_one'
    description = 'Should we substitute production yt pools from rem_processes\' configs with a custom one?'
    default_value = True
    sub_fields = {'true': [CustomYtPool.name]}
    group = ACCESS_PARAMETERS


class CustomYTToken(sp.SandboxStringParameter):
    name = 'custom-yt-token'
    description = 'Name of secret with YT token (from sb-vault)'
    group = ACCESS_PARAMETERS


class CustomYTTokenOwner(sp.SandboxStringParameter):
    name = 'yt_token_owner'
    description = 'Owner of secret with YT token (default: owner of task)'
    group = ACCESS_PARAMETERS


class YtToken(sp.SandboxStringParameter):
    name = 'yt-token'
    description = 'Токен доступа.'
    group = ACCESS_PARAMETERS
    choices = [(p, p) for p in ['CI-token', 'recalc-token', 'custom']]
    default = 'CI-token'
    sub_fields = {'custom': [CustomYTToken.name, CustomYTTokenOwner.name]}


NON_MANUAL_RECALC_MODE_SUB_FIELDS = (
    YtToken,
    SubstituteYtPoolsWithCutomOne,
)


class ManualRecalcMode(sp.SandboxBoolParameter):
    name = 'manual_recalc_mode'
    description = ('CAREFUL WITH THIS. Turns on manual-recalc-mode. If True: uses robot-make-sessions\'s token; doesn\'t substitute prod-yt-pools '
                   'with testing one; launches rem_processes with option --manual-recalc (to use special recalc-prod-yt-pools).')
    default_value = False
    group = ACCESS_PARAMETERS
    sub_fields = {'false': [param.name for param in NON_MANUAL_RECALC_MODE_SUB_FIELDS]}


class AntifraudYtToken(sp.SandboxStringParameter):
    name = 'antifraud_yt_token'
    description = 'Токен доступа для antifraud. Если пусто, то во время исполнения берётся стандартный токен'
    default_value = ''
    group = ANTIFRAUD_PARAMETERS


class ApplyAntifraudLogic(sp.SandboxBoolParameter):
    name = 'apply_antifraud_logic'
    description = 'Применять ли логику antifraud. Если False, то скиппает её - заместо нарезки просто копирует сырую сессию'
    default_value = False
    group = ANTIFRAUD_PARAMETERS
    sub_fields = {'true': [AntifraudYtToken.name]}


class FetchPath(sp.SandboxStringParameter):
    name = 'fetch_path'
    description = ('Optional! Custom working directory path (without prefix "//"). It has to contain folder "raw_logs" with folders (or links to folders) "home", '
                   '"logs" and "cooked_logs". This list may expand if logs of new types will appear. If empty, takes from the freshest resource '
                   'USER_SESSIONS_PROCESSES_FRESHDATA_INFO. This field is ignored if "Custom whole prefix-path" is specified')
    default_value = ''
    group = ACCESS_PARAMETERS


class CustomWholePrefixPath(sp.SandboxStringParameter):
    name = 'custom_whole_prefix_path'
    description = ('Optional! Custom whole prefix-path. It has to be created in advance. And you also have to configure it by yourself (it'
                   ' has to contain "user_sessions/build/tmp", maybe "home" and links or folders to "logs", "cooked_logs" (depends on your task)')
    default_value = ''
    group = ACCESS_PARAMETERS


class DaysToStoreOutput(sp.SandboxIntegerParameter):
    name = 'days_to_store_output'
    description = 'Days to store output'
    default_value = DEFAULT_DAYS_TO_STORE_OUTPUT
    group = ACCESS_PARAMETERS


class SetOutputRemovalAttrs(sp.SandboxBoolParameter):
    name = 'set_output_removal_attrs'
    description = 'Whether we should set removal attrs for output folder. If true, it will be deleted in N days (without rounding)'
    default_value = False
    sub_fields = {'true': [DaysToStoreOutput.name]}
    group = ACCESS_PARAMETERS


class KeepBuildTables(sp.SandboxBoolParameter):
    name = 'keep_build_tables'
    description = 'Keep intermediate user_sessions/build tables after task execution'
    default_value = True
    group = ACCESS_PARAMETERS


class FastDatetime(sp.SandboxStringParameter):
    name = 'fast_datetime'
    description = 'Optional! Fast datetime in format {}. If empty, takes datetime from the freshest USER_SESSIONS_PROCESSES_FRESHDATA_INFO resource'.format(TIME_FORMAT)
    default_value = ''
    group = FAST_SESSIONS_PARAMETERS


class UseDefaultPacketLabelsForFast(sp.SandboxBoolParameter):
    name = 'use_default_packet_labels_for_fast'
    description = ('Whether we should use default packet labels {} and {} (when antifraud is disabled) for fast packets. It is highly recommended to use '
                   'them!'.format(','.join(DEFAULT_PACKET_LABELS), DISABLE_ANTIFRAUD_RULES_PACKET_LABEL))
    default_value = True
    group = FAST_SESSIONS_PARAMETERS


class ExtraLabelsForFast(sp.SandboxStringParameter):
    name = 'extra_labels_for_fast'
    description = 'Optional! Extra labels for fast sessions, \',\'-separated'
    description = ('Optional! Extra labels for fast sessions, \',\'-separated. If you want the task to launch create_sessions.py several times'
                   ' with different sets of labels, specify here these sets \';\'-separated, labels in every set \',\'-separated')
    default_value = ''
    group = FAST_SESSIONS_PARAMETERS


class FastQueueWorkingLimit(sp.SandboxIntegerParameter):
    name = 'fast_queue_working_limit'
    description = 'Fast queue working limit'
    default_value = QUEUE_WORKING_LIMIT
    group = FAST_SESSIONS_PARAMETERS


class FastSessionsToCheck(sp.SandboxStringParameter):
    name = 'fast_sessions_to_check'
    description = 'Fast sessions to check, \',\'-separated. It is not required. If empty, checks ALL sessions. Recommended if extra labels are used.'
    default_value = ''
    group = FAST_SESSIONS_PARAMETERS


FAST_SUB_FIELDS = (
    FastDatetime,
    UseDefaultPacketLabelsForFast,
    ExtraLabelsForFast,
    FastSessionsToCheck,
    FastQueueWorkingLimit,
)


class LaunchFast(sp.SandboxBoolParameter):
    name = 'launch_fast'
    description = 'Launch building fast sessions'
    default_value = True
    sub_fields = {'true': [param.name for param in FAST_SUB_FIELDS]}
    group = FAST_SESSIONS_PARAMETERS


class DailyDatetime(sp.SandboxStringParameter):
    name = 'daily_datetime'
    description = 'Optional! Daily datetime in format {}. If empty, takes datetime from the freshest USER_SESSIONS_PROCESSES_FRESHDATA_INFO resource'.format(TIME_FORMAT)
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class UseDefaultPacketLabelsForDaily(sp.SandboxBoolParameter):
    name = 'use_default_packet_labels_for_daily'
    description = ('Whether we should use default packet labels {} and {} (when antifraud is disabled) for daily packets. It is highly recommended to use '
                   'them!'.format(','.join(DEFAULT_PACKET_LABELS), DISABLE_ANTIFRAUD_RULES_PACKET_LABEL))
    default_value = True
    group = DAILY_SESSIONS_PARAMETERS


class ExtraLabelsForDaily(sp.SandboxStringParameter):
    name = 'extra_labels_for_daily'
    description = ('Optional! Extra labels for daily sessions, \',\'-separated. If you want the task to launch create_sessions.py several times'
                   ' with different sets of labels, specify here these sets \';\'-separated, every set \',\'-separated')
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class DailySessionsToCheck(sp.SandboxStringParameter):
    name = 'daily_sessions_to_check'
    description = 'Daily sessions to check, \',\'-separated. It is not required. If empty, checks ALL sessions. Recommended if extra labels are used.'
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class DailyQueueWorkingLimit(sp.SandboxIntegerParameter):
    name = 'daily_queue_working_limit'
    description = 'Daily queue working limit'
    default_value = QUEUE_WORKING_LIMIT
    group = DAILY_SESSIONS_PARAMETERS


class DailyScarabUserSessionsSourceLogsToSkip(sp.SandboxStringParameter):
    name = 'daily_scarab_user_sessions_logs_to_skip'
    description = 'Daily scarab logs, which will be ignored during building daily-search user_sessions, \',\'-separated. Use this as last resort'
    default_value = ''
    group = DAILY_SESSIONS_PARAMETERS


class LateRecalcMode(sp.SandboxStringParameter):
    name = 'late_recalc_mode'
    description = 'How to recalc late sessions'
    choices = [(x, x) for x in ('', 'all_backlog', 'current_only')]
    default_value = None
    group = DAILY_SESSIONS_PARAMETERS


DAILY_SUB_FIELDS = (
    DailyDatetime,
    UseDefaultPacketLabelsForDaily,
    ExtraLabelsForDaily,
    DailySessionsToCheck,
    DailyQueueWorkingLimit,
    DailyScarabUserSessionsSourceLogsToSkip,
    LateRecalcMode,
)


class LaunchDaily(sp.SandboxBoolParameter):
    name = 'launch_daily'
    description = 'Launch building daily sessions.'
    default_value = True
    sub_fields = {'true': [param.name for param in DAILY_SUB_FIELDS]}
    group = DAILY_SESSIONS_PARAMETERS


def RunProcess(cmd, env, log_prefix=None, exception_if_nonzero_code=True):
    cmd_str = ' '.join(cmd)
    process = run_process(
        cmd_str,
        outs_to_pipe=True, check=False, shell=True, wait=True,
        environment=env,
        log_prefix=log_prefix,
    )
    result, error = process.communicate()
    if exception_if_nonzero_code and process.returncode != 0:
        raise Exception(error)
    return result, error


def GetYtFullPath(yt_path):
    yt_prefix_appendable_name = yt_path.lstrip('/')
    return os.environ.get('YT_PREFIX', '//') + yt_prefix_appendable_name


class UserSessionsProcessesTest(SandboxTask):
    ''' Task for auto-testing rem-scripts of logs-team '''

    type = 'USER_SESSIONS_PROCESSES_TEST'
    environment = [
        environments.PipEnvironment('yandex-yt', version='0.9.23'),  # LOGSTAT-6658 ssh veles02.search.yandex.net 'pip list 2>/dev/null | grep "yandex-yt "'
        environments.PipEnvironment('future'),
    ]

    cores = 2
    required_ram = 32 * 1024

    input_parameters = (
        (
            MRCluster,
            ArcadiaUrl,
            BinariesFetchMethod,
            CustomBinariesArcadiaUrl,
            CustomizeBinaries,
        ) +
        tuple(itertools.chain.from_iterable(BINS_CUSTOM_INPUT_PARAMS)) +
        (
            ArcadiaPatch,
            CheckProgressTries,
            NextCheckProgressAfter,
            MonitorErrorStats,
            ManualRecalcMode,
            YtToken,
            CustomYTToken,
            CustomYTTokenOwner,
            SubstituteYtPoolsWithCutomOne,
            CustomYtPool,
            # ApplyAntifraudLogic,
            # AntifraudYtToken,
            FetchPath,
            CustomWholePrefixPath,
            SetOutputRemovalAttrs,
            DaysToStoreOutput,
            KeepBuildTables,
            LaunchFast,
        ) +
        FAST_SUB_FIELDS +
        (
            LaunchDaily,
        ) +
        DAILY_SUB_FIELDS
    )

    yt_token = None
    rem_runner = None

    def GetSvnUrl(self):
        if not self.ctx.get(ArcadiaUrl.name, ''):
            raise Exception("You have to specify main arcadia url (default arcadia url for all components)")
        return self.ctx[ArcadiaUrl.name]

    def ShouldApplyBinariesCashingLogic(self, arcadia_url):
        return GetRevision(arcadia_url) != 'HEAD' and not self.ctx.get(ArcadiaPatch.name)

    def GetFetchBinMethod(self, bin_name):
        if self.ctx.get(CustomizeBinaries.name):
            fetch_method_class = create_custom_input_params_classes(bin_name)[0]
            fetch_bin_method = self.ctx[fetch_method_class.name]
            if not fetch_bin_method:
                fetch_bin_method = fetch_method_class.default
            if fetch_bin_method != 'use_default_binaries_fetching_method':
                return fetch_bin_method

        fetch_bin_method = self.ctx[BinariesFetchMethod.name]
        if not fetch_bin_method:
            fetch_bin_method = BinariesFetchMethod.default
        return fetch_bin_method

    def CreateBuildSubtasksIfNeeded(self):
        if BIN_NAME_TO_TASK_ID_AND_SVN_URL not in self.ctx:
            # haven't created build_tasks yet
            bin_name_to_task_id_and_svn_url = {}
            bin_name_to_ready_resource_id = {}
            self.ctx["bins_to_rsync"] = []

            for binary_name, binary_spec in REM_BINARIES.items():
                binary_resource = binary_spec.resource_type
                fetch_method = self.GetFetchBinMethod(binary_name)
                if fetch_method == "dont_fetch_this_binary":
                    continue
                elif fetch_method == "rsync":
                    self.ctx["bins_to_rsync"].append(binary_name)
                    continue
                else:
                    if fetch_method == "custom_binary_arcadia_url":
                        custom_binary_arcadia_url_class = create_custom_input_params_classes(binary_name)[1]
                        build_from_svn_url = self.ctx[custom_binary_arcadia_url_class.name]
                    elif fetch_method == "custom_binaries_arcadia_url":
                        build_from_svn_url = self.ctx[CustomBinariesArcadiaUrl.name]
                    else:
                        build_from_svn_url = self.GetSvnUrl()

                    attrs_for_fetching_ready_bin = {BUILD_SVN_PATH_ATTR_NAME: build_from_svn_url} if self.ShouldApplyBinariesCashingLogic(build_from_svn_url) else None

                    (bin_resource_id, bin_build_task_id) = get_binary_from_svn_path(binary_name, binary_resource,
                                                                                    self, build_from_svn_url, attrs_for_fetching_ready_bin)

                    if bin_resource_id is not None:
                        bin_name_to_ready_resource_id[binary_name] = bin_resource_id
                    else:
                        bin_name_to_task_id_and_svn_url[binary_name] = (bin_build_task_id, build_from_svn_url)

            self.ctx[BIN_NAME_TO_TASK_ID_AND_SVN_URL] = bin_name_to_task_id_and_svn_url
            self.ctx[BIN_NAME_TO_READY_RESOURCE_ID] = bin_name_to_ready_resource_id
            utils.wait_all_subtasks_stop()
        elif not utils.check_all_subtasks_done():
            utils.restart_broken_subtasks()

    def CheckoutArcadiaSubfolder(self, arcadia_subfolder):
        svn_url = self.GetSvnUrl()
        pos = svn_url.rfind('@')

        if pos != -1:
            dir_url = pj(svn_url[:pos], arcadia_subfolder) + svn_url[pos:]
        else:
            dir_url = pj(svn_url, arcadia_subfolder)

        arcadia_subfolder_local_abs_path = pj(self.arcadia_src_dir, arcadia_subfolder)
        with arcadia_sdk.mount_arc_path(dir_url, use_arc_instead_of_aapi=True) as target_dir:
            copy_path(str(target_dir), arcadia_subfolder_local_abs_path)
            sys.path.insert(0, arcadia_subfolder_local_abs_path)

        return arcadia_subfolder_local_abs_path

    def PrepareArcadia(self):
        self.arcadia_src_dir = pj(self.abs_path(), 'local_arcadia')

        self.rem_processes_dir = self.CheckoutArcadiaSubfolder('quality/user_sessions/rem_processes')
        self.CheckoutArcadiaSubfolder('rem/client')
        self.CheckoutArcadiaSubfolder('logfeller/python')
        self.CheckoutArcadiaSubfolder('mapreduce/library/mr_packet_lib')
        self.rem_tools_dir = self.CheckoutArcadiaSubfolder('quality/rem-tools')

        Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get(ArcadiaPatch.name), self.abs_path())

    def PrepareBinaries(self):
        import sessions_processes_config as scripts_config  # noqa

        local_bin_folder = self.ConvertProductionPathToLocal(scripts_config.binHome)
        make_folder(local_bin_folder)

        self.SyncBinaries(local_bin_folder)

    def DoRsyncFromProductionServer(self, filename, production_folder, local_scripts_resources_folder):
        cmd = ['rsync', 'rsync://' + PRODUCTION_SERVER + pj(production_folder, filename), local_scripts_resources_folder]
        result, error = RunProcess(cmd, self.GetEnv(), 'rsync_' + filename)

    def RsyncFromProductionServer(self):
        import sessions_processes_config as scripts_config  # noqa
        import antifraud_packets_config

        PRODUCTION_FOLDER_TO_FILES_FOR_RSYNC = {
            scripts_config.binHome: [
                'chevent_hitlog_joiner',
            ],
            antifraud_packets_config.antifraudBinPath: ['fraud_filter', 'clean_sessions', 'sessions_monsters'],
            antifraud_packets_config.antifraudExternalDataPath: [
                'whitelist.lst',
                "blockstat.dict",
                "bad_hosts.lst",
                "bad_user_agents.lst",
                "good_hosts.lst",
                "referers.lst",
                "proxy.lst",
                "browser.xml",
                "bad_queries.lst",
                "rules.txt",
                "geodata4.bin",
                "white.cfg",
            ],
            antifraud_packets_config.antifraudDataPath: ['antifraud.cfg'],
        }

        if hasattr(scripts_config, 'usersBlackListPath'):
            make_folder(os.path.dirname(self.ConvertProductionPathToLocal(scripts_config.usersBlackListPath)))
            os.system('echo y123 > {}'.format(self.ConvertProductionPathToLocal(scripts_config.usersBlackListPath)))

        PRODUCTION_FOLDER_TO_FILES_FOR_RSYNC[scripts_config.binHome] += self.ctx['bins_to_rsync']

        # rsync from PRODUCTION_SERVER
        for production_folder, filenames in PRODUCTION_FOLDER_TO_FILES_FOR_RSYNC.iteritems():
            local_scripts_resources_folder = self.ConvertProductionPathToLocal(production_folder)
            make_folder(local_scripts_resources_folder)
            for filename in filenames:
                try:
                    self.DoRsyncFromProductionServer(filename, production_folder, local_scripts_resources_folder)
                except Exception:
                    self.logging_message_for_task_main_page += ('WARNING: Failed to rsync file {}. This might (or not ) cause "file-not-found"-FAILURE'
                                                                ' during creating REM packets'.format(filename) + '\n')

    def GetSandboxResources(self):
        import sessions_processes_config as scripts_config  # noqa
        COPY_SANDBOX_RESOURCES_TO_FOLDER = {
            scripts_config.statdataHome: {
                'ylogin_enc_key.txt': resources.SESSIONS_YLOGIN_ENC_KEY_TXT,
                'blockstat.dict': resources.SESSIONS_BLOCKSTAT,
                'blockstat.dict.json': resources.SESSIONS_BLOCKSTAT_JSON,
                'direct_pageids': resources.SESSIONS_DIRECT_PAGEIDS,
                'rtb_pageids': resources.SESSIONS_RTB_PAGEIDS,
                'direct_resourceno_dict': resources.SESSIONS_DIRECT_RESOURCENO_DICT,
                'direct_ads_descriptions': resources.SESSIONS_DIRECT_ADS_DESCRIPTIONS,
            },
            scripts_config.geodataHome: {
                'geodata6-xurma.bin': geo_resource.GEODATA6BIN_XURMA_STABLE,
            },
            scripts_config.qualitydataHome: {
                'beta_list.txt': resources.SESSIONS_BETA_LIST,
            },
        }
        # get latest resources from sandbox
        for production_folder, filenames in COPY_SANDBOX_RESOURCES_TO_FOLDER.iteritems():
            local_scripts_resources_folder = self.ConvertProductionPathToLocal(production_folder)
            make_folder(local_scripts_resources_folder)
            for filename, resource_type in filenames.items():
                try:
                    resource_path = self.sync_resource(apihelpers.get_last_resource(resource_type).id)
                    copy_path(resource_path, pj(local_scripts_resources_folder, filename))
                except Exception:
                    self.logging_message_for_task_main_page += ('WARNING: Failed to copy file {}. This might (or not ) cause "file-not-found"-FAILURE'
                                                                ' during creating REM packets'.format(filename) + '\n')

    def ReplaceScriptConfigPaths(self):
        import sessions_processes_config as scripts_config  # noqa
        import antifraud_packets_config

        scripts_config.binHome = self.ConvertProductionPathToLocal(scripts_config.binHome)
        scripts_config.geodataHome = self.ConvertProductionPathToLocal(scripts_config.geodataHome)
        scripts_config.statdataHome = self.ConvertProductionPathToLocal(scripts_config.statdataHome)
        scripts_config.qualitydataHome = self.ConvertProductionPathToLocal(scripts_config.qualitydataHome)
        scripts_config.sessionsToolsHome = self.ConvertProductionPathToLocal(scripts_config.sessionsToolsHome)
        antifraud_packets_config.antifraudBinPath = self.ConvertProductionPathToLocal(antifraud_packets_config.antifraudBinPath)
        antifraud_packets_config.antifraudExternalDataPath = self.ConvertProductionPathToLocal(antifraud_packets_config.antifraudExternalDataPath)
        antifraud_packets_config.antifraudDataPath = self.ConvertProductionPathToLocal(antifraud_packets_config.antifraudDataPath)

        if hasattr(scripts_config, 'usersBlackListPath'):
            scripts_config.usersBlackListPath = self.ConvertProductionPathToLocal(scripts_config.usersBlackListPath)

    def ModifySessionsConfig(self):
        import sessions_config

        dailyScarabUserSessionsSourceLogsToSkip = self.ctx.get(DailyScarabUserSessionsSourceLogsToSkip.name, '').strip(',').split(',')
        sessions_config.dailyScarabUserSessionsSourceLogs = [log for log in sessions_config.dailyScarabUserSessionsSourceLogs if log not in dailyScarabUserSessionsSourceLogsToSkip]

        # LOGSTAT-6549
        late_recalc_mode = self.ctx.get(LateRecalcMode.name)
        if late_recalc_mode:
            for period_cfg in sessions_config.lateSessions.values():
                for log_cfg in period_cfg.values():
                    backlog = log_cfg.get("backlog")
                    if backlog:
                        if late_recalc_mode == 'current_only':
                            log_cfg["recalcDays"] = set([0])
                        elif late_recalc_mode == 'all_backlog':
                            log_cfg["recalcDays"] = set([x for x in range(0, backlog + 1)])
                        else:
                            raise NotImplementedError()

    def HandleAntifraud(self):
        import antifraud_packets_config
        import time_periods

        if not self.ctx.get(ApplyAntifraudLogic.name, False):
            # for substituting clean-packets with fake packets which just copy and sort
            antifraud_packets_config.fraudFilterConfig[time_periods.Periods.FAST] = {}
            antifraud_packets_config.fraudFilterConfig[time_periods.Periods.DAILY] = {}
        else:
            # later here we'll have configuring antifraud's env
            pass

    def SetTokens(self):
        self.yt_token = self.GetYtToken()

        import sessions_processes_config as scripts_config  # noqa
        import antifraud_packets_config

        yt_token_path = pj(self.abs_path(), 'yt_token_file')
        os.system('echo {} > {}'.format(self.yt_token, yt_token_path))
        for token in scripts_config.YTTokenPath.keys():
            scripts_config.YTTokenPath[token] = yt_token_path

        antifraud_yt_token = self.ctx.get(AntifraudYtToken.name, '')
        if antifraud_yt_token and self.ctx.get(ApplyAntifraudLogic.name, False):
            antifraud_yt_token_path = pj(self.abs_path(), 'antifraud_yt_token_file')
            os.system('echo {} > {}'.format(antifraud_yt_token, antifraud_yt_token_path))
            antifraud_packets_config.ytAntifraudTokenPath = antifraud_yt_token_path
        else:
            antifraud_packets_config.ytAntifraudTokenPath = yt_token_path

    def GetEnv(self):
        env = dict(os.environ)
        if self.yt_token:
            env['MR_RUNTIME'] = 'YT'
            env['YT_PREFIX'] = '//'
            env['YT_TOKEN'] = self.yt_token
            env['YT_PROXY'] = self.ctx[MRCluster.name]
        env['PYTHONPATH'] = ':'.join(sys.path)

        return env

    def StripYTSessionsBasePaths(self):
        import sessions_lib

        sessions_lib.YT_SESSIONS_BASE_PATH = sessions_lib.YT_SESSIONS_BASE_PATH.lstrip('/')
        sessions_lib.YT_RAW_SESSIONS_BASE_PATH = sessions_lib.YT_RAW_SESSIONS_BASE_PATH.lstrip('/')

    def InitWorkingFolder(self):
        self.ctx[WORKING_FOLDER] = MRPathJoin(self.ctx[FetchPath.name], 'user_sessions_processes_testing_r{}_{}'.format(
            GetRevision(self.GetSvnUrl()),
            ''.join(random.SystemRandom(time.time()).choice(
                string.ascii_lowercase + string.ascii_uppercase + string.digits
            ) for _ in range(20)),
        ))

        cmd = ['yt', 'create', '-ri', 'map_node', self.ctx[WORKING_FOLDER]]
        result, error = RunProcess(cmd, self.GetEnv())

        if self.ctx.get(SetOutputRemovalAttrs.name, False):
            import yt.wrapper as yt
            working_folder_yt_full_path = GetYtFullPath(self.ctx[WORKING_FOLDER])
            days_to_store = self.ctx.get(DaysToStoreOutput.name, DEFAULT_DAYS_TO_STORE_OUTPUT)
            yt.set_attribute(working_folder_yt_full_path, 'expiration_time', int(time.time() + days_to_store * 24 * 60 * 60) * 1000)

        cmd = ['yt', 'create', '-ri', 'map_node', pj(self.ctx[WORKING_FOLDER], 'user_sessions/build/tmp')]
        result, error = RunProcess(cmd, self.GetEnv())

        raw_logs_folder = MRPathJoin(self.ctx[FetchPath.name], 'raw_logs')
        cmd = ['yt', 'list', raw_logs_folder]
        result, error = RunProcess(cmd, self.GetEnv())

        def _yt_link_paths(src_subnodes, src_path, dst_path):
            for subnode in src_subnodes:
                src_path_abs = MRPathJoin(src_path, subnode)
                dst_path_abs = MRPathJoin(dst_path, subnode)
                cmd = ['yt', 'link', '-i', src_path_abs, dst_path_abs]
                result, error = RunProcess(cmd, self.GetEnv())

        raw_logs_subnodes = filter(lambda p: p != 'home', result.strip('\n').split('\n'))
        _yt_link_paths(raw_logs_subnodes, raw_logs_folder, self.ctx[WORKING_FOLDER])

        # SCARAB-932
        raw_logs_home_path = MRPathJoin(self.ctx[WORKING_FOLDER], 'home')
        cmd = ['yt', 'create', '-ri', 'map_node', raw_logs_home_path]
        result, error = RunProcess(cmd, self.GetEnv())

        raw_logs_home_folder = MRPathJoin(raw_logs_folder, 'home')
        cmd = ['yt', 'list', raw_logs_home_folder]
        result, error = RunProcess(cmd, self.GetEnv())

        raw_logs_home_subnodes = filter(lambda p: p != 'mobilesearch', result.strip('\n').split('\n'))
        _yt_link_paths(raw_logs_home_subnodes, raw_logs_home_folder, raw_logs_home_path)

    def SetFetchedLogsParameters(self):
        if 'path_resource_id' in self.ctx:
            freshdata_resource_id = self.ctx['path_resource_id']
        else:
            freshdata_resource_id = apihelpers.get_last_resource_with_attrs(
                rst.USER_SESSIONS_PROCESSES_FRESHDATA_INFO,
                su.parse_attrs('debug=False'),
                all_attrs=True,
            ).id

        freshdata_resource_path = self.sync_resource(freshdata_resource_id)
        with open(freshdata_resource_path, 'r') as f:
            content_lines = f.read().split('\n')
            if not self.ctx.get(FetchPath.name, ''):
                self.ctx[FetchPath.name] = content_lines[0].strip()
            if not self.ctx.get(DailyDatetime.name, ''):
                self.ctx[DailyDatetime.name] = content_lines[2].strip()
            if not self.ctx.get(FastDatetime.name, ''):
                self.ctx[FastDatetime.name] = content_lines[4].strip()

    def SetYtPools(self):
        import sessions_processes_config as scripts_config

        try:
            if not self.ctx.get(ManualRecalcMode.name) and self.ctx.get(SubstituteYtPoolsWithCutomOne.name, True):
                customYtPool = self.ctx.get(CustomYtPool.name, DEFAULT_CI_YT_POOL)
                for ytPoolKey in scripts_config.YTPools.keys():
                    scripts_config.YTPools[ytPoolKey] = customYtPool
        except Exception:
            self.ctx['TRIED_AND_FAILED_TO_SUBSTITUTE_YT_POOLS'] = True

    def GetYtToken(self):
        if self.ctx[ManualRecalcMode.name]:
            return self.get_vault_data("USERSESSIONSTOOLS", "recalc_token")
        else:
            yt_token = self.ctx[YtToken.name]
            if not yt_token:
                yt_token = YtToken.default
            if yt_token == 'recalc-token':
                return self.get_vault_data("USERSESSIONSTOOLS", "recalc_token")
            elif yt_token == 'CI-token':
                return self.get_vault_data("USERSESSIONSTOOLS", "userdata-sessions-build-ci-token")
            else:
                owner = self.ctx.get(CustomYTTokenOwner.name, '')
                if not owner:
                    owner = self.owner
                return self.get_vault_data(owner, self.ctx[CustomYTToken.name])

    def SetYtClusters(self):
        import sessions_processes_config as scripts_config  # noqa

        if hasattr(scripts_config, 'sessionsTypeToYTClustersSpec'):
            customClustersSpec = scripts_config.YTClustersSpec(self.ctx[MRCluster.name], [self.ctx[MRCluster.name]])
            for sessionsTypeKey in scripts_config.sessionsTypeToYTClustersSpec.keys():
                scripts_config.sessionsTypeToYTClustersSpec[sessionsTypeKey] = customClustersSpec
        else:
            scripts_config.prodYTClusters = [self.ctx[MRCluster.name]]
            if hasattr(scripts_config, 'robotYTClusters'):
                scripts_config.robotYTClusters = []  # we don't want to test syncing to other clusters
                scripts_config.allYTClusters = scripts_config.prodYTClusters + scripts_config.robotYTClusters
            else:
                scripts_config.robotYTCluster = None  # we don't want to test syncing to other clusters
                scripts_config.allYTClusters = scripts_config.prodYTClusters

    def ConfigureYtEnv(self):
        self.SetTokens()
        self.SetYtPools()
        self.StripYTSessionsBasePaths()
        self.SetFetchedLogsParameters()

        import yt.wrapper as yt
        yt.config["proxy"]["url"] = self.ctx[MRCluster.name]
        yt.config["token"] = self.yt_token

        self.SetYtClusters()
        if self.ctx.get(CustomWholePrefixPath.name, ''):
            self.ctx[WORKING_FOLDER] = self.ctx.get(CustomWholePrefixPath.name, '').lstrip('/')
        else:
            self.InitWorkingFolder()

        import sessions_processes_config as scripts_config  # noqa
        scripts_config.mrPrefix = self.ctx[WORKING_FOLDER]
        if scripts_config.mrPrefix != '' and scripts_config.mrPrefix[-1] != '/':
            scripts_config.mrPrefix += '/'  # scripts require one slash in the end of the mrPrefix

    def ConfigureRemScriptsOutputLifetimes(self):
        import sessions_processes_config as scripts_config  # noqa

        for output_type in scripts_config.OutputTypeToLifetimeDays.keys():
            scripts_config.OutputTypeToLifetimeDays[output_type] = 'inf'

    def ConfigureStepUrls(self):
        import sessions_processes_config as scripts_config

        if hasattr(scripts_config, 'prodStepUrl'):
            if not hasattr(scripts_config, 'testStepUrl'):
                self.KillLocalRem()
                eh.check_failed('sessions_proceses_config.testStepUrl is needed to be used instead of '
                                   'sessions_proceses_config.prodStepUrl during testing!')
            scripts_config.prodStepUrl = scripts_config.testStepUrl

    def GetPacketLabelsBunches(self, should_use_default_packet_labels_param_class, extra_labels_param_class):
        extra_labels_bunches = extract_labels(self.ctx.get(extra_labels_param_class.name, ''))

        common_packet_labels = []
        if self.ctx.get(should_use_default_packet_labels_param_class.name, True):
            common_packet_labels += DEFAULT_PACKET_LABELS
            if not self.ctx.get(ApplyAntifraudLogic.name, False):
                common_packet_labels.append(DISABLE_ANTIFRAUD_RULES_PACKET_LABEL)

        if len(extra_labels_bunches) != 0:
            return [extra_labels_bunch + common_packet_labels for extra_labels_bunch in extra_labels_bunches]
        else:
            return [common_packet_labels]

    def LaunchRemScripts(self):
        import time_periods
        import rem_tasks

        def test_create_sessions(rem_address, date_time, time_period, should_use_default_packet_labels_param_class, extra_labels_param_class):
            options = rem_tasks.Options(
                rem_address=rem_address,
                date_time=date_time,
                time_period=time_period,
            )
            options.set_manual_recalc_mode(self.ctx.get(ManualRecalcMode.name, False))
            packet_labels_bunches = self.GetPacketLabelsBunches(should_use_default_packet_labels_param_class, extra_labels_param_class)
            for packet_labels_bunch in packet_labels_bunches:
                options.set_packet_labels(packet_labels_bunch)
                try:
                    import create_sessions
                    create_sessions.create_sessions(options)
                except Exception:
                    diagnosis = ''.join(traceback.format_exception(*sys.exc_info()))
                    self.KillLocalRem()
                    eh.check_failed(diagnosis)

        if self.ctx[LaunchFast.name]:
            self.dt_fast = datetime.datetime.strptime(self.ctx[FastDatetime.name], TIME_FORMAT)
            test_create_sessions(self.rem_server_address, self.dt_fast, time_periods.Periods.FAST, UseDefaultPacketLabelsForFast, ExtraLabelsForFast)

        if self.ctx[LaunchDaily.name]:
            self.dt_daily = datetime.datetime.strptime(self.ctx[DailyDatetime.name], TIME_FORMAT)
            test_create_sessions(self.rem_server_address, self.dt_daily, time_periods.Periods.DAILY, UseDefaultPacketLabelsForDaily, ExtraLabelsForDaily)

    def SetDanglingDeps(self):
        cmd = ['/skynet/python/bin/python', pj(self.rem_tools_dir, 'set-dangling-deps.py'), '-r', self.rem_server_address]
        result, error = RunProcess(cmd, self.GetEnv(), 'set-dangling-deps')

    def ChangeQueueWorkingLimit(self, rem_connector, time_period, working_limit):
        import sessions_common

        sessions_queues_names = []
        if hasattr(sessions_common, 'getProdOrPreprodQueue'):
            for is_preprod in [False, True]:
                sessions_queues_names.append(sessions_common.getProdOrPreprodQueue(isPreprod=is_preprod, period=time_period, cluster=str(self.ctx[MRCluster.name])))
        else:
            sessions_queues_names.append(sessions_common.getSessionsQueue(period=time_period, cluster=str(self.ctx[MRCluster.name])))

        for sessions_queue_name in sessions_queues_names:
            try:
                sessions_queue = rem_connector.Queue(sessions_queue_name)
                sessions_queue.ChangeWorkingLimit(working_limit)
            except Exception:
                # queue hasn't been created
                pass

    def ChangeQueuesWorkingLimits(self):
        import remclient
        from time_periods import Periods

        rem_connector = remclient.Connector(url="http://" + self.rem_server_address + "/")

        if self.ctx[LaunchFast.name]:
            working_limit = self.ctx.get(FastQueueWorkingLimit.name, QUEUE_WORKING_LIMIT)
            self.ChangeQueueWorkingLimit(rem_connector, Periods.FAST, working_limit)
        if self.ctx[LaunchDaily.name]:
            working_limit = self.ctx.get(DailyQueueWorkingLimit.name, QUEUE_WORKING_LIMIT)
            self.ChangeQueueWorkingLimit(rem_connector, Periods.DAILY, working_limit)

    def LaunchLocalRem(self):
        def get_extra_env_vars():
            extra_env_vars = {}

            # rem's packets use precisely 'mapreduce-yt', not './mapreduce-yt' for example
            mr_yt_binary_path = self.sync_resource(apihelpers.get_last_released_resource("MAPREDUCE_YT_EXECUTABLE").id)
            extra_env_vars['PATH'] = ':'. join([
                os.path.dirname(mr_yt_binary_path),
                os.environ.get('PATH', '')
            ])

            return extra_env_vars

        self.rem_runner = common_rem.REMRunner("arcadia:/arc/trunk/arcadia", pj(self.abs_path(), 'local_rem'), environ=get_extra_env_vars())
        self.rem_runner.setup_server()
        self.rem_server_address = self.rem_runner.rem_url.replace('http://', '')

    def KillLocalRem(self):
        if self.rem_runner:
            self.rem_runner.teardown_server()

    def UpdateMessage(self, cmd, out, err):
        self.tools_messages += ' '.join(cmd) + '\n'
        self.tools_messages += 'STDOUT' + '\n'
        self.tools_messages += out + '\n'
        self.tools_messages += 'STDERR' + '\n'
        self.tools_messages += err + '\n'

    def RunCheckSessionsCmd(self, cmd, log_prefix, exception_if_nonzero_code=True):
        import sessions_processes_config as scripts_config  # noqa

        env = self.GetEnv()
        env['YT_PREFIX'] = '//' + scripts_config.mrPrefix
        return RunProcess(cmd, env, log_prefix, exception_if_nonzero_code)

    def GetVerboseInfoAboutErroredPackets(self, out_result):
        import remclient
        import re

        rem_connector = remclient.Connector(url='http://' + self.rem_server_address + '/')
        for packet_info in out_result.split('\n'):
            if 'PACKET' in packet_info and 'ERROR' in packet_info:
                packet_id = packet_info.split('\t')[1]
                pck = rem_connector.PacketInfoByAnyIdentity(packet_id)
                self.tools_messages += 'INFO about PACKET {}:\n'.format(packet_id)

                for job in pck.jobs:
                    results_joined = "\n".join(i.data for i in job.results)
                    matches = re.findall(r'\bstarted: (\d{4}/\d\d/\d\d \d\d:\d\d:\d\d)', results_joined)
                    job.last_start_time = int(datetime.datetime.strptime(matches[-1], '%Y/%m/%d %H:%M:%S').strftime('%s')) \
                        if matches else None

                format = "\n[{id}]\t{command}"
                for job in sorted(pck.jobs, key=lambda j: j.last_start_time):
                    fields = dict(
                        descr=job.desc,
                        id=job.id,
                        command=job.shell.encode('utf-8'),
                    )

                    self.tools_messages += format.format(**fields) + '\n'

                    if not getattr(job, 'wait_jobs', None):
                        self.tools_messages += job.state + '\n'
                    else:
                        self.tools_messages += '%s (waiting for jobs: %s)' % (job.state, ', '.join(str(j.id) for j in job.wait_jobs)) + '\n'
                    self.tools_messages += '\n'.join(i.data.replace('\n', '\n\t') for i in job.results) + '\n'

    def CheckSessionProgress(self, session_name, time_period_str, date_time):
        def is_success(out, err):
            not_success_signs = ['TAG', 'ERROR', 'PENDING', 'WORKABLE', 'WAITING']

            success = not err and 'SUCCESSFULL' in out
            for sign in not_success_signs:
                success = success and sign not in out
            if 'ERROR' in out:
                self.success_wont_happen = True

            return success

        cmd_base = ['/skynet/python/bin/python', pj(self.rem_processes_dir, 'check_progress.py')]
        cmd_options = ['-r', self.rem_server_address, '-c', self.ctx[MRCluster.name], '-s', session_name, '-p', time_period_str, '-t', date_time]
        cmd = cmd_base + cmd_options
        result_progress, error_progress = self.RunCheckSessionsCmd(cmd, 'check_progress_{}_{}'.format(session_name, time_period_str))
        self.UpdateMessage(cmd, result_progress, error_progress)
        self.GetVerboseInfoAboutErroredPackets(result_progress)

        return is_success(result_progress, error_progress)

    def CheckSessionsExistence(self, time_period_str, date_time):
        cmd_base = ['/skynet/python/bin/python', pj(self.rem_processes_dir, 'check_sessions_existence.py')]
        cmd_options = ['-r', self.rem_server_address, '-c', self.ctx[MRCluster.name], '-p', time_period_str, '--time', date_time]
        cmd = cmd_base + cmd_options
        result, error = self.RunCheckSessionsCmd(cmd, 'check_sessions_existence_{}'.format(time_period_str), False)
        self.UpdateMessage(cmd, result, error)
        return not error

    def GetSessionsNames(self, joined_sessions_names):
        return joined_sessions_names.strip(",").split(",")

    def CheckProgress(self):
        import sessions_lib  # to execute registering sessions
        sessions_lib.YT_SESSIONS_BASE_PATH

        from sessions_registry import SessionsRegistry
        from time_periods import Periods

        def common_sessions_names(time_period):
            black_list = ["search_preprod"]
            if hasattr(SessionsRegistry(), 'reactor_sessions'):
                black_list += SessionsRegistry().reactor_sessions(time_period)
            return [name for name in SessionsRegistry().names(time_period) if name not in black_list]

        is_success = True
        self.all_sessions = []

        if self.ctx[LaunchFast.name]:
            custom_fast_sessions_to_check = self.ctx.get(FastSessionsToCheck.name, '')
            if custom_fast_sessions_to_check:
                fast_sessions_names = custom_fast_sessions_to_check.strip(',').split(',')
            else:
                fast_sessions_names = common_sessions_names(Periods.FAST)

            for session_name in fast_sessions_names:
                self.all_sessions.append([session_name, Periods.FAST, self.dt_fast])
                is_success = self.CheckSessionProgress(session_name, '30min', self.ctx[FastDatetime.name]) and is_success

        if self.ctx[LaunchDaily.name]:
            custom_daily_sessions_to_check = self.ctx.get(DailySessionsToCheck.name, '')
            if custom_daily_sessions_to_check:
                daily_sessions_names = custom_daily_sessions_to_check.strip(',').split(',')
            else:
                daily_sessions_names = common_sessions_names(Periods.DAILY)

            for session_name in daily_sessions_names:
                self.all_sessions.append([session_name, Periods.DAILY, self.dt_daily])
                is_success = self.CheckSessionProgress(session_name, '1d', self.ctx[DailyDatetime.name]) and is_success

        if self.ctx[LaunchFast.name] and not self.ctx.get(FastSessionsToCheck.name, ''):
            is_success = self.CheckSessionsExistence('30min', self.ctx[FastDatetime.name]) and is_success
        if self.ctx[LaunchDaily.name] and not self.ctx.get(DailySessionsToCheck.name, ''):
            is_success = self.CheckSessionsExistence('1d', self.ctx[DailyDatetime.name]) and is_success

        return is_success

    def ConvertProductionPathToLocal(self, production_path):
        return pj(self.abs_path(), production_path.lstrip('/'))

    def DoSyncBinary(self, res_id, bin_name, bin_folder):
        bin_path = self.sync_resource(res_id)
        copy_path(bin_path, pj(bin_folder, bin_name))

    def SyncBinaries(self, local_bin_folder):
        # build subtasks have been created and have already finished
        import sessions_processes_config as scripts_config  # noqa

        # sync binaries which have already been built long ago
        for bin_name, res_id in self.ctx[BIN_NAME_TO_READY_RESOURCE_ID].iteritems():
            self.DoSyncBinary(res_id, bin_name, local_bin_folder)

        failed_builds_count = 0
        # sync binaries which our subtasks have just built
        for bin_name, build_task_id_and_svn_url in self.ctx[BIN_NAME_TO_TASK_ID_AND_SVN_URL].iteritems():
            build_task_id, build_from_svn_url = build_task_id_and_svn_url
            task = channel.sandbox.get_task(build_task_id)
            if task.is_failure():
                # FAILURE status might mean that project of binary doesn't exist yet in the given svn url.
                # It means that we are currently testing some old version (svn url) of user_sessions processes,
                # where this binary is not needed and shouldn't have been tried to build in the first place
                failed_builds_count += 1
                self.logging_message_for_task_main_page += ('WARNING: Failed to build binary {}. This might (or not ) cause "file-not-found"-FAILURE'
                                                            ' during creating REM packets'.format(bin_name) + '\n')
                continue
            res_id = task.ctx["%s_resource_id" % bin_name]
            self.DoSyncBinary(res_id, bin_name, local_bin_folder)
            attrs_to_set = {
                "ttl": "14",
            }
            if self.ShouldApplyBinariesCashingLogic(build_from_svn_url):
                attrs_to_set[BUILD_SVN_PATH_ATTR_NAME] = build_from_svn_url
            set_resource_attributes(res_id, attrs_to_set)
        if failed_builds_count > 0 and self.ctx.get('exception_on_build_fail', True):
            raise Exception("Some binaries failed to build!")

    def CopySomeFilesFromLocalArcadia(self):
        import sessions_processes_config as scripts_config  # noqa

        sessions_tools = pj(self.rem_processes_dir, "tools")
        PRODUCTION_PATH_TO_LOCAL_ARCADIA_PATH = {
            scripts_config.sessionsToolsHome: sessions_tools,
        }

        for production_path, local_arcadia_path in PRODUCTION_PATH_TO_LOCAL_ARCADIA_PATH.iteritems():
            production_path_local_analog = self.ConvertProductionPathToLocal(production_path)
            copy_path(local_arcadia_path, production_path_local_analog)

    def ConfigureRemScriptsEnv(self):
        self.PrepareBinaries()
        self.CopySomeFilesFromLocalArcadia()
        self.RsyncFromProductionServer()
        self.GetSandboxResources()
        self.ReplaceScriptConfigPaths()
        self.ModifySessionsConfig()
        self.HandleAntifraud()
        self.ConfigureYtEnv()
        self.ConfigureRemScriptsOutputLifetimes()
        self.ConfigureStepUrls()

    def MonitorProgress(self):
        for try_index in xrange(self.ctx[CheckProgressTries.name]):
            time.sleep(self.ctx[NextCheckProgressAfter.name])

            self.tools_messages = ''
            self.success_wont_happen = False

            is_success = self.CheckProgress()

            logging.info('Tools\' messages {}:\n{}'.format(try_index, self.tools_messages))
            if self.success_wont_happen:
                return False
            if is_success:
                return True

        return False

    def CleanBuildTables(self):
        if self.ctx.get(KeepBuildTables.name, True):
            return

        import yt.wrapper as yt
        from sessions_registry import SessionsRegistry

        for session_name, period, dt in self.all_sessions:
            sessions = SessionsRegistry().get(session_name, period)

            if not hasattr(sessions, 'RawPaths'):
                logging.info('skip cleaning build tables for sessions {} coz rem_processes is too old and havent RawPaths attribute')
                continue

            for build_path_rel in sessions.RawPaths(dt):
                build_path_abs = GetYtFullPath(MRPathJoin(self.ctx[WORKING_FOLDER], build_path_rel))
                logging.info('cleaning ' + build_path_abs)
                yt.remove(build_path_abs, recursive=True, force=True)

    def on_execute(self):
        self.logging_message_for_task_main_page = ''

        self.CreateBuildSubtasksIfNeeded()

        self.PrepareArcadia()
        self.LaunchLocalRem()
        self.ConfigureRemScriptsEnv()
        self.LaunchRemScripts()
        self.SetDanglingDeps()
        self.ChangeQueuesWorkingLimits()

        is_success = self.MonitorProgress()
        self.KillLocalRem()
        eh.ensure(is_success, self.logging_message_for_task_main_page + self.tools_messages)

        self.CleanBuildTables()


__Task__ = UserSessionsProcessesTest
