from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk import environments

import logging
from os.path import join as pj
import os
import sys
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.input_params2 as rm_params
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.logs.common import GetRevision, RunProcess
from sandbox.projects.logs.common.reactor import RegisterReactorFiles


# NOTE(ngc224): unfortunately we cannot use binary build now,
# so we cannot reuse quality/user_sessions/reactor/us_processes/logging.py code here..
def PatchRootLogger():
    rootLogger = logging.getLogger()
    formatter = logging.Formatter(
        fmt='%(asctime)s +%(relativeCreated)-8d %(levelname)8s %(name)s (%(threadName)s): %(message)s',
        datefmt='%F %T',
    )

    for handler in rootLogger.handlers:
        handler.setFormatter(formatter)


class UsReactorScriptsReleaseInfo(sdk2.Resource):
    """
        File with versions of reactor graph
    """
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True

class UsReactorFallbackScriptsReleaseInfo(sdk2.Resource):
    """
        File with versions of fallback reactor graph
    """
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    branch = sdk2.parameters.String()


@rm_notify.notify2()
class ReleaseUserSessionsScripts(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('networkx', version='2.2', use_wheel=True),
        ]
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(rm_params.ComponentName2):
        reg_general_arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for reactor (should be with revision). For example: arcadia:/arc/branches/user_sessions/stable-<number>/arcadia@<revision>', required=True)
        reactor_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with reactor token", required=True)
        reactor_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with reactor token", required=True)
        is_debug = sdk2.parameters.Bool("Use mock for testing", default=False)
        threads_count = sdk2.parameters.Integer("Threads count for Reactor API requests. 0 for default value, NOT to disable", default=0)
        fallback_mode = sdk2.parameters.Bool("Use for big problems and long cluster works", default=False)
        revert_from_fallback = sdk2.parameters.Bool("Use for revert from fallback mode", default=False)

        kill_timeout = 8 * 3600

        with sdk2.parameters.Group("Time period") as period_group:
            period_daily = sdk2.parameters.Bool("1d", default=True)
            period_fast = sdk2.parameters.Bool("30min", default=True)

    def GetReactorTokenPath(self):
        secret_owner = self.Parameters.reactor_token_secret_owner
        secret_name = self.Parameters.reactor_token_secret_name
        secret_content = sdk2.Vault.data(secret_owner, secret_name)
        reactor_token_path = pj(str(self.path()), 'reactor_token_file')
        os.system('echo {} > {}'.format(secret_content, reactor_token_path))

        return reactor_token_path

    def get_reactor_url(self):
        if self.Parameters.is_debug:
            return "MOCK"
        return "reactor.yandex-team.ru"

    def GetEnv(self):
        env = dict(os.environ)
        env['PYTHONPATH'] = ':'.join(sys.path)
        return env

    def GetSvnUrl(self, scripts_url):
        revision = GetRevision(self.Context.arcadia_url)
        branch_path = None
        if self.Parameters.reg_general_arcadia_url.count("@") != 1:
            branch_path = self.Context.arcadia_url
        else:
            branch_path = self.Context.arcadia_url.split("@")[0]
        return '{}/{}@{}'.format(branch_path, scripts_url, revision)

    def CheckoutArcadiaSubfolder(self, arcadia_subfolder, use_cache=True):
        dir_url = self.GetSvnUrl(arcadia_subfolder)
        arcadia_subfolder_local_abs_path = pj(self.arcadia_src_dir, arcadia_subfolder)

        if use_cache:
            with arcadia_sdk.mount_arc_path(dir_url, use_arc_instead_of_aapi=True) as p:
                sdk2.paths.copy_path(str(p), arcadia_subfolder_local_abs_path)
        else:
            sdk2.svn.Arcadia.checkout(dir_url, arcadia_subfolder_local_abs_path)
        return arcadia_subfolder_local_abs_path

    def PrepareArcadia(self):
        self.arcadia_src_dir = pj(str(self.path()), 'local_arcadia')
        self.reactor_dir = self.CheckoutArcadiaSubfolder('quality/user_sessions/reactor', use_cache=False)
        sys.path.insert(0, self.reactor_dir)

    def ParseParameters(self):
        self.time_period_opts = []
        if self.Parameters.period_daily:
            self.time_period_opts.extend(["-p", "1d"])
        if self.Parameters.period_fast:
            self.time_period_opts.extend(["-p", "30min"])
        assert self.time_period_opts, "Time period is required"

    def prepare_branch_url(self):
        return self.Parameters.reg_general_arcadia_url.replace('arcadia:/arc/branches/', '').split('/arcadia@')[0]


    def on_execute(self):
        if self.Parameters.revert_from_fallback:
            self.ReleaseGraph()
        else:
            PatchRootLogger()

            self.Context.arcadia_url = self.Parameters.reg_general_arcadia_url
            self.ParseParameters()

            logging.info('ReleaseUserSessionsScriptsTask: Start')
            self.version_file = pj(str(self.path()), "version_file.txt")
            self.PrepareArcadia()
            logging.info('DoBuild result: {}'.format(str(self.DoBuild())))
            logging.info('DoRegisterFiles result: {}'.format(str(self.DoRegisterFiles())))

            if self.Parameters.fallback_mode:
                resource = UsReactorFallbackScriptsReleaseInfo(
                        self,
                        "Version file",
                        self.version_file,
                        branch=self.prepare_branch_url()
                )
            else:
                resource = UsReactorScriptsReleaseInfo(self, "Version file", self.version_file)
            sdk2.ResourceData(resource)
            self.Context.resource_id = resource.id
            logging.info('revision is {}'.format(str(GetRevision(self.Parameters.reg_general_arcadia_url))))

    def DoRegisterFiles(self):
        env = self.GetEnv()
        reac_token_path = self.GetReactorTokenPath()
        return RegisterReactorFiles(file_name="all", token_path=reac_token_path, env=env)

    def DoBuild(self):
        reac_token_path = self.GetReactorTokenPath()
        cmd = [
            sys.executable,
            "-m", "us_processes.create_sessions",
            "-s", self.get_reactor_url(),
            "--svn-url", self.Parameters.reg_general_arcadia_url,
            "-t", reac_token_path,
            "--prod", "--quota-project", "user-sessions",
            "--graph-owner", "robot-make-sessions",
            "--version-file", self.version_file,
            "--verbose", "--debug",
        ] + self.time_period_opts

        if self.Parameters.threads_count:
            cmd.extend(['--threads-count', str(self.Parameters.threads_count)])

        if self.Parameters.fallback_mode:
            cmd.append('--fallback_mode')

        logging.info("command {}".format(str(cmd)))
        result, _ = RunProcess(cmd, self.GetEnv(), log_prefix="do_release_bin_artifacts", exception_if_nonzero_code=True)
        return result

    def GetVersionFromResource(self):
        if self.Parameters.fallback_mode:
            version_file = UsReactorFallbackScriptsReleaseInfo.find(attrs=dict(branch=self.prepare_branch_url())).first()
        else:
            version_file = sdk2.Resource["US_REACTOR_SCRIPTS_RELEASE_INFO"].find(
               id=self.Context.resource_id,
            ).first()

        assert version_file, "US_REACTOR_SCRIPTS_RELEASE_INFO with id={} is not found".format(self.Context.resource_id)
        logging.info('Res is {}'.format(version_file))
        version_file_path = sdk2.ResourceData(version_file).path
        version = None
        with open(str(version_file_path)) as f:
            version = f.readline()
            f.close()
        assert version, "No version from {}".format(version_file)
        if self.Parameters.revert_from_fallback:
            version = version.replace("_fallback", "")
        return version

    def DoRelease(self):
        reac_token_path = self.GetReactorTokenPath()
        version = self.GetVersionFromResource()
        cmd = [
            sys.executable,
            "-m",
            "us_processes.release_graph",
            "-s", self.get_reactor_url(),
            "-gv", version,
            "-t", reac_token_path,
        ] + self.time_period_opts

        if self.Parameters.fallback_mode:
            cmd.append("-i")

        if self.Parameters.revert_from_fallback:
            cmd.append("-fb")

        logging.info("command {}".format(str(cmd)))
        result, _ = RunProcess(cmd, self.GetEnv(), log_prefix="do_release_bin_artifacts", exception_if_nonzero_code=True)
        return result

    def ReleaseGraph(self):
        self.Context.arcadia_url = Arcadia.ARCADIA_TRUNK_URL
        self.ParseParameters()
        self.PrepareArcadia()
        self.DoRelease()

    def on_release(self, release_params):
        if not self.Parameters.revert_from_fallback:
            self.ReleaseGraph()
