# -*- coding: utf-8 -*-

import re
import sys
import time
import os
from os.path import join as pj
import logging
import shutil

from sandbox.common.context import NullContextmanager
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.svn import Arcadia, SvnError
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.arcadia import sdk


PR_WATCHERS = ['kender']


def MRPathJoin(*args):
    return '/'.join([str(arg).strip('/') for arg in args])


def ReactorPathJoin(*args):
    return pj("/", *[arg.strip('/') for arg in args])


def GetRevision(url):
    if url.count("@") != 1:
        return "HEAD"
    return url.split("@")[-1]


class LocalMountPoint(object):
    def __init__(self, path):
        self.__path = path

    def __enter__(self, *args):
        return self.__path

    def __exit__(self, *args):
        pass


def GetArcadiaAAPI(arcadia_url, path=None):
    if path:
        parsed_url = Arcadia.parse_url(arcadia_url)

        if parsed_url.subpath:
            path = pj(parsed_url.subpath, path)

        if parsed_url.trunk:
            arcadia_url = Arcadia.trunk_url(path=path, revision=parsed_url.revision)
        elif parsed_url.branch:
            arcadia_url = Arcadia.branch_url(branch=parsed_url.branch, path=path, revision=parsed_url.revision)
        elif parsed_url.tag:
            arcadia_url = Arcadia.tag_url(tag=parsed_url.tag, path=path, revision=parsed_url.revision)

    return sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True)


def GetArcadia(arcadia_url, path=None, fuse=True):
    if fuse and sdk.fuse_available():
        return GetArcadiaAAPI(arcadia_url, path)

    if not path:
        return LocalMountPoint(Arcadia.get_arcadia_src_dir(arcadia_url))

    #  XXX selective checkout
    counter = 2
    arcadia_dst = 'arcadia'
    arcadia_dst_dir = arcadia_dst
    while os.path.exists(arcadia_dst_dir) and counter < 100:
        arcadia_dst_dir = arcadia_dst + str(counter)
        counter += 1
    assert counter < 100, "Can't get arcadia_dst_dir"

    arcadia_dst_dir = os.path.realpath(arcadia_dst_dir)
    Arcadia.checkout(arcadia_url, arcadia_dst_dir, depth="empty")
    Arcadia.update(pj(arcadia_dst_dir, '.arcadia.root'))

    paths = path if isinstance(path, (tuple, list)) else [path]
    for path in paths:
        path_abs = pj(arcadia_dst_dir, path)
        logging.debug("Selective checkout %s", path_abs)
        Arcadia.update(path_abs, parents=True)

    return LocalMountPoint(arcadia_dst_dir)


#  XXX copy-paste from quality/user_sessions/reactor/us_processes/util.py
def get_arcadia_root(start):
    found_dir = start

    while found_dir != '/' and not os.path.exists(pj(found_dir, '.arcadia.root')):
        found_dir = os.path.dirname(found_dir)

    if found_dir == '/':
        return None

    return found_dir


class CommitResult(object):
    def __init__(self):
        self.output = ''
        self.review_url = None
        self.review_id = None
        self.revision = None
        self.tests_status = None
        self.auto_merge = None

    def parse_commit(self, output):
        self.output = output

        if not output:
            #  No svn diff, already committed
            self.revision = -1
            return True

        revision_match = re.search(r'Committed revision (\d+)', output)
        if revision_match:
            self.revision = revision_match.group(1)
            return True

        review_match = re.search(r'https?://a.yandex-team.ru/review/(\d+)', output)
        assert review_match, "Can't parse review url from\n{}".format(output.replace('\n', '\n\t'))

        self.review_url = review_match.group(0)
        self.review_id = review_match.group(1)

    def parse_pr(self, output):
        assert self.review_url, "No review url"

        self.output = output
        self.tests_status = None

        if "Automatic merging is enabled" not in output:
            if self.auto_merge:
                raise TaskFailure("Automerge for PR was disabled")
            else:
                raise Exception("No automerge for PR {}".format(self.review_url))
        else:
            self.auto_merge = True

        merge_match = re.search(r'merged as (\d+)', output)
        if merge_match:
            self.revision = merge_match.group(1)
            return True

        tests_match = re.search(r'Merge requirements:\s*\n(.+?)\n\n', output, re.S)
        assert tests_match, "Can't parse tests status:\n{}".format(output.replace('\n', '\n\t'))

        self.tests_status = tests_match.group(1)

    def status(self):
        if re.search(r'Pull-request.*\(closed\)', self.output):
            return "CLOSED"
        elif "In progress" in self.tests_status:
            return "IN_PROGRESS"
        elif "Failed" in self.tests_status:
            return "FAILED"
        elif "Success" in self.tests_status:
            return "SUCCESS"
        else:
            return "UNKNOWN"


def make_commit(task, paths, message, user='zomb-sandbox-rw', need_review=True, auto_commit=True, author=None, wait_commit=True, root_dir=None, ya_token=None):
    if not author:
        author = task.author

    message += "\n\nTask {} by {}\n{}".format(task.type, author, lb.task_link(task.id, plain=True))

    if not isinstance(paths, (list, tuple)):
        paths = [paths]

    revprop = []
    revprop.append('arcanum:review-reviewers=' + ','.join(list(set([author] + PR_WATCHERS))))

    if auto_commit and need_review:
        message += "\nSKIP_REVIEW"  # ignore NEED_REVIEW directive if exists
    elif auto_commit and not need_review:
        revprop.append('arcanum:check-skip=yes')
        message += "\nSKIP_REVIEW"  # ignore NEED_REVIEW directive if exists
    elif not auto_commit and need_review:
        revprop.extend(["arcanum:review=new",
                        "arcanum:review-publish=yes",
                        "arcanum:review-automerge=yes"])
    elif not auto_commit and not need_review:
        raise Exception('any option is required to be set: need_review, auto_commit')

    commit = CommitResult()

    try:
        commit_output = Arcadia.commit(paths, message, user=user, with_revprop=revprop)
    except SvnError as e:
        commit_output = str(e)

    logging.debug(commit_output)
    if commit.parse_commit(commit_output):
        return commit

    if not wait_commit:
        return commit

    logging.info('Wait commit to be merged')

    source_dir = paths[0]
    if root_dir:
        arcadia_root = root_dir
    else:
        arcadia_root = get_arcadia_root(source_dir)
        assert arcadia_root, "Can't detect arcadia root from {}".format(source_dir)

    ya_tool = pj(arcadia_root, 'ya')
    assert os.path.exists(ya_tool), "Not exists: {}".format(ya_tool)
    ya_pr_st = [sdk._python(), ya_tool, 'pr', 'status', '-v', '-i', commit.review_id]

    WAIT_SECONDS = 60 * 60.0
    MAX_RETRIES = 1000

    retry = 1
    duration = None
    t_start = time.time()

    env = os.environ.copy()
    env['YA_TOKEN'] = ya_token

    while retry < MAX_RETRIES and (not duration or duration < WAIT_SECONDS) and not commit.revision:
        logging.info("make_commit[#%d]: duration: %s", retry, duration)
        logging.debug(str(ya_pr_st))

        t_end = time.time()
        output = None

        try:
            proc = subprocess.Popen(ya_pr_st, cwd=source_dir, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env)
            out, err = proc.communicate()
            output = out + err
        except subprocess.CalledProcessError as cpe:
            logging.debug("PR failed:\n{}".format(cpe.output))

        if output:
            logging.debug(output)
            if commit.parse_pr(output):
                break

            if commit.status() in ['FAILED', 'CLOSED']:
                raise TaskFailure("PR {} status {}".format(commit.review_url, commit.status()))

            if duration is not None or commit.status() == 'IN_PROGRESS':
                duration = t_end - t_start

        retry += 1
        time.sleep(20)

    assert commit.revision, "PR {} was not merged. Status: {}".format(commit.review_url, commit.status())

    return commit


def copy_and_add(from_path, to_path, include=None):
    if not os.path.exists(to_path):
        logging.debug("Make directory %s", to_path)
        os.mkdir(to_path)

    for p, dirs, files in os.walk(from_path, topdown=True):
        dst = os.path.normpath(os.path.join(to_path, os.path.relpath(p, from_path)))
        dirs[:] = [d for d in dirs if not d.startswith('.')]

        for d in dirs:
            src_path = os.path.join(p, d)
            dst_path = os.path.join(dst, d)
            if not os.path.exists(dst_path):
                logging.debug("Make directory %s", os.path.join(p, d))
                os.mkdir(dst_path)

        files[:] = [f for f in files if not f.startswith('.')]
        if include:
            files = include(p, files)

        for f in files:
            src_path = os.path.join(p, f)
            dst_path = os.path.join(dst, f)
            logging.debug("Copy file %s to %s", src_path, dst_path)
            shutil.copyfile(src_path, dst_path)


def prune_empty_dirs(path):
    items = map(lambda x: os.path.join(path, x), os.listdir(path))
    if not items:
        logging.debug('pruned empty dir: ' + path)
        os.rmdir(path)
        return True

    dirs = filter(os.path.isdir, items)
    pruned = 0

    for d in dirs:
        if prune_empty_dirs(d):
            pruned += 1

    if len(items) != len(dirs):
        return False
    elif pruned == len(dirs):
        logging.debug('pruned empty dir: ' + path)
        os.rmdir(path)
        return True

    return False


def svn_safe_sync(path):
    status = Arcadia.status(path)
    added, removed = 0, 0

    for status_line in status.splitlines():
        if status_line.startswith('?'):
            filename = status_line.split().pop()
            Arcadia.add(filename)
            added += 1
        elif status_line.startswith('!'):
            filename = status_line.split().pop()
            Arcadia.delete(filename)
            removed += 1

    return status, added, removed


def RunProcess(cmd, env, log_prefix=None, exception_if_nonzero_code=True):
    cmd_str = ' '.join([str(cmd_elem) for cmd_elem in cmd])
    # NOTE: don't add outs_to_pipe=True,
    # as it will create deadlock in communicate for some reason
    process = run_process(
        cmd_str,
        check=False, shell=True, wait=True,
        close_fds=True,
        environment=env,
        log_prefix=log_prefix,
    )

    process.communicate()

    def read(path):
        if path is not None:
            with open(path, 'r') as file_:
                return file_.read().strip()
        else:
            return ""

    stdout = read(process.stdout_path)
    stderr = read(process.stderr_path)

    if stderr:
        logging.info(stderr)

    if exception_if_nonzero_code and process.returncode != 0:
        raise Exception(stderr)

    return stdout, stderr


def RunProcesses(commands, env, silent=False):
    stdouts = []

    for cmd in commands:
        stdout, stderr = RunProcess(
            cmd, env,
            exception_if_nonzero_code=not silent
        )

        stdouts.append(stdout)

    return stdouts


def RunSimultaneousProcessesWithFiles(commands, default_env, max_processes_count, silent=False, env_getter=None, log_prefix=None, backoff_time=None):
    files = [None] * len(commands)
    running_cmds_inds = []
    processes = []
    done = 0
    while len(running_cmds_inds) > 0 or done < len(commands):
        while len(running_cmds_inds) < max_processes_count and done < len(commands):
            cmd = commands[done]
            cmd_str = ' '.join(cmd)
            process = run_process(cmd_str, outs_to_pipe=False, check=False, shell=True, wait=False,
                                  log_prefix="{}_{}".format(log_prefix, done),
                                  environment=default_env if env_getter is None else env_getter(cmd))
            processes.append(process)
            running_cmds_inds.append(done)
            done += 1
            if backoff_time:
                time.sleep(backoff_time)

        for cmd_ind in running_cmds_inds:
            process_ind = cmd_ind
            process = processes[process_ind]
            if process.poll() is not None:
                process.communicate()
                if process.returncode != 0 and not silent:
                    logging.info('error running command ' + ' '.join(commands[cmd_ind]))
                    error_file_path = process.stderr_path

                    error = None
                    if error_file_path is not None:
                        with open(error_file_path, 'r') as error_file:
                            error = error_file.read().strip()
                    else:
                        error = "Return code is {}, but stderr is empty".format(process.returncode)

                    raise Exception(error)

                files[cmd_ind] = process.stdout_path
                running_cmds_inds.remove(cmd_ind)

    return files


def RunSimultaneousProcesses(commands, default_env, max_processes_count, silent=False, results_callable=None, env_getter=None, backoff_time=None):
    results = [None] * len(commands)
    running_cmds_inds = []
    processes = []
    done = 0
    while len(running_cmds_inds) > 0 or done < len(commands):
        while len(running_cmds_inds) < max_processes_count and done < len(commands):
            cmd = commands[done]
            cmd_str = ' '.join(cmd)
            process = run_process(cmd_str, outs_to_pipe=True, check=False, shell=True, wait=False,
                                  environment=default_env if env_getter is None else env_getter(cmd))
            processes.append(process)
            running_cmds_inds.append(done)
            done += 1
            if backoff_time:
                time.sleep(backoff_time)

        for cmd_ind in running_cmds_inds:
            process_ind = cmd_ind
            process = processes[process_ind]
            if process.poll() is not None:
                result, error = process.communicate()
                if process.returncode != 0 and not silent:
                    logging.info('error running command ' + ' '.join(commands[cmd_ind]))
                    raise Exception(error)

                if results_callable is not None:
                    results_callable(commands[cmd_ind], result.strip())

                results[cmd_ind] = result.strip()
                running_cmds_inds.remove(cmd_ind)

    return results


def GetYTLogName(log, need_checkout=True):
    context = NullContextmanager(enter_obj="")
    if need_checkout:
        context = sdk.mount_arc_path(
            "arcadia:/arc/trunk/arcadia/quality/user_sessions/reactor/", use_arc_instead_of_aapi=True
        )

    with context as arc_path:
        if arc_path:
            sys.path.append(str(arc_path))

        # В следующей строке комментарий "# noqa" нужен, чтобы сандбокс не искал ошибки в импортруемом модуле
        from us_processes import yamr2yt_log_names_converter  # noqa

        return yamr2yt_log_names_converter.getYTLogName(log)


class YtPathsManager(object):
    def __init__(self, fetch_path, suffix_from_params, need_checkout=True):
        self.fetch_path = fetch_path
        self.suffix_from_params = suffix_from_params
        self.need_checkout = need_checkout

    def CreateSessionsWorkdir(self):
        return self.fetch_path

    def CreateSessionsSuffix(self):
        return self.suffix_from_params

    def LogTablePath(self, log):
        return MRPathJoin(self.fetch_path, 'raw_logs', GetYTLogName(log, self.need_checkout))

    def SessionDirPathWithoutSessionsSuffix(self, log):
        return MRPathJoin(self.fetch_path, 'user_sessions/build/logs', GetYTLogName(log, self.need_checkout), '1d')

    def SessionPath(self, log):
        return MRPathJoin(self.SessionDirPathWithoutSessionsSuffix(log), self.CreateSessionsSuffix(), 'hashed')

    def ValidationResultPath(self, log):
        return MRPathJoin(self.fetch_path, 'user_sessions/build/logs', GetYTLogName(log, self.need_checkout), '1d', self.CreateSessionsSuffix(), 'validation_result')
