import os
import logging
from functools import partial

from sandbox import sdk2
import sandbox.projects.common.constants as consts
from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk.task import SandboxTask

import sandbox.projects.logs.resources as resources
import sandbox.projects.geobase.Geodata6BinXurmaStable.resource as geo_resource


CUSTOM_ARCADIA_URL = "custom_arcadia_url"
GENERAL_ARCADIA_URL = "general_arcadia_url"
DO_NOT_BUILD = "do_not_build"
DEFAULT_GENERAL_METHOD = "default_general_method"


class BinarySpec(object):
    def __init__(self, resource_type, regular_release=True):
        assert resource_type.executable, "{} is not executable".format(resource_type)
        self.name = resources.get_binary_name(resource_type)
        self.target = os.path.dirname(resource_type.arcadia_build_path)
        self.resource_type = resource_type
        self.regular_release = regular_release


REACTOR_BINARIES_LIST = [
    BinarySpec(resources.CHECK_SESSIONS_EXISTENCE_EXECUTABLE),
    BinarySpec(resources.COMMON_LOGS_JOINER_EXECUTABLE),
    BinarySpec(resources.CREATE_NANO_SESSIONS_EXECUTABLE),
    BinarySpec(resources.CREATE_SCARAB_SESSIONS_EXECUTABLE),
    BinarySpec(resources.CREATE_SESSIONS_EXECUTABLE),
    BinarySpec(resources.ENTITY_RECOMMENDER_ACCESS2SCARAB_EXECUTABLE),
    BinarySpec(resources.EXTRACT_TAKEOUT_EXECUTABLE),
    BinarySpec(resources.FILTER_BS_DSP_LOG_EXECUTABLE),
    BinarySpec(resources.FILTER_BS_RTB_LOG_EXECUTABLE),
    BinarySpec(resources.HTTP_ADAPTER_ACCESS2SCARAB_EXECUTABLE),
    BinarySpec(resources.IMGREQANS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.IMG_REQANS_FILTER_MERGER),
    BinarySpec(resources.JOINED_BS_DSP_AND_BS_RTB_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.LOGSNG2SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_ACCESS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_APP_METRICA_EXECUTABLE),
    BinarySpec(resources.MARKET_ALICE_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_ORDERS_BILLED_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_ORDERS_PLACED_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CPA_CLICKS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_DELIVERED_ORDERS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_FEATURE_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_ORDERS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_RECOMMEND_CLICK_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_RECOMMEND_PREVIEW_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_REPORT_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_SHOWS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.RALIB_RULES_EXECUTABLE),
    BinarySpec(resources.SPLIT_IMGREQANS_EXECUTABLE),
    BinarySpec(resources.SPLIT_REQANS_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_ACTIVITY_JOIN_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_SESSION_ACTIVITY_EXTRACTION_EXECUTABLE),
    BinarySpec(resources.SURPLUS_METRICS_EXECUTABLE, regular_release=False),
    BinarySpec(resources.TYPE_PARSE_LIB_ERROR_EXECUTABLE, regular_release=False),
    BinarySpec(resources.UNIQUE_WATCH_LOG_EXECUTABLE),
    BinarySpec(resources.USER_SESSIONS_SAMPLE_BY_UID_EXECUTABLE),
    BinarySpec(resources.MERGE_SESSIONS_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_SESIONS_URL_EXTRACTION_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_SESIONS_URL_FILTER_EXECUTABLE),
    BinarySpec(resources.SPLIT_RT_SESSIONS_EXECUTABLE),
    BinarySpec(resources.SESSIONS_TTL_PROPS_REMOVER_EXECUTABLE),
]

REM_BINARIES_LIST = [
    BinarySpec(resources.COMMON_LOGS_JOINER_EXECUTABLE),
    BinarySpec(resources.CREATE_SCARAB_SESSIONS_EXECUTABLE),
    BinarySpec(resources.CREATE_SESSIONS_EXECUTABLE),
    BinarySpec(resources.CREATE_NANO_SESSIONS_EXECUTABLE),
    BinarySpec(resources.ERROR_STATS_EXECUTABLE),
    BinarySpec(resources.FILTER_BS_DSP_LOG_EXECUTABLE),
    BinarySpec(resources.FILTER_BS_RTB_LOG_EXECUTABLE),
    BinarySpec(resources.IMGREQANS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.JOINED_BS_DSP_AND_BS_RTB_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.LOGSNG2SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_ACCESS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_API_ACCESS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_APP_METRICA_EXECUTABLE),
    BinarySpec(resources.MARKET_BLUE_TABLE_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_ORDERS_BILLED_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_ORDERS_PLACED_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_ORDERS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_DELIVERED_ORDERS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CLICKS_VENDOR_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_CPA_CLICKS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_FEATURE_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_GOALS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_RECOMMEND_CLICK_ACTIVITY_EXECUTABLE),
    BinarySpec(resources.MARKET_RECOMMEND_CLICK_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_RECOMMEND_PREVIEW_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_REPORT_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.MARKET_SHOWS_TO_SCARAB_EXECUTABLE),
    BinarySpec(resources.SPLIT_IMGREQANS_EXECUTABLE),
    BinarySpec(resources.SPLIT_MARKET_ACCESS_EXECUTABLE),
    BinarySpec(resources.SPLIT_MARKET_CLICKS_EXECUTABLE),
    BinarySpec(resources.SPLIT_MARKET_CPA_CLICKS_EXECUTABLE),
    BinarySpec(resources.SPLIT_MARKET_REPORT_EXECUTABLE),
    BinarySpec(resources.SPLIT_MARKET_SHOWS_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_ACTIVITY_JOIN_EXECUTABLE),
    BinarySpec(resources.SPY_LOG_SESSION_ACTIVITY_EXTRACTION_EXECUTABLE),
    BinarySpec(resources.TYPE_PARSE_LIB_ERROR_EXECUTABLE),
    BinarySpec(resources.UNIQUE_WATCH_LOG_EXECUTABLE),
    BinarySpec(resources.WEATHER_ACCESS_TO_SCARAB_EXECUTABLE),
]


REACTOR_BINARIES = {bs.name: bs for bs in REACTOR_BINARIES_LIST}
REM_BINARIES = {bs.name: bs for bs in REM_BINARIES_LIST}


class ParamBinaryBuildPolicy(sdk2.Parameters):
    with sdk2.parameters.String("How to build {binary}") as how_to_build_binary:
        how_to_build_binary.values[DEFAULT_GENERAL_METHOD] = how_to_build_binary.Value(DEFAULT_GENERAL_METHOD, default=True)
        how_to_build_binary.values[CUSTOM_ARCADIA_URL] = how_to_build_binary.Value(CUSTOM_ARCADIA_URL)
        how_to_build_binary.values[DO_NOT_BUILD] = how_to_build_binary.Value(DO_NOT_BUILD)

    with how_to_build_binary.value[CUSTOM_ARCADIA_URL]:
        arcadia_url_for = sdk2.parameters.ArcadiaUrl('svn url for {binary}', required=True)


class ParamBinarySpecsRegular(sdk2.Parameters):
    for name in sorted(REACTOR_BINARIES.keys()):
        if REACTOR_BINARIES[name].regular_release:
            sdk2.helpers.set_parameter('ext_' + name,
                                       ParamBinaryBuildPolicy(suffix='_' + name,
                                                              label_substs={'binary': name}))


class ParamBinarySpecsNonRegular(sdk2.Parameters):
    for name in sorted(REACTOR_BINARIES.keys()):
        if not REACTOR_BINARIES[name].regular_release:
            sdk2.helpers.set_parameter('ext_' + name,
                                       ParamBinaryBuildPolicy(suffix='_' + name,
                                                              label_substs={'binary': name}))


class FileSpec(object):
    def __init__(self, production_server_path, resource_type, ttl_days_callable, archive_name_if_dir=None, get_last_from_sandbox=False):
        self.production_server_path = production_server_path
        self.resource_type = resource_type
        self.ttl_days_callable = ttl_days_callable
        self.archive_name_if_dir = archive_name_if_dir
        self.get_last_from_sandbox = get_last_from_sandbox

    def get_final_file_name(self, file_name):
        return self.archive_name_if_dir or file_name


def getTTLDays(fetchFilePolicy, intDaysBetweenReleases):
    from us_processes.file_artifact_spec import FileFetchPolicy
    if fetchFilePolicy == FileFetchPolicy.LAST:
        return intDaysBetweenReleases + 20
    elif fetchFilePolicy == FileFetchPolicy.TRIGGERED:
        return intDaysBetweenReleases + 25
    else:
        raise Exception("Unknown fetch file policy: {}!".format(fetchFilePolicy))


FILE_NAME_TO_SPEC = {
    "user_type_resolver": FileSpec("/Berkanavt/geodata", resources.SESSIONS_USER_TYPE_RESOLVER, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "geohelper": FileSpec("/Berkanavt/geodata", resources.SESSIONS_GEOHELPER, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "blockstat.dict": FileSpec("/Berkanavt/statdata", resources.SESSIONS_BLOCKSTAT, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "blockstat.dict.json": FileSpec("/Berkanavt/statdata", resources.SESSIONS_BLOCKSTAT_JSON, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "direct_pageids": FileSpec("/Berkanavt/statdata", resources.SESSIONS_DIRECT_PAGEIDS, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "direct_ads_descriptions": FileSpec("/Berkanavt/statdata", resources.SESSIONS_DIRECT_ADS_DESCRIPTIONS,
                                        ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "rtb_pageids": FileSpec("/Berkanavt/statdata", resources.SESSIONS_RTB_PAGEIDS, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "direct_resourceno_dict": FileSpec("/Berkanavt/statdata", resources.SESSIONS_DIRECT_RESOURCENO_DICT, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "beta_list.txt": FileSpec("/Berkanavt/quality_data", resources.SESSIONS_BETA_LIST, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1)),
    "geodata6-xurma.bin": FileSpec("", geo_resource.GEODATA6BIN_XURMA_STABLE, ttl_days_callable=partial(getTTLDays, intDaysBetweenReleases=1), get_last_from_sandbox=True),
}


def escape_file_parameter(file_name):
    return file_name.replace(".", "_").replace("-", "_")


def get_file_parameter(file_name):
    return "{}_param".format(escape_file_parameter(file_name))


class ParamFileReleasePolicy(sdk2.Parameters):
    _param = sdk2.parameters.Bool("Should we release {file_name}")


class ParamFileReleaseSpec(sdk2.Parameters):
    for file_name in sorted(FILE_NAME_TO_SPEC.keys()):
        param_prefix = escape_file_parameter(file_name)
        sdk2.helpers.set_parameter('ext_' + param_prefix,
                                   ParamFileReleasePolicy(prefix=param_prefix,
                                                          label_substs={'file_name': file_name}))


def get_binary_from_svn_path(bin_name, res_type, parent_sandbox_task, build_from_svn_url, attrs_for_fetching_ready_bin=None, patch=None):
    resource = None
    if attrs_for_fetching_ready_bin:
        logging.info("TRYING TO FIND PREVIOUSLY BUILD %s BINARY WITH ATTRS: %s" % (res_type, attrs_for_fetching_ready_bin))
        resource = apihelpers.get_last_resource_with_attrs(res_type, attrs_for_fetching_ready_bin, all_attrs=True)

    if resource:
        res_id = str(resource.id)
        logging.info("FOUND %s RESOURCE WITH ID: %s" % (res_type, res_id))
        return (res_id, None)

    else:
        logging.info("NO PREV RES WITH ATTRS:" + str(attrs_for_fetching_ready_bin))
        logging.info("BUILDING %s BIN FOR : %s" % (res_type, build_from_svn_url))

        params = {
            consts.ARCADIA_URL_KEY: build_from_svn_url,
            'build_' + bin_name: True,
            'use_obj_cache': True,
            'arcadia_patch': patch or parent_sandbox_task.ctx.get('arcadia_patch'),
            'build_system': 'semi_distbuild',
            'use_aapi_fuse': True,
            'fallback_aapi': True,
        }

        subtask = SandboxTask.create_subtask(
            parent_sandbox_task,
            task_type='BUILD_SEARCH',
            description='Building %s binary' % res_type,
            input_parameters=params,
        )

        return (None, subtask.id)
