import os
import re
from sandbox import sdk2
import shutil

import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.error_handlers import check_failed
from sandbox.projects.logs.common import make_commit

import logging


class DeployAmonPython(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        upload_pypi = sdk2.parameters.Bool(
            'Upload to PyPI',
            required=True,
            default_value=True
        )
        commit_changes = sdk2.parameters.Bool(
            'Commit changes to Arcadia',
            required=True,
            default_value=True
        )
        increment_tiny_version = sdk2.parameters.Bool(
            'Increment PyAmon tiny version',
            required=True,
            default_value=True
        )
        with increment_tiny_version.value[False]:
            major_version = sdk2.parameters.Integer(
                'PyAmon new major version',
                required=True,
                default_value=0
            )
            minor_version = sdk2.parameters.Integer(
                'PyAmon new minor version',
                required=True,
                default_value=0
            )
            tiny_version = sdk2.parameters.Integer(
                'PyAmon new tiny version',
                required=True,
                default_value=0
            )

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    def on_execute(self):
        logging.info('pyamon checkout...')
        pyamon_svn_url = SandboxArcadiaUrlParameter.default_value + '/quality/user_sessions/py_amon@HEAD'
        pyamon_svn_dir = Arcadia.get_arcadia_src_dir(pyamon_svn_url)

        version = self.update_version(pyamon_svn_dir)
        revision = Arcadia.info(pyamon_svn_dir)['entry_revision']
        comment = "Amon {} library version {} built from revision {}".format('Python',
                                                                             version,
                                                                             revision)


        self.log_status(pyamon_svn_dir)

        if self.Parameters.upload_pypi:
            pypirc_svn_url = SandboxArcadiaUrlParameter.default_value + '/scarab/static/python3@HEAD'
            pypirc_svn_dir = Arcadia.get_arcadia_src_dir(pypirc_svn_url)
            self.upload_to_pypi(pyamon_svn_dir, pypirc_svn_dir)

        if self.Parameters.commit_changes:
            result = make_commit(self, pyamon_svn_dir, comment, need_review=False)
            if result.revision:
                if result.revision == -1:
                    self.set_info("No changes")
                else:
                    rev = result.revision
                    self.set_info("OK, committed as <a href='https://a.yandex-team.ru/arc/commit/{}'>{}</a>".format(rev, rev), do_escape=False)
            elif result.review_url:
                self.set_info("Created pull-request: <a href='{}'>{}</a>".format(result.review_url, result.review_url), do_escape=False)
            else:
                raise Exception("FAILED: {}".format(result.output))
        else:
            loggin.info("COMMIT MESSAGE=" + comment)

        logging.info('Done!')

    def upload_to_pypi(self, pyamon_dir, pypirc_dir):
        logging.info('Uploading to PyPi repository https://pypi.yandex-team.ru/simple/.')

        pypirc = os.path.join(os.path.expanduser('~'), '.pypirc')
        scarab_pypirc = os.path.join(pypirc_dir, '.pypirc')
        pypirc_bak = None

        logging.info('Copy user .pypirc and replace it with scarab python .pypirc')

        if os.path.isfile(pypirc):
            pypirc_bak = os.path.join(str(self.path()), 'pypirc.bak')
            shutil.move(pypirc, pypirc_bak)
        shutil.copy(scarab_pypirc, pypirc)

        run_process(
            ['python', 'setup.py', 'sdist', 'upload', '-r', 'yandex'],
            work_dir=pyamon_dir,
            log_prefix='PyPI'
        )

        if pypirc_bak is not None:
            logging.info('Restore user .pypirc')
            os.unlink(pypirc)
            shutil.move(pypirc_bak, pypirc)

        logging.info('Check PyPI status')
        with open(get_logs_folder() + '/PyPI.out.txt') as pypi_result:
            for line in pypi_result:
                if line.startswith('Upload failed'):
                    check_failed(line + '\t see PyPI.out.txt log')

    def update_version(self, pyamon_dir):
        logging.debug("Update version")

        old_setup_path = pyamon_dir + '/setup.py'
        new_setup_path = pyamon_dir + '/newsetup'

        with open(old_setup_path) as setup, open(new_setup_path, "w") as newsetup:
            for line in setup:
                m = re.match(r"(\s+)version='(\d+)\.(\d+)\.(\d+)(.*)'(.*)\n", line)
                if m:
                    indent, major, minor, tiny, suffix, rest = m.group(1, 2, 3, 4, 5, 6)

                    new_major = int(major)
                    new_minor = int(minor)
                    new_tiny = int(tiny)

                    if self.Parameters.increment_tiny_version:
                        new_tiny += 1
                    else:
                        new_major = self.Parameters.major_version
                        new_minor = self.Parameters.minor_version
                        new_tiny = self.Parameters.tiny_version

                    logging.debug('Old version is {}.{}.{}{}. Insert new version {}.{}.{}'.format(
                        major, minor, tiny, rest,
                        new_major, new_minor, new_tiny
                    ))
                    newsetup.write("{}version='{}.{}.{}'{}\n".format(indent, new_major, new_minor, new_tiny, rest))
                else:
                    newsetup.write(line)
        os.unlink(old_setup_path)
        os.rename(new_setup_path, old_setup_path)
        return '.'.join([str(new_major), str(new_minor), str(new_tiny)])

    def log_status(self, pyamon_dir):
        status = Arcadia.status(pyamon_dir)
        logging.info('ARCADIA STATUS...\n%s', status)
        run_process(['svn', 'diff'], work_dir=pyamon_dir, log_prefix='svn_diff')
        run_process(['svn', 'st'], work_dir=pyamon_dir, log_prefix='svn_st')
