import sandbox.common.types.misc as ctm
import os
import re
import shutil
import logging

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import SvnEnvironment
import sandbox.sandboxsdk.svn as svn

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.logs.common import make_commit


class TargetParameter(SandboxArcadiaUrlParameter):
    name = 'target'
    description = 'Arcadia path'
    required = True


class PublishPackages(SandboxBoolParameter):
    name = 'publish_packages'
    description = 'Publish packages to PyPi and Ubuntu repositories'
    required = True
    default_value = True


class DeployBaobabPythonApi(SandboxTask):
    type = "DEPLOY_BAOBAB_PYTHON_API"

    environment = [SvnEnvironment()]

    input_parameters = [TargetParameter, PublishPackages]

    # ask for IPv4
    dns = ctm.DnsType.DNS64

    def set_published_revision(self, svn_stdout):
        if 'published_revisions' not in self.ctx:
            self.ctx['published_revisions'] = []

        m = re.search(r'\s+(\d+)\b', svn_stdout)
        assert m, 'Cant parse committed svn revision from: {}'.format(svn_stdout)

        published_revision = int(m.group(1))
        self.ctx['published_revisions'].append(published_revision)

        return published_revision

    def on_execute(self):
        baobab_svn_url = SandboxArcadiaUrlParameter.default_value + '/quality/logs/baobab/api/python@HEAD'
        baobab_svn_dir = Arcadia.get_arcadia_src_dir(baobab_svn_url)
        status = Arcadia.status(baobab_svn_dir)
        logging.info('ARCADIA STATUS...\n%s', status)

        self.revision = 'HEAD'

        path = svn.Arcadia.parse_url(self.ctx['target']).path + '/quality/logs/baobab/api/python@HEAD'
        svn.Arcadia.checkout('arcadia:' + path, path=self.path(), revision=self.revision)

        baobab_version = None
        logging.debug("Increment version")
        with open(self.path() + '/setup.py') as setup, open(self.path() + "/newsetup", "w") as newsetup:
            for line in setup:
                m = re.match(r'(\s+)version="(\d+)\.(\d+)\.(\d+)"(.*)\n', line)
                if m:
                    indent, major, minor, tiny, rest = m.group(1, 2, 3, 4, 5)
                    tiny = int(tiny)
                    logging.debug("Old version is {}. Insert new version {}".format(tiny, tiny+1))
                    baobab_version = "{}.{}.{}".format(major, minor, tiny+1)
                    newsetup.write('{}version="{}"{}\n'.format(indent, baobab_version, rest))
                else:
                    newsetup.write(line)

        os.unlink(self.path() + '/setup.py')
        os.rename(self.path() + "/newsetup", self.path() + '/setup.py')

        self.ctx['published_version'] = baobab_version

        logging.info('publish packages:%s', self.ctx.get(PublishPackages.name))

        if self.ctx.get(PublishPackages.name):
            logging.info("Uploading to PyPi repository https://pypi.yandex-team.ru/simple/.")
            pypirc = os.path.join(os.path.expanduser('~'), '.pypirc')
            pypirc_my = os.path.join(baobab_svn_dir, '.pypirc')
            pypirc_bak = None
            if os.path.isfile(pypirc):
                pypirc_bak = os.path.join(self.path(), 'pypirc.bak')
                shutil.move(pypirc, pypirc_bak)
            shutil.copy(pypirc_my, pypirc)

            run_process(
                ['python', 'setup.py', 'sdist', 'upload', '-r', 'yandex'],
                work_dir=self.path(),
                log_prefix='PyPI'
            )

            if pypirc_bak is not None:
                logging.info('Restore user .pypirc')
                os.unlink(pypirc)
                shutil.move(pypirc_bak, pypirc)

            logging.info('Check PyPI status')
            with open(get_logs_folder() + '/PyPI.out.txt') as pypi_result:
                for line in pypi_result:
                    if line.startswith('Upload failed'):
                        eh.check_failed(line + '\t see PyPI.out.txt log')

            comment = "Baobab Python API version {}".format(baobab_version)
            result = make_commit(self, [baobab_svn_dir, self.path()], comment)
            if result.revision:
                if result.revision == -1:
                    self.set_info("No changes")
                else:
                    rev = result.revision
                    self.set_info("OK, committed as <a href='https://a.yandex-team.ru/arc/commit/{}'>{}</a>".format(rev, rev), do_escape=False)
            elif result.review_url:
                self.set_info("Created pull-request: <a href='{}'>{}</a>".format(result.review_url, result.review_url), do_escape=False)
            else:
                raise Exception("FAILED: {}".format(result.output))

        logging.info("Done!")
