# -*- coding: utf-8 -*-
import os
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.release_machine.input_params2 as rm_params

RULE_MONITORING_PATH = 'quality/user_sessions/ralib_rules'
YQL_TOKEN_OWNER = 'USERSESSIONSTOOLS'
YQL_TOKEN_NAME = 'userdata-sessions-build-ci-token'


class UserSessionsRuleMonitoringBinary(sdk2.Resource):
    """
        Rule monitoring bin
    """
    releasable = False
    any_arch = True
    executable = True
    auto_backup = True
    branch = sdk2.parameters.String()
    parent_task_id = sdk2.parameters.String()

class UserSessionsRuleMonitoringResult(sdk2.Resource):
    """
        File with rule monitoring result
    """
    releasable = False
    any_arch = True
    executable = False
    auto_backup = True
    branch = sdk2.parameters.String()
    table = sdk2.parameters.String()
    parent_task_id = sdk2.parameters.String()


class RunRuleMonitoring(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('yandex-yt', version='0.10.8')]
        cores = 1
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(rm_params.ComponentName2):
        with sdk2.parameters.RadioGroup("Cluster name") as cluster:
            cluster.values.hahn = cluster.Value("hahn", default=True)
            cluster.values.arnold = cluster.Value("arnold")
        input_table_path = sdk2.parameters.String(
            'input_table_path',
            description='Path for input table',
            multiline=False,
            required=True
        )
        rule_monitoring_path = sdk2.parameters.ArcadiaUrl('Arcadia url for rule monitoring', required=True)

    def clone_and_build(self, arcadia_url, api, result_dir):
        with sdk.mount_arc_path(arcadia_url,
                                use_arc_instead_of_aapi=True) as arcadia_dir:
            sdk.do_build(
                build_system=consts.YMAKE_BUILD_SYSTEM,
                source_root=arcadia_dir,
                targets=[api],
                build_type=consts.RELEASE_BUILD_TYPE,
                results_dir=str(result_dir),
                clear_build=False,
                use_dev_version=False,
                sanitize=False
            )

    def prepare_result_file_as_resource(self, path):
        sdk2.ResourceData(UserSessionsRuleMonitoringResult(
            self,
            "Rule monitoring output",
            str(path),
            branch=self.Parameters.rule_monitoring_path,
            parent_task_id=self.id,
            table=self.Parameters.input_table_path,
        ))

    def prepare_bin_as_resource(self, path):
        sdk2.ResourceData(UserSessionsRuleMonitoringBinary(
            self,
            "Rule monitoring bin",
            str(path),
            branch=self.Parameters.rule_monitoring_path,
            parent_task_id=self.id,
        ))

    def search_bin(self):
        resource = UserSessionsRuleMonitoringBinary.find(attrs=dict(branch=self.Parameters.rule_monitoring_path)).first()
        if resource:
            try:
               resource_str = str(sdk2.ResourceData(resource).path)
               self.Context.use_bin_from_res = True
               return resource_str
            except:
                return None
        return None

    def search_prepared_rm_result(self):
        resource = UserSessionsRuleMonitoringResult.find(attrs=dict(
            branch = self.Parameters.rule_monitoring_path,
            table = self.Parameters.input_table_path,
        )).first()
        if resource:
            self.Context.use_prepared_rm_res = True
            return str(sdk2.ResourceData(resource).path)
        return None

    def on_execute(self):
        prep_res = self.search_prepared_rm_result()
        if not prep_res:
            token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
            os.environ["YT_TOKEN"] = token
            result_path = self.path('builded')
            work_dir = str(self.path())
            rule_monitoring_bin_path = self.search_bin()
            if not rule_monitoring_bin_path:
                work_dir = str(result_path) + '/' + RULE_MONITORING_PATH
                self.clone_and_build(self.Parameters.rule_monitoring_path, RULE_MONITORING_PATH, result_path)
                rule_monitoring_bin_path = str(work_dir)+'/ralib_rules'
            run_process(
                [rule_monitoring_bin_path,
                '-i', self.Parameters.input_table_path,
                '-s', self.Parameters.cluster,
                '-o', str(self.path('res.txt')),
                '-p  //home/userdata-sessions-build-ci/rule_monitoring/{}/samples'.format(self.id)],
                work_dir=work_dir,
                shell=True,
                timeout=10000,
                log_prefix='run_rule_monitoring',
                outputs_to_one_file=True
            )
            if not self.Context.use_bin_from_res:
                self.prepare_bin_as_resource(str(work_dir)+'/ralib_rules')
            self.prepare_result_file_as_resource(self.path('res.txt'))

