from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process  # maybe sdk2.helpers.ProcessLog?
from sandbox.projects.common import error_handlers as eh
from sandbox.common.types import task as ctt
from sandbox.projects.common import apihelpers

from sandbox.projects import resource_types as rst
from sandbox.projects.common.utils import set_resource_attributes

from sandbox import common
import json

import sandbox.projects.common.constants as consts

import logging

from os.path import join as pj
import os
import shutil

from sandbox.sandboxsdk import environments
from sandbox.projects.common.yql import run_query

import datetime


TSAR_PATH_DIRNAME = "//home/bs/logs/AdsJoinedProfilesLog/1h"
HOUR_TABLE_NAME_FORMAT = "%Y-%m-%dT%H:%M:%S"
DATE_FORMAT = "%Y-%m-%d"

USUAL = "usual"
TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK = "take_dates_and_yql_progress_from_other_task"


def to_date_datetime(dt_str):
    return datetime.datetime.strptime(dt_str, DATE_FORMAT)


def to_date_datetime_str(dt_dt):
    return datetime.datetime.strftime(dt_dt, DATE_FORMAT)


def to_hour_datetime_str(dt_dt):
    return datetime.datetime.strftime(dt_dt, HOUR_TABLE_NAME_FORMAT)


def RunProcess(cmd, env, log_prefix=None, exception_if_nonzero_code=True):
    cmd_str = ' '.join([str(cmd_elem) for cmd_elem in cmd])
    process = run_process(cmd_str,
                          wait=True,
                          outs_to_pipe=False,
                          close_fds=True,
                          check=False,
                          shell=True,
                          log_prefix=log_prefix,
                          environment=env
                          )
    process.communicate()
    if process.returncode != 0:
        exc_msg = ''

        error_file_path = process.stderr_path
        if error_file_path is not None:
            error = None
            with open(error_file_path, 'r') as error_file:
                error = '\n'.join(error_file.readlines()[-30:])
            exc_msg += 'error: {}\n'.format(error)

        result_file_path = process.stdout_path
        if result_file_path is not None:
            result = None
            with open(result_file_path, 'r') as result_file:
                result = '\n'.join(result_file.readlines()[-30:])
            exc_msg += 'result: {}\n'.format(result)

        if exception_if_nonzero_code:
            raise Exception(exc_msg)
        else:
            return exc_msg

    return ""


class LtpProfilePriemka(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
        ]

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String("Mode", required=True) as mode:
            mode.values[USUAL] = USUAL
            mode.values[TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK] = TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK

        with mode.value[TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK]:
            task_to_take_dates_and_yql_prepare_results = sdk2.parameters.Task(
                "task_to_take_dates_and_yql_prepare_results. For neshtat-case: task from which we can "
                " take dates and results of yql", required=True)

        with mode.value[USUAL]:
            approximate_datetime_str = sdk2.parameters.String("Approximate datetime for LtpProfile in format YYYY-MM-DD. "
                                                              "Final datetime might differ - the task corrects it and puts final into the context.", required=True)
            how_many_15day_packs_back_we_should_recalc = sdk2.parameters.Integer(
                "Will be multiplied with 15 to get quantity of days to calc back. The less days the faster it will be."
                " Recommended default: 6", default=6)

        logos_release_id_prefix = sdk2.parameters.String("logos_release_id_prefix. Datetime-suffix will be added. Reactor path will be "
                                                         " /logos/graphs/ads/dev/<login>/<logos_release_id_prefix><suf>)", required=True)
        login = sdk2.parameters.String("login. Logos and Yt user login: will be used for forming logos-token-secret-name in nirvana-vault <login>_logos_token. And also will be used for forming "
                                       " working directory on Yt //home/ads/logos/dev/<login>/<logos_release_id_prefix><suf>", required=True)
        yt_pool = sdk2.parameters.String("Yt pool. <login> has to have access to it. If not specified then default pool will be used", default="")
        nirvana_quota = sdk2.parameters.String("Nirvana quota. <login> has to have access to it. If not specified then default one will be used", default="")
        arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia url for binaries (must be with @<rev>). For example: arcadia:/arc/trunk/arcadia@<rev>', required=True)
        arcadia_patch = sdk2.parameters.String("Arcadia patch", default="")

        keep_reactor_graph = sdk2.parameters.Bool("Keep reactor logos dev graph", default=False)

        with sdk2.parameters.Group("Logos token - has to be the token of <login>") as logos_token_block:
            logos_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with logos token", required=True)
            logos_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with logos token", required=True)

        with sdk2.parameters.Group("Yt token - any token able to read from //home/ads/logos/dev/<login> and from {}".format(TSAR_PATH_DIRNAME)) as yt_token_block:
            yt_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with yt token", required=True)
            yt_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with yt token", required=True)

        with sdk2.parameters.Group("Yql pool and token. The token has to have access to the pool. And to be able to read from "
                                   " //home/bs/logs/LtpProfileWithOldVectors/1d, //home/bs/logs/LtpProfileBuildState/1d, {} and to write to "
                                   " //home/ads/logos/dev/<login>".format(TSAR_PATH_DIRNAME)) as yql_token_block:
            yql_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with yql token", required=True)
            yql_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with yql token", required=True)
            yt_pool_for_yql_prepare = sdk2.parameters.String("yt_pool_for_yql_prepare. If not specified then default pool will be used", default="")

    class Context(sdk2.Context):
        cluster = "hahn"
        initialized = False
        prepared = False
        build_logos_subtask_id = None
        logos_bin_res_id = None
        first_tsar_datetime_shift = 0
        tsar_days_count = 8
        logos_bin_dir = "logos/projects/ads/graph/bin"
        logos_launch_happened = False

    def GetEnv(self, need_logos_user_config=False):
        env = dict(os.environ)
        env['YT_TOKEN_PATH'] = self.yt_token_path

        secret_owner = self.Parameters.logos_token_secret_owner
        secret_name = self.Parameters.logos_token_secret_name
        env['LOGOS_TOKEN'] = sdk2.Vault.data(secret_owner, secret_name)

        if need_logos_user_config:
            env['LOGOS_USER_CUSTOM_CONFIG'] = "step={},sample_uids_table={}".format(15, self.Context.yt_tsar_uids)

        YTSpec = {
            "job_io": {
                "table_writer": {
                    "max_row_weight": 128 * 1024 * 1024
                }
            }
        }
        env['YT_SPEC'] = json.dumps(YTSpec)

        return env

    def get_yql_query_start(self):
        query = "PRAGMA yt.StaticPool=\"{}\";\n\n".format(self.Parameters.yt_pool_for_yql_prepare) if self.Parameters.yt_pool_for_yql_prepare else ""
        query += "PRAGMA yt.AutoMerge = \"disabled\";\n\n"
        return query

    def get_fetch_tsar_uids_query(self):
        query = self.get_yql_query_start()

        if self.Parameters.mode == TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK:
            prev_task = self.Parameters.task_to_take_dates_and_yql_prepare_results
            assert self.Context.first_tsar_date == prev_task.Context.first_tsar_date
            assert self.Context.last_tsar_date == prev_task.Context.last_tsar_date

            query += """
            insert into `{out}` with truncate
            from `{prev_out}`
            select *
            assume order by key;
            """.format(out=self.Context.yt_tsar_uids, prev_out=prev_task.Context.yt_tsar_uids)
        else:
            query += """
            insert into `{out}` with truncate
            from (
                from range(`{src_base}`, `{src_first}`, `{src_last}`)
                select distinct UniqID as key
            )
            select "y" || cast(key as string) as key
            order by key;

            """.format(out=self.Context.yt_tsar_uids, src_base=TSAR_PATH_DIRNAME, src_first=self.Context.tsar_src_path_first_hour_basename,
                       src_last=self.Context.tsar_src_path_last_hour_basename)

        return query

    def get_sample_profiles_query(self):
        query = self.get_yql_query_start()

        if self.Parameters.mode == TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK:
            prev_task = self.Parameters.task_to_take_dates_and_yql_prepare_results
            assert self.Context.start_datetime_to_build_test_profile == prev_task.Context.start_datetime_to_build_test_profile

            query += """
            insert into `{ltp_prod_new_date_sample}` with truncate
            from `{prev_ltp_prod_new_date_sample}`
            select *
            assume order by key;

            insert into `{ltp_prod_old_date_sample}` with truncate
            from `{prev_ltp_prod_old_date_sample}`
            select *
            assume order by key;

            insert into `{ltp_prod_build_old_date_sample}` with truncate
            from `{prev_ltp_prod_build_old_date_sample}`
            select *
            assume order by key;
            """.format(
                ltp_prod_new_date_sample=self.Context.yt_prod_profile_new_date_sample,
                ltp_prod_old_date_sample=self.Context.yt_prod_profile_start_date_sample,
                ltp_prod_build_old_date_sample=self.Context.yt_prod_build_profile_start_date_sample,
                prev_ltp_prod_new_date_sample=prev_task.Context.yt_prod_profile_new_date_sample,
                prev_ltp_prod_old_date_sample=prev_task.Context.yt_prod_profile_start_date_sample,
                prev_ltp_prod_build_old_date_sample=prev_task.Context.yt_prod_build_profile_start_date_sample,
            )
        else:
            query += """
            insert into `{ltp_prod_new_date_sample}` with truncate
            from `{ltp_prod_new_date}` as ltp left semi join `{tsar_uids}` as tsar_uids using(key)
            select *
            assume order by key;

            insert into `{ltp_prod_old_date_sample}` with truncate
            from `{ltp_prod_old_date}` as ltp left semi join `{tsar_uids}` as tsar_uids using(key)
            select *
            assume order by key;

            insert into `{ltp_prod_build_old_date_sample}` with truncate
            from `{ltp_prod_build_old_date}` as ltp left semi join `{tsar_uids}` as tsar_uids using(key)
            select *
            assume order by key;

            """.format(
                ltp_prod_new_date_sample=self.Context.yt_prod_profile_new_date_sample,
                ltp_prod_new_date=self.Context.yt_prod_profile_new_date,
                ltp_prod_old_date_sample=self.Context.yt_prod_profile_start_date_sample,
                ltp_prod_old_date=self.Context.yt_prod_profile_start_date,
                ltp_prod_build_old_date_sample=self.Context.yt_prod_build_profile_start_date_sample,
                ltp_prod_build_old_date=self.Context.yt_prod_build_profile_start_date,
                tsar_uids=self.Context.yt_tsar_uids,
            )

        return query

    def CreateBuildSubtask(self):
        subtask_type = sdk2.Task["YA_MAKE"]

        params = {
                'arch': 'linux',
                'checkout_arcadia_from_url': self.Parameters.arcadia_url,
                'arcadia_patch': self.Parameters.arcadia_patch,
                'targets': self.Context.logos_bin_dir,
                'arts': pj(self.Context.logos_bin_dir, "logos_tool"),
                'result_rt': "ARCADIA_PROJECT",
                'build_system': 'semi_distbuild',
                'build_type': 'release',
                'use_aapi_fuse': True,
                'aapi_fallback': True,
                'check_return_code': True,
                'result_single_file': True,
                'result_ttl': '30',
                'build_output_ttl': 1,
                'build_output_html_ttl': 1,
                'allure_report_ttl': 1,
                consts.STRIP_BINARIES: True,
        }

        subtask = subtask_type(self, description='Building logos_too binary', **params)
        sdk2.Task.server.task[subtask.id].update({'requirements': {'disk_space': 80737418240, 'ram': 4096}})
        subtask.enqueue()
        return subtask

    def get_15_nearest(self, dt_dt):
        date_defining_15_grid = "2021-07-01"
        dt_dt_from_15_grid = to_date_datetime(date_defining_15_grid)
        if dt_dt > dt_dt_from_15_grid:
            remainder = (dt_dt - dt_dt_from_15_grid).days % 15
            if remainder <= 7:
                return dt_dt - datetime.timedelta(days=remainder)
            else:
                return dt_dt + datetime.timedelta(days=15 - remainder)
        else:
            remainder = (dt_dt_from_15_grid - dt_dt).days % 15
            if remainder <= 7:
                return dt_dt + datetime.timedelta(days=remainder)
            else:
                return dt_dt - datetime.timedelta(days=15 - remainder)

    def get_back_if_needed_to_have_future_tsar(self, dt_dt):
        import yt.wrapper as yt

        table_names = set(yt.list(TSAR_PATH_DIRNAME))

        while True:
            first_tsar_cand = dt_dt + datetime.timedelta(days=self.Context.first_tsar_datetime_shift)
            last_tsar_cand = first_tsar_cand + datetime.timedelta(days=self.Context.tsar_days_count - 1, hours=23)

            bad = False
            cur_tsar_cand = first_tsar_cand
            while cur_tsar_cand <= last_tsar_cand:
                if to_hour_datetime_str(cur_tsar_cand) not in table_names:
                    bad = True
                    break
                cur_tsar_cand += datetime.timedelta(hours=1)

            if bad:
                dt_dt -= datetime.timedelta(days=15)
                continue
            else:
                return dt_dt

    def Initialize(self):
        self.Context.logos_release_id_prefix = self.Parameters.logos_release_id_prefix
        self.Context.arcadia_url = self.Parameters.arcadia_url
        self.Context.arcadia_patch = self.Parameters.arcadia_patch if self.Parameters.arcadia_patch else None
        self.Context.revision = self.Parameters.arcadia_url.rsplit('@', 1)[-1]
        self.Context.step = 15
        self.Context.now_from_logos_release_id = to_hour_datetime_str(datetime.datetime.now())
        self.Context.logos_release_id = "{}_{}_r{}_step{}".format(self.Parameters.logos_release_id_prefix, self.Context.now_from_logos_release_id, self.Context.revision, self.Context.step)
        self.Context.reactor_task_path = "/logos/graphs/ads/dev/{}/{}/tasks/{}/LtpProfileTask".format(self.Parameters.login, self.Context.logos_release_id, self.Context.cluster)

        self.Context.yt_working_dir = "//home/ads/logos/dev/{}/{}".format(self.Parameters.login, self.Context.logos_release_id)

        if self.Parameters.mode == TAKE_DATES_AND_YQL_PROGRESS_FROM_OTHER_TASK:
            prev_task = self.Parameters.task_to_take_dates_and_yql_prepare_results
            dt_dt = to_date_datetime(prev_task.Context.ltp_profile_datetime)
            self.Context.how_many_15day_packs_back_we_should_recalc = prev_task.Context.how_many_15day_packs_back_we_should_recalc
        else:
            dt_dt = to_date_datetime(self.Parameters.approximate_datetime_str)
            dt_dt = self.get_15_nearest(dt_dt)
            dt_dt = self.get_back_if_needed_to_have_future_tsar(dt_dt)
            self.Context.how_many_15day_packs_back_we_should_recalc = self.Parameters.how_many_15day_packs_back_we_should_recalc

        self.Context.test_profile_new_data_days = 15 * self.Context.how_many_15day_packs_back_we_should_recalc

        self.Context.ltp_profile_datetime = to_date_datetime_str(dt_dt)
        self.Context.start_datetime_to_build_test_profile = to_date_datetime_str(dt_dt - datetime.timedelta(days=self.Context.test_profile_new_data_days))
        self.Context.yt_prod_profile_new_date = "//home/bs/logs/LtpProfileWithOldVectors/1d/{}".format(self.Context.ltp_profile_datetime)
        self.Context.yt_prod_profile_start_date = "//home/bs/logs/LtpProfileWithOldVectors/1d/{}".format(self.Context.start_datetime_to_build_test_profile)
        self.Context.yt_prod_build_profile_start_date = "//home/bs/logs/LtpProfileBuildState/1d/{}".format(self.Context.start_datetime_to_build_test_profile)

        self.Context.yt_test_profile_new_date_sample = os.path.join(self.Context.yt_working_dir, self.Context.yt_prod_profile_new_date.strip('/'))
        self.Context.yt_prod_profile_new_date_sample = self.Context.yt_test_profile_new_date_sample + "_PROD_SAMPLE"

        self.Context.yt_prod_profile_start_date_sample = os.path.join(self.Context.yt_working_dir, self.Context.yt_prod_profile_start_date.strip('/'))
        self.Context.yt_prod_build_profile_start_date_sample = os.path.join(self.Context.yt_working_dir, self.Context.yt_prod_build_profile_start_date.strip('/'))

        self.Context.first_tsar_date = to_date_datetime_str(to_date_datetime(self.Context.ltp_profile_datetime) + datetime.timedelta(days=self.Context.first_tsar_datetime_shift))
        self.Context.last_tsar_date = to_date_datetime_str(to_date_datetime(self.Context.first_tsar_date) + datetime.timedelta(days=self.Context.tsar_days_count - 1))
        self.Context.tsar_src_path_first_hour_basename = to_hour_datetime_str(to_date_datetime(self.Context.first_tsar_date))
        self.Context.tsar_src_path_last_hour_basename = to_hour_datetime_str(to_date_datetime(self.Context.last_tsar_date) + datetime.timedelta(hours=23))

        self.Context.yt_tsar_uids = os.path.join(self.Context.yt_working_dir, "tsar_uids", "{}_{}".format(self.Context.first_tsar_date, self.Context.last_tsar_date))

        # TODO validate and fail if some out paths already exist?

    def Prepare(self):
        self.Context.save()  # dump context
        fetch_tsar_uids_query = self.get_fetch_tsar_uids_query()
        logging.info("fetch tsar uids - started")
        run_query("fetch_tsar_uids_query", fetch_tsar_uids_query, sdk2.Vault.data(self.Parameters.yql_token_secret_owner, self.Parameters.yql_token_secret_name), self.Context.cluster)
        logging.info("fetch tsar uids - successfull")

        sample_profiles_query = self.get_sample_profiles_query()
        logging.info("sample profiles - started")
        run_query("sample_profiles", sample_profiles_query, sdk2.Vault.data(self.Parameters.yql_token_secret_owner, self.Parameters.yql_token_secret_name), self.Context.cluster)
        logging.info("sample profiles - successfull")

    def DoSyncBinary(self, res_id, bin_name):
        resource = sdk2.Resource.find(id=res_id).limit(1).first()
        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        p = sdk2.path.Path(resource_path)
        new_resource_path = pj(os.path.dirname(str(p)), bin_name)
        shutil.move(str(p), new_resource_path)
        return new_resource_path

    def ConfigureYtTokenAndCluster(self):
        secret_owner = self.Parameters.yt_token_secret_owner
        secret_name = self.Parameters.yt_token_secret_name
        yt_token = sdk2.Vault.data(secret_owner, secret_name)

        import yt.wrapper as yt
        yt.config["token"] = yt_token
        yt.config["proxy"]["url"] = self.Context.cluster
        self.yt_token_path = pj(str(self.path()), 'yt_token_file')
        os.system('echo {} > {}'.format(yt_token, self.yt_token_path))

    def GetBinResourceAttrDict(self):
        return {
            "target": self.Context.logos_bin_dir,
            "build_url": self.Context.arcadia_url,

            # "None" is fine. We can't use None and can't omit key "arcadia_patch" - in this cases we might have problems with correct fetching from cache
            "arcadia_patch": str(self.Context.arcadia_patch),
        }

    def TryFindCachedBin(self):
        resource = apihelpers.get_last_resource_with_attrs(rst.ARCADIA_PROJECT, self.GetBinResourceAttrDict(), all_attrs=True)
        if resource:
            return resource.id
        else:
            return None

    def on_execute(self):
        logging.info('LtpProfilePriemkaTask: Start')

        self.ConfigureYtTokenAndCluster()

        if not self.Context.initialized:
            self.Initialize()
            self.Context.initialized = True  # has to be set after and not before Initialize() in case Initialize raises exception

        if self.Context.logos_bin_res_id is None and self.Context.build_logos_subtask_id is None:
            self.Context.logos_bin_res_id = self.TryFindCachedBin()
        if self.Context.logos_bin_res_id is None:
            if self.Context.build_logos_subtask_id is None:
                subtask = self.CreateBuildSubtask()
                self.Context.build_logos_subtask_id = subtask.id

        if not self.Context.prepared:
            self.Prepare()
            self.Context.prepared = True  # has to be set after and not before Prepare() in case Prepare raises exception

        if self.Context.logos_bin_res_id is None:
            subtask = sdk2.Task[self.Context.build_logos_subtask_id]
            if subtask.status not in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
                raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
            if subtask.status in (ctt.Status.FAILURE, ctt.Status.EXCEPTION):
                self.Context.build_logos_subtask_id = None
                raise common.errors.TaskError("logos build subtask failed - its status is {}. You may restart me (the priemka-task) and I will create new build subtask".format(str(subtask.status)))
            res_id = subtask.Context.ap_packs["project"]
            set_resource_attributes(int(res_id), self.GetBinResourceAttrDict())
            self.Context.logos_bin_res_id = res_id

        bin_local_path = self.DoSyncBinary(self.Context.logos_bin_res_id, "logos_tool")

        last_date = to_hour_datetime_str(to_date_datetime(self.Context.ltp_profile_datetime))
        # TODO get rid of wait-localy? or not
        cmd = [
            bin_local_path, " autorun-tasks  --subgraph LtpProfileTask --tasks LtpProfileTask  --release-id {release_id}  --nirvana-secret-name {login}_logos_token --user {login}"
            " --first-date {first_date} --last-date {last_date}  {yt_pool_option} --no-diff --step 15 --lookup-inputs self {nirvana_quota_option} --prod-inputs "
            " LtpSearchClicks={first_date_for_logs}..{last_date_for_logs}   AdsUserSessionClickedDocs={first_date_for_logs}..{last_date_for_logs} "
            " AdsUserSessionQueries={first_date_for_logs}..{last_date_for_logs}   AdsUserSessionMiscEvents={first_date_for_logs}..{last_date_for_logs} --wait-localy".format(
                release_id=self.Context.logos_release_id,
                login=self.Parameters.login,
                first_date=to_hour_datetime_str(to_date_datetime(self.Context.start_datetime_to_build_test_profile) + datetime.timedelta(days=15)),
                last_date=last_date,
                yt_pool_option="--yt-pool {}".format(self.Parameters.yt_pool) if self.Parameters.yt_pool else "",
                nirvana_quota_option="--nirvana-quota {}".format(self.Parameters.nirvana_quota) if self.Parameters.nirvana_quota else "",
                first_date_for_logs=to_hour_datetime_str(to_date_datetime(self.Context.start_datetime_to_build_test_profile) + datetime.timedelta(days=1)),
                last_date_for_logs=last_date,
            )]

        # TODO Maybe we should not wait for it to finish and instead we should regularly check whether the final table exists. I mean we should pass wait=False and poll. But the task
        # has to be executing anyway!!!
        errorMsg = ""
        if not self.Context.logos_launch_happened:
            self.Context.logos_launch_happened = True
            self.Context.save()  # dump context
            errorMsg = RunProcess(cmd, self.GetEnv(True), log_prefix="run_logos", exception_if_nonzero_code=False)

        import yt.wrapper as yt
        if not yt.exists(self.Context.yt_test_profile_new_date_sample):
            errorMsg += "\n\n\nFinal table {} doesn't exist! Maybe it's still being built. "
            " You might check in reactor - {}. \n\n\n That's why your logos graph in reactor hasn't been deleted, do it yourself when it's not needed anymore".format(
                self.Context.yt_test_profile_new_date_sample, self.Context.reactor_task_path)
        else:
            if not self.Parameters.keep_reactor_graph:
                cmd = [bin_local_path,
                       " delete-release --release-id {release_id} --nirvana-secret-name {login}_logos_token".format(release_id=self.Context.logos_release_id, login=self.Parameters.login)]
                deleteErrorMsg = RunProcess(cmd, self.GetEnv(), log_prefix="delete_logos_graph", exception_if_nonzero_code=False)
                if deleteErrorMsg:
                    errorMsg += "FINAL TABLE EXISTS AND ONLY DELETION OF REACTOR-LOGOS-GRAPH FAILED: {}".format(deleteErrorMsg)

        if errorMsg:
            eh.check_failed(errorMsg)
