# -*- coding: utf-8 -*-


from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
import smtplib

import datetime
import logging
import time

from os.path import join as pj
import sys


class NotifyOnUsDelay(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('networkx', version='2.2', use_wheel=True),
        ]

    class Parameters(sdk2.Parameters):
        pass

    class Context(sdk2.Context):
        pass

    def GetSvnUrl(self):
        # in this task all we take is from trunk.  Because we need the freshest info and tools - we do not release anything
        return "arcadia:/arc/trunk/arcadia"

    def CheckoutArcadiaSubfolder(self, arcadia_subfolder, use_cache=True):
        svn_url = self.GetSvnUrl()

        dir_url = pj(svn_url, arcadia_subfolder)

        arcadia_subfolder_local_abs_path = pj(self.arcadia_src_dir, arcadia_subfolder)

        if use_cache:
            with arcadia_sdk.mount_arc_path(dir_url, use_arc_instead_of_aapi=True) as p:
                sdk2.paths.copy_path(str(p), arcadia_subfolder_local_abs_path)
        else:
            sdk2.svn.Arcadia.checkout(dir_url, arcadia_subfolder_local_abs_path)

        return arcadia_subfolder_local_abs_path

    def PrepareArcadia(self):
        self.arcadia_src_dir = pj(str(self.path()), 'local_arcadia')  # TODO check abs_path() - maybe no have it

        self.CheckoutArcadiaSubfolder('quality/user_sessions/reactor')
        self.arcadia_user_sessions_local_abs_path = pj(self.arcadia_src_dir, 'quality/user_sessions/reactor')
        sys.path.append(self.arcadia_user_sessions_local_abs_path)
        # further we'll use this code through 'import us_reactor'

    def send_email(self, server, subject, msg_text):
        from_addr = 'logs-team@yandex-team.ru'
        to_addr = 'logs-team-internal@yandex-team.ru'
        recipients = ['logs-tech@yandex-team.ru', 'gavrgavr@yandex-team.ru']

        msg = MIMEMultipart()
        msg['Subject'] = subject
        msg['From'] = from_addr
        msg['To'] = to_addr
        msg['Cc'] = ', '.join(recipients)

        msg.attach(MIMEText(msg_text))

        server.sendmail(from_addr, [to_addr] + recipients, msg.as_string())

    def do_notify_on_email(self, text):
        DATE_FORMAT = "%Y-%m-%d"

        server = smtplib.SMTP('yabacks.yandex.ru', port=25)
        subject = "[{}] Произошла задержка построения Поисковых сессий".format(datetime.datetime.strftime(self.dateTime, DATE_FORMAT))

        self.send_email(server, subject, text)
        server.quit()

    def bad_notify_on_email(self):
        text = ("ВНИМАНИЕ!!!\n"
        "\n"
        "Произошла задержка построения дневных Поисковых сессий\n"
        "В связи с этим будет создан тикет https://st.yandex-team.ru/USOPS\n"
        "Проведется расследование и будут приняты меры по устранению подобных проблем в будущем.\n"
        "Приносим свои извинения за доставленные неудобства.\n"
        "\n"
        "С уважением, команда Пользовательских сессий\n"
        "logs-team@yandex-team.ru")

        self.do_notify_on_email(text)

    def good_notify_on_email(self):
        text = ("Сессии построены!!!\n"
        "С уважением, команда Logs\n"
        "logs-team@yandex-team.ru")

        self.do_notify_on_email(text)

    def GetReactorToken(self):
        secret_owner = "USERSESSIONSTOOLS"
        secret_name = "prod_reactor_token"
        secret_content = sdk2.Vault.data(secret_owner, secret_name)

        return secret_content

    def IsSessionsReady(self):
        from us_reactor.lib import model
        from us_reactor.lib.client import ReactorAPIClient
        from us_processes.constants import REACTOR_SERVER
        from us_processes.time_util import _convert_to_unixtime, datetime_round

        reactor_client = ReactorAPIClient(REACTOR_SERVER, self.GetReactorToken())

        start_datetime_dt = datetime.datetime.now()
        period = datetime.timedelta(days=1)
        self.dateTime = datetime_round(start_datetime_dt - period, period)
        usertime_ts = _convert_to_unixtime(self.dateTime)
        art_insts = reactor_client.get_artifact_range(namespace=model.Namespace.from_namespace_path("/user_sessions/hahn/pub/search/1d"), limit=100,
                                                      from_user_ts=usertime_ts, to_user_ts=usertime_ts + 1, statuses=["ACTIVE"])["range"]
        return bool(art_insts)

    def on_execute(self):
        logging.info('Task: Start')

        self.PrepareArcadia()

        if self.IsSessionsReady():
            if self.Context.bad_notify_done == True:
                self.good_notify_on_email()
            return
        else:
            if (self.Context.bad_notify_done == ctm.NotExists) or (self.Context.bad_notify_done is False):
                self.bad_notify_on_email()
                self.Context.bad_notify_done = True
            raise sdk2.WaitTime(15 * 60)  # 15min
