# -*- coding: utf-8 -*-

import logging
import requests

from datetime import datetime, timedelta
from simplejson.decoder import JSONDecodeError

from sandbox import sdk2

from sandbox.common.errors import TaskFailure


class CalendarApi(object):
    API_URL = 'http://calendar-api.tools.yandex.net/internal'
    API_TIME_FMT = '%Y-%m-%dT%H:%M:%S'

    def _get_request(self, operation, params=None, headers=None, api_timeout=120.0, exception=TaskFailure):
        # prepare request
        url = self.API_URL + operation
        if headers is None:
            headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip, identity'
            }
            if hasattr(self, 'token'):
                headers['Authorization'] = 'OAuth {}'.format(self.token)

        # make request
        try:
            r = requests.get(url, params=params, headers=headers, timeout=api_timeout)
            r.raise_for_status()
        except requests.exceptions.Timeout:
            msg = 'API is silent'
            raise exception(msg)
        except requests.exceptions.RequestException as err:
            raise exception(err.message)

        # dump result
        try:
            return r.json()
        except JSONDecodeError:
            raise exception('Unable to parse json answer')

    def _get_event(self, params):
        operation = '/get-events'
        return self._get_request(operation, params)

    def _find_longest_event(self, events):
        res_idx, max_duration = None, timedelta(seconds=0)
        for idx, event in enumerate(events):
            start, end = event['startTs'], event['endTs']
            duration = datetime.strptime(end, self.API_TIME_FMT) - datetime.strptime(start, self.API_TIME_FMT)
            if duration > max_duration:
                max_duration = duration
                res_idx = idx
        return res_idx

    def get_duty(self, calendar_id):
        logging.debug("Get duty login from {} calendar".format(calendar_id))
        now = datetime.now()
        finish_duty_time = now.replace(day=now.day-now.weekday(), hour=6, minute=30)
        params = {
            'from': finish_duty_time.strftime(self.API_TIME_FMT),
            'to': finish_duty_time.strftime(self.API_TIME_FMT),
            'tz': 'Europe/Moscow',
            'layerId': calendar_id
        }
        body = self._get_event(params)
        if 'events' not in body or len(body['events']) < 1:
            return None

        events = body['events']
        if len(events) > 1:
            idx = self._find_longest_event(events)
            event = events[idx]
        else:
            event = events[0]

        if 'attendees' not in event or len(event['attendees']) < 1:
            return None
        return event['attendees'][0]['login']


class UserSessionsDutyHelper(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 360
        calendar_id = sdk2.parameters.Integer("Calendar id with duty schedule",
                                              required=True)
        duty_login = sdk2.parameters.String("Duty login (empty for getting from calendar)", required=False)

    def on_execute(self):
        duty_login = CalendarApi().get_duty(self.Parameters.calendar_id)

        logging.info(str(duty_login))
        self.Context.duty_login = duty_login
