from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.logs.common import ReactorPathJoin, RunProcess
from sandbox.projects.logs.common.reactor import RegisterReactorFiles
from sandbox.projects.logs.common.binaries import (
    FILE_NAME_TO_SPEC,
    ParamFileReleaseSpec,
    get_file_parameter,
)

import json
import logging
from sandbox import common
import time
from os.path import join as pj
import os
import sys
import shutil


PRODUCTION_SERVER = 'veles02.search.yandex.net'
DEFAULT_NIRVANA_QUOTA_NAME = "user-sessions"


class ReleaseUsFilesToNirvana(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment('requests'),
            PipEnvironment('networkx', version='2.2', use_wheel=True),
        ]

    class Parameters(sdk2.Parameters):
        ext_file_release_spec = ParamFileReleaseSpec()

        with sdk2.parameters.Group("Reactor token") as reactor_token_block:
            reactor_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with reactor token", required=True)  # on practice: USERSESSIONSTOOLS. But do not specify default here for safety
            reactor_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with reactor token", required=True)  # on practice: robot-make-sessions-reactor-token (TODO: create secret). But do not specify default here for safety
            # TODO: maybe add defaults at least to description?

        with sdk2.parameters.Group("Nirvana token") as nirvana_token_block:
            nirvana_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with nirvana token", required=True)  # on practice: USERSESSIONSTOOLS. But do not specify default here for safety
            nirvana_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with nirvana token", required=True)  # on practice: robot-make-sessions-nirvana-token (TODO: create secret). But do not specify default here for safety
            # TODO: maybe add defaults at least to description?

        with sdk2.parameters.Group("Nirvana: other") as nirvana_other_block:
            nirvana_quota_name = sdk2.parameters.String("Quota name on nirvana: on practice we use: {}".format(DEFAULT_NIRVANA_QUOTA_NAME), required=True)

        with sdk2.parameters.Group("Debug") as debug:
            debug_reactor_prefix = sdk2.parameters.String("Debug reactor prefix", default="")

    class Context(sdk2.Context):
        pass

    def GetNirvanaTokenPath(self):
        secret_owner = self.Parameters.nirvana_token_secret_owner
        secret_name = self.Parameters.nirvana_token_secret_name
        secret_content = sdk2.Vault.data(secret_owner, secret_name)
        nirvana_token_path = pj(str(self.path()), 'nirvana_token_file')
        os.system('echo {} > {}'.format(secret_content, nirvana_token_path))

        return nirvana_token_path

    def GetReactorTokenPath(self):
        secret_owner = self.Parameters.reactor_token_secret_owner
        secret_name = self.Parameters.reactor_token_secret_name
        secret_content = sdk2.Vault.data(secret_owner, secret_name)
        reactor_token_path = pj(str(self.path()), 'reactor_token_file')
        os.system('echo {} > {}'.format(secret_content, reactor_token_path))

        return reactor_token_path

    def GetNirvanaResourceDescription(self, file_name, file_spec):
        data_json = {"file_name": file_name, "production_server": PRODUCTION_SERVER, "server_dir": file_spec.production_server_path}
        return json.dumps(data_json)

    def GetEnv(self):
        env = dict(os.environ)
        env['PYTHONPATH'] = ':'.join(sys.path)

        return env

    def DoSyncFileToNirvana(self, file_name, file_spec, local_path):
        from us_processes import file_artifact_spec

        fetchFilePolicy = file_artifact_spec.getFileArtifactSpec(file_name).fetchFilePolicy

        nirv_token_path = self.GetNirvanaTokenPath()
        ttl_days = file_spec.ttl_days_callable(fetchFilePolicy)
        description = self.GetNirvanaResourceDescription(file_name, file_spec)

        cmd = ["{} {}".format(sys.executable, pj(self.reactor_dir, "us_nirvana", "tools", "sync_file_to_nirvana.py")), "-t", nirv_token_path, "--ttl-days", ttl_days, "--description", "'{}'".format(description.replace("'", "\"")),
               "-p", local_path, "-q", self.Parameters.nirvana_quota_name]
        result, error = RunProcess(cmd, self.GetEnv(), log_prefix="sync_to_nirvana_{}".format(file_name), exception_if_nonzero_code=True)
        return result

    def CheckoutArcadiaSubfolder(self, arcadia_subfolder, use_cache=True):
        svn_url = Arcadia.ARCADIA_TRUNK_URL

        dir_url = pj(svn_url, arcadia_subfolder)

        arcadia_subfolder_local_abs_path = pj(self.arcadia_src_dir, arcadia_subfolder)

        if use_cache:
            with arcadia_sdk.mount_arc_path(dir_url, use_arc_instead_of_aapi=True) as p:
                sdk2.paths.copy_path(str(p), arcadia_subfolder_local_abs_path)
        else:
            sdk2.svn.Arcadia.checkout(dir_url, arcadia_subfolder_local_abs_path)

        return arcadia_subfolder_local_abs_path

    def PrepareArcadia(self):
        self.arcadia_src_dir = pj(str(self.path()), 'local_arcadia')
        self.reactor_dir = self.CheckoutArcadiaSubfolder('quality/user_sessions/reactor')
        sys.path.insert(0, self.reactor_dir)

    def GetArtifactNamespaceToStringMetaValue(self, file_name, final_file_name):
        from us_processes import file_artifact_spec

        artifact_to_release = file_artifact_spec.getFileArtifactSpec(final_file_name).artifactToRelease
        return {artifact_to_release: self.Context.file_name_to_nirvana_data_id[file_name]}

    def PrepareReactorReleaseConfig(self, file_name, final_file_name):
        import us_reactor.tools.do_release_string_artifacts_lib as config_lib

        artifact_namespace_to_string_meta_value = self.GetArtifactNamespaceToStringMetaValue(file_name, final_file_name)

        data_json = []
        for artifact_namespace, string_meta_value in artifact_namespace_to_string_meta_value.iteritems():
            data_json.append({
                config_lib.FROM_SVN: False,
                config_lib.ATTRS: {
                    "nirvana_data_id": self.Context.file_name_to_nirvana_data_id[file_name],
                },
                config_lib.NAMESPACE: ReactorPathJoin(self.Parameters.debug_reactor_prefix, artifact_namespace),
                config_lib.STRING_VALUE: string_meta_value,
            })

        return config_lib.serialize(data_json)

    def ReleaseToReactor(self, not_rsynced_file_to_err_msg, not_nirvana_synced_file_to_err_msg):
        from us_processes import file_artifact_spec

        reactor_token_path = self.GetReactorTokenPath()
        self.Context.file_name_to_config_content = {}

        not_released_file_to_err_msg = {}

        for file_name, file_spec in FILE_NAME_TO_SPEC.iteritems():
            if file_name in not_rsynced_file_to_err_msg or file_name in not_nirvana_synced_file_to_err_msg:
                continue

            final_file_name = file_spec.get_final_file_name(file_name)

            try:
                should_release = getattr(self.Parameters, get_file_parameter(file_name))
                if should_release:
                    config_content = self.PrepareReactorReleaseConfig(file_name, final_file_name)
                    self.Context.file_name_to_config_content[file_name] = config_content

                    config_path = pj(str(self.path()), 'do_release_config_file')
                    print >>open(config_path, 'w'), config_content

                    fetchFilePolicy = file_artifact_spec.getFileArtifactSpec(final_file_name).fetchFilePolicy
                    ttl_days = file_spec.ttl_days_callable(fetchFilePolicy)

                    if file_spec.get_last_from_sandbox:
                        file_spec.resource_type = None #Not release same resource to sandbox

                    self.DoRelease(file_name, reactor_token_path, config_path, rst_class_obj=file_spec.resource_type, sb_ttl_days=ttl_days)
            except Exception as e:
                not_released_file_to_err_msg[file_name] = e

        return not_released_file_to_err_msg

    def SyncToNirvana(self, not_rsynced_file_to_err_msg):
        not_nirvana_synced_file_to_err_msg = {}

        for file_name, file_spec in FILE_NAME_TO_SPEC.iteritems():
            if file_name in not_rsynced_file_to_err_msg:
                continue

            final_file_name = file_spec.get_final_file_name(file_name)

            try:
                should_release = getattr(self.Parameters, get_file_parameter(file_name))
                if should_release:
                    if file_spec.archive_name_if_dir:
                        local_full_path = pj(self.local_dir_for_files, file_name, file_spec.archive_name_if_dir)
                    else:
                        local_full_path = pj(self.local_dir_for_files, file_name)
                    if file_spec.get_last_from_sandbox:
                        resource = file_spec.resource_type.find(
                            attrs={"released": "stable"}
                        ).first()
                        resource_data = sdk2.ResourceData(resource)
                        shutil.copy(str(sdk2.path.Path(resource_data.path)), local_full_path)
                    result = self.DoSyncFileToNirvana(final_file_name, file_spec, local_full_path)
                    result_json = json.loads(result.strip())
                    self.Context.file_name_to_nirvana_data_id[file_name] = result_json["data_id"]
                    self.Context.file_name_to_nirvana_storage_id[file_name] = result_json["storage_id"]
            except Exception as e:
                not_nirvana_synced_file_to_err_msg[file_name] = str(e)

        return not_nirvana_synced_file_to_err_msg

    def RsyncFromProductionServer(self):
        not_rsynced_file_to_err_msg = {}

        for file_name, file_spec in FILE_NAME_TO_SPEC.iteritems():
            if file_spec.get_last_from_sandbox:
                continue
            try:
                should_release = getattr(self.Parameters, get_file_parameter(file_name))
                if should_release:
                    if file_spec.archive_name_if_dir:
                        cmd = ['rsync', '-r', 'rsync://' + PRODUCTION_SERVER + pj(file_spec.production_server_path, file_name), self.local_dir_for_files]
                    else:
                        cmd = ['rsync', 'rsync://' + PRODUCTION_SERVER + pj(file_spec.production_server_path, file_name), self.local_dir_for_files]

                    rsynced = False
                    tries = 0
                    while not rsynced and tries < 3:
                        result, error = RunProcess(cmd, self.GetEnv(), 'rsync_' + file_name)
                        if os.path.exists(pj(self.local_dir_for_files, file_name)):
                            rsynced = True
                        else:
                            time.sleep(10)

                        tries += 1

                    if not rsynced:
                        raise Exception("Rsync failed!")
                    if file_spec.archive_name_if_dir:
                        os.chdir(pj(self.local_dir_for_files, file_name))
                        cmd = ["tar -cf", file_spec.archive_name_if_dir, "*"]
                        result, error = RunProcess(cmd, self.GetEnv(), 'tar_' + file_name)

            except Exception as e:
                not_rsynced_file_to_err_msg[file_name] = str(e)

        return not_rsynced_file_to_err_msg

    def DoRegisterFiles(self):
        env = self.GetEnv()
        reactor_token_path = self.GetReactorTokenPath()
        return RegisterReactorFiles(file_name="all", token_path=reactor_token_path, env=env)

    def on_execute(self):
        logging.info('ReleaseUsFilesToNirvanaTask: Start')

        # Init
        self.Context.file_name_to_nirvana_data_id = {}
        self.Context.file_name_to_nirvana_storage_id = {}
        self.local_dir_for_files = str(self.path())
        # self.local_dir_for_files = pj(str(self.path()), 'local_dir_for_files')
        # os.makedirs(self.local_dir_for_files)
        #######################

        self.PrepareArcadia()
        logging.info('DoRegisterFiles result: {}'.format(str(self.DoRegisterFiles())))

        not_rsynced_file_to_err_msg = self.RsyncFromProductionServer()
        not_nirvana_synced_file_to_err_msg = self.SyncToNirvana(not_rsynced_file_to_err_msg)
        not_released_file_to_err_msg = self.ReleaseToReactor(not_rsynced_file_to_err_msg, not_nirvana_synced_file_to_err_msg)

        # Finishing
        if not_rsynced_file_to_err_msg or not_nirvana_synced_file_to_err_msg or not_released_file_to_err_msg:
            final_err_msg = ""

            final_err_msg += "These files didn't rsync:\n"
            for filename, msg in not_rsynced_file_to_err_msg.iteritems():
                final_err_msg += "    {}: {}\n".format(filename, msg)

            final_err_msg += "\n\n"
            final_err_msg += "These files didn't sync to nirvana:\n"
            for filename, msg in not_nirvana_synced_file_to_err_msg.iteritems():
                final_err_msg += "    {}: {}\n".format(filename, msg)

            final_err_msg += "\n\n"
            final_err_msg += "These files didn't release to reactor:\n"
            for filename, msg in not_released_file_to_err_msg.iteritems():
                final_err_msg += "    {}: {}\n".format(filename, msg)

            raise common.errors.TaskFailure(final_err_msg)
        ########################

    def DoRelease(self, file_name, reac_token_path, config_path, rst_class_obj, sb_ttl_days):
        cmd = [
           sys.executable,
           pj(self.reactor_dir, "us_reactor", "tools", "do_release_string_artifacts.py"),
           "-t", reac_token_path,
           "-c", config_path,
           "-s", "reactor.yandex-team.ru"
        ]

        result, error = RunProcess(cmd, self.GetEnv(), log_prefix="do_release_file_artifacts_{}".format(file_name))

        if rst_class_obj is not None:
            local_full_path = pj(self.local_dir_for_files, file_name)
            new_res = sdk2.ResourceData(rst_class_obj(
                self,
                "Output file {}".format(file_name),
                local_full_path,
                released="stable",
                ttl=sb_ttl_days,
            ))
            new_res.ready()
