import os
import os.path
import logging

import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.process import run_process
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.paths as sdk_paths

from sandbox.projects.logs.scarab.common import DeployScarabBase
from sandbox.projects.logs.scarab.config.deploy_android import providers

from sandbox.projects.common.build import parameters
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.environments import SandboxMavenEnvironment


class TargetParameter(sdk_parameters.SandboxStringParameter):
    name = 'target'
    description = 'Target path, related to arcadia'
    required = True


class BuildVersionParameter(sdk_parameters.SandboxStringParameter):
    name = 'version'
    description = 'Version number of build. Become part of deploy path.'
    required = True
    default_value = '1.1'


class BuildTargetParameter(sdk_parameters.SandboxRadioParameter):
    name = 'buildTarget'
    description = 'Target for build'
    choices = list((x, x) for x in providers)
    per_line = 1
    default_value = 'mobile-minimal'


class DryRun(sdk_parameters.SandboxBoolParameter):
    name = 'dry_run'
    description = 'dry run?'
    required = True
    default_value = False


class DeployScarabJavaApi(DeployScarabBase):
    type = "DEPLOY_SCARAB_JAVA_API"

    environment = DeployScarabBase.environment + (
        SandboxJavaJdkEnvironment('1.8.0'),
        SandboxMavenEnvironment('3.3.9'),
    )

    input_parameters = DeployScarabBase.input_parameters + [
        BuildVersionParameter,
        DryRun,
        BuildTargetParameter
    ]

    # ask for IPv4
    dns = ctm.DnsType.DNS64

    def on_execute(self):
        # acquire parameters
        version = self.ctx.get(BuildVersionParameter.name)
        provider = self.ctx.get(BuildTargetParameter.name)
        build_target = 'java-mobile-minimal/' + provider

        pom_suffix = '-mobile-minimal'
        suffix = '-' + provider

        # init variables
        self.test_and_generate(api=build_target, url=self.ctx.get(parameters.ArcadiaUrl.name), static_dir='scarab/static/java')

        # setup logging paths
        mvnlog_path = os.path.join(sdk_paths.get_logs_folder(), 'mvn.log')
        mvnerr_path = os.path.join(sdk_paths.get_logs_folder(), 'mvn.err')

        # compile and deploy
        mvn_cmd = ['mvn', 'clean', 'compile', 'jar:jar'] + (['deploy:deploy-file'] if not self.ctx.get(DryRun.name) else []) + [
            '-f', os.path.join(self.work_dir, 'static', 'pom%s.xml' % pom_suffix),
            '-Dscarab.artifact.suffix=%s' % suffix,
            '-Dscarab.version=%s.%s' % (version, self.revision),
            '-Dmaven.project.artifactId=scarab-',
            '-Dmaven.project.groupId=ru.yandex.se.log',
            '-Dmaven.project.name=scarab-',
            '-Dmaven.project.version=1.1.0',
            '--settings', os.path.join(self.work_dir, 'static', 'settings.xml')
        ]

        if build_target != 'general':
            mvn_cmd.append('-Pandroid')

        with open(mvnlog_path, 'a') as mvn_log, open(mvnerr_path, 'a') as mvn_err:
            run_process(mvn_cmd, shell=True, check=True, timeout=10000, work_dir=self.work_dir, stdout=mvn_log, stderr=mvn_err)

        base = 'http://artifactory.yandex.net/yandex_mobile_releases/ru/yandex/se/log'
        directory = 'scarab' + suffix
        name = '%s.%s' % (version, self.revision)
        link = '%s/%s/%s/' % (base, directory, name)
        self.set_info('\nNew artifact version: %s\nArtifacts placed <a href="%s">here</a>' % (name, link), do_escape=False)
        logging.info("Artifacts placed: %s" % link)


__Task__ = DeployScarabJavaApi
