import os.path
import logging

import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk.process import run_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.build import parameters
from sandbox.projects.logs.scarab.common import DeployScarabBase, generate_commit_message
from sandbox.projects.logs.common import copy_and_add


class TargetReports(sdk_parameters.SandboxStringParameter):
    name = 'target_reports'
    description = 'Target arcadia reports paths. Deploy to all perl-reports: web/report=SERP-50658;yweb/news/report;web/yaca/report'
    required = True
    default_value = ''


class DryRun(sdk_parameters.SandboxBoolParameter):
    name = 'dry_run'
    description = 'dry run?'
    required = True
    default_value = False


class DeployScarabPerlApi(DeployScarabBase):
    type = "DEPLOY_SCARAB_PERL_API"

    environment = DeployScarabBase.environment

    input_parameters = DeployScarabBase.input_parameters + [TargetReports, DryRun]

    # ask for IPv4
    dns = ctm.DnsType.DNS64

    def on_execute(self):
        logging.info('test/gen/untar...')
        self.test_and_generate(api='perl', url=self.ctx.get(parameters.ArcadiaUrl.name), static_dir='scarab/static/perl')
        self.ctx['published_version'] = 'TODO'

        for target_report in self.ctx[TargetReports.name].strip().strip(';').split(';'):
            report_config = target_report.strip().strip('=').split('=')
            if not report_config:
                continue

            base_report_url = report_config[0]
            if not base_report_url:
                continue

            commit_message_prefix = report_config[1] if len(report_config) > 1 else ''

            report_svn_url = os.path.join(SandboxArcadiaUrlParameter.default_value, base_report_url, 'contrib/lib/perl5')
            logging.info('checkout target report scarab {}'.format(report_svn_url))
            report_svn_dir = Arcadia.get_arcadia_src_dir(report_svn_url)
            logging.info("Deploy scarab into {}".format(report_svn_dir))

            logging.info("Copy generated files")
            copy_and_add(self.path() + '/generated-sources/perl/Scarab/Common', report_svn_dir + '/Scarab/Common')
            copy_and_add(self.path() + '/generated-sources/perl/Scarab/Report', report_svn_dir + '/Scarab/Report')

            logging.info("Copy static files")
            copy_and_add(self.path() + '/static', report_svn_dir + '/Scarab')

            self.make_readme(report_svn_dir + '/Scarab')

            logging.info("Add all new files to vcs")
            opts = Arcadia.SvnOptions()
            opts.force = True
            Arcadia.svn('add', opts=opts, path=report_svn_dir)

            status = Arcadia.status(report_svn_dir)
            logging.info('ARCADIA STATUS... %s', status)
            run_process(["svn", "diff"], work_dir=report_svn_dir, log_prefix='svn_diff')
            run_process(["svn", "st"], work_dir=report_svn_dir, log_prefix='svn_diff')

            if self.ctx.get(DryRun.name):
                logging.info("Dry run. Commit skipped.")
            else:
                comment = generate_commit_message('perl')
                if commit_message_prefix:
                    comment = commit_message_prefix + ' ' + comment
                svn_stdout = Arcadia.commit(report_svn_dir, comment, 'zomb-sandbox-rw')
                scarab_revision = self.set_published_revision(svn_stdout)
                logging.info("COMMITED REVISION {} WITH COMMENT={}".format(scarab_revision, comment))

        logging.info("Done!")


__Task__ = DeployScarabPerlApi
