import os
import os.path
import sys
import logging
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox import projects
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.logs.scarab.common import DeployScarabBase
from sandbox.projects.logs.scarab.config.deploy_ios import providers
from sandbox.projects.logs.scarab.common.monorepo import Monorepo
from sandbox.projects.common.build import parameters
import sandbox.projects.common.environments
from sandbox import sdk2

BB_TOKEN_OWNER = 'bagiro44'
BB_TOKEN_NAME = 'SCARAB_MONOREPO_TOKEN'


class SpecialBuild(sdk_parameters.SandboxRadioParameter):
    name = 'specialBuildName'
    description = 'Special build name for required target.'
    choices = list((x, x) for x in providers)
    per_line = 1
    default_value = 'yandex_ios'


class TemplateBranch(sdk_parameters.SandboxStringParameter):
    name = 'templateBranchName'
    description = 'Name of branch that store proj template'
    required = True
    default_value = 'sandbox'


class SwiftRepositoryUrl(sdk_parameters.SandboxStringParameter):
    name = 'SwiftRepositoryUrl'
    description = 'Target repository URL'
    required = True
    default_value = 'git@bitbucket.browser.yandex-team.ru:ml/scarabey-api-swift.git'


class SwiftRepositoryUser(sdk_parameters.SandboxStringParameter):
    name = 'SwiftRepositoryUser'
    description = 'Target repository User (staff login usually)'
    required = True
    default_value = 'bagiro44'  # TODO make empty


class DeployScarabSwiftApi(DeployScarabBase):
    type = "DEPLOY_SCARAB_SWIFT_API"

    input_parameters = DeployScarabBase.input_parameters + [
        SpecialBuild,
        SwiftRepositoryUrl,
        SwiftRepositoryUser,
        TemplateBranch
    ]

    environment = (
        environments.PipEnvironment('coverage'),
        environments.PipEnvironment('pbxproj', version="2.7.0"),
        environments.SvnEnvironment(),
        sandbox.projects.common.environments.SandboxGitEnvironment()
    )

    # client_tags = "CUSTOM_SCARAB"  # ios-test47-dom0

    def create_pr(self, branch_name, isTesting, sbTaskId):
        mono = Monorepo()
        token = sdk2.Vault.data(BB_TOKEN_OWNER, BB_TOKEN_NAME)
        mono.create_pr(branch_name, isTesting, sbTaskId, token)

    def on_execute(self):
        api = 'swift/' + 'swift-' + self.ctx.get(SpecialBuild.name)
        self.test_and_generate(api=api, url=self.ctx.get(parameters.ArcadiaUrl.name))
        scarab_api_dir = os.path.join(self.work_dir, 'scarabey-api-swift')
        scarab_source_dir = os.path.join(self.work_dir, 'generated-sources')
        task_source_directory = os.path.dirname(projects.logs.scarab.DeployScarabSwiftApi.__file__)
        script_path = os.path.join(task_source_directory, 'updateSwiftScript')
        logging.info('WORKING_DIR_!!!!: %s', scarab_source_dir)
        logging.info('GIT_DIR_!!!!: %s', scarab_api_dir)

        gitlog_path = os.path.join(sdk_paths.get_logs_folder(), 'git.log')
        self.ctx['published_version'] = 'TODO'
        result_branch = self.ctx.get(SpecialBuild.name)
        logging.info('RESULT BRANCH _____ %s', result_branch)

        with ssh.Key(self, self.ctx.get(SwiftRepositoryUser.name), "Bitbucket"):
            with open(gitlog_path, 'a') as git_log:
                # old bitbucket deploy
                logging.info('===Clone scarabey-api-swift===')
                run_process(['git', 'clone', self.ctx.get(SwiftRepositoryUrl.name)+''],
                            shell=True, timeout=10000, work_dir=self.work_dir,
                            stdout=git_log, outputs_to_one_file=True)

                logging.info('===Remove old and add new events===')
                run_process(['sh', 'update.sh', scarab_api_dir+'', scarab_source_dir+'',
                            self.ctx.get(TemplateBranch.name)+'', result_branch, self.ctx.get(SwiftRepositoryUser.name)+'',
                            self.revision, sys.executable], shell=True, timeout=10000, work_dir=script_path,
                            log_prefix='git_xproj', outputs_to_one_file=True)

                logging.info('===Run xcodebuild===')
                run_process(['xcodebuild'], shell=True, timeout=10000, work_dir=scarab_api_dir, log_prefix='run_xcodebuild', outputs_to_one_file=True)

                # new bb deploy
                logging.info('===Clone monorepo===')
                monorepo_dir = self.path('monorepo')
                if not os.path.exists(monorepo_dir):
                    os.makedirs(monorepo_dir)

                run_opts = {'work_dir': monorepo_dir, 'shell': True, 'timeout': 10000, 'stdout': git_log, 'outputs_to_one_file': True}
                run_process(['git', 'init', '.'], **run_opts)
                run_process(['git', 'remote', 'add', 'origin', 'ssh://git@bb.yandex-team.ru/mobile/monorepo.git'], **run_opts)
                run_process(['git', 'config', 'core.sparsecheckout', 'true'], **run_opts)
                run_process(['echo', '"/common/ios/swissknife/*"', '>>', '.git/info/sparse-checkout'], **run_opts)
                run_process(['echo', '"scarab/ios/*"', '>>', '.git/info/sparse-checkout'], **run_opts)
                run_process(['git', 'pull', '--depth', '1', 'origin', 'master'], **run_opts)
                run_process(['git', 'fetch', 'origin', 'scarab/ios/{}:scarab/ios/{} || true'.format(result_branch, result_branch)], **run_opts)
                run_process(['git', 'checkout', '-B', 'scarab/ios/{}'.format(result_branch)], **run_opts)

                dest_branch = 'scarab/ios/{}'.format(result_branch)

                run_opts = {'work_dir': script_path, 'shell': True, 'timeout': 10000, 'log_prefix': 'git_xproj', 'outputs_to_one_file': True}
                run_process(['sh', 'prepare_monorepo.sh',
                            str(self.path('monorepo')),
                            dest_branch,
                            str(scarab_source_dir), str(script_path),
                            sys.executable],
                            **run_opts)

                logging.info('===Push to monorepo===')
                run_process(['sh', 'push_monorepo.sh',
                            str(self.path('monorepo')),
                            dest_branch,
                            self.revision],
                            **run_opts)

                self.create_pr(dest_branch, False, self.revision)
