import os
import shutil
import re
import os.path
from sandbox import projects

from sandbox.common.types import task as ctt
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.process import run_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.logs.scarab.common import DeployScarabBase
from sandbox.projects.logs.common import copy_and_add
from sandbox import sdk2
from sandbox.projects.common.build import parameters
import sandbox.projects.common.environments


import logging


class TargetParameter(sdk_parameters.SandboxStringParameter):
    name = 'target'
    description = 'Target path, related to arcadia'
    required = True


class PublishPackages(sdk_parameters.SandboxBoolParameter):
    name = 'publish_packages'
    description = 'Publish packages to PyPi and Ubuntu repositories'
    required = True
    default_value = True


class DryRun(sdk_parameters.SandboxBoolParameter):
    name = 'dry_run'
    description = 'dry run?'
    required = True
    default_value = False


class ScarabCrossLangTest(DeployScarabBase):
    type = "SCARAB_CROSS_LANG_TEST"

    environment = (
        environments.PipEnvironment('rstr'),
        environments.SvnEnvironment(),
        sandbox.projects.common.environments.SandboxGitEnvironment()
    )

    input_parameters = DeployScarabBase.input_parameters + [DryRun, PublishPackages]

    # ask for IPv4
    dns = ctm.DnsType.DNS64

    def on_execute(self):
        with self.memoize_stage.create_java_children:
            java_subtask = self.create_subtask(
                task_type="SCARAB_CROSS_LANG_JAVA_TASK",
                description="Java autotest task",
            )
            self.ctx['java_task_id'] = java_subtask.id

            cpp_subtask = self.create_subtask(
                task_type="SCARAB_CROSS_LANG_CPP_TASK",
                description="Cpp  autotest task",
            )
            self.ctx['cpp_task_id'] = cpp_subtask.id

            perl_subtask = self.create_subtask(
                task_type="SCARAB_CROSS_LANG_PERL_TASK",
                description="Perl autotest task",
            )
            self.ctx['perl_task_id'] = perl_subtask.id

            swift_subtask = self.create_subtask(
                task_type="SCARAB_CROSS_LANG_SWIFT_TASK",
                description="Swift autotest task",
            )
            self.ctx['swift_task_id'] = swift_subtask.id

            self.wait_tasks([cpp_subtask, java_subtask, perl_subtask, swift_subtask], ctt.Status.Group.FINISH, wait_all=True)

        java_resource = sdk2.Resource["PLAIN_TEXT"].find(
            attrs=dict(lang='java', task_id=self.ctx['java_task_id'])
        ).first()

        cpp_resource = sdk2.Resource["PLAIN_TEXT"].find(
            attrs=dict(lang='cpp', task_id=self.ctx['cpp_task_id'])
        ).first()

        perl_resource = sdk2.Resource["PLAIN_TEXT"].find(
            attrs=dict(lang='perl', task_id=self.ctx['perl_task_id'])
        ).first()

        swift_resource = sdk2.Resource["PLAIN_TEXT"].find(
            attrs=dict(lang='swift', task_id=self.ctx['swift_task_id'])
        ).first()

        java_events_path = self.sync_resource(java_resource)
        logging.info('#### - ' + str(java_events_path))

        cpp_events_path = self.sync_resource(cpp_resource)
        logging.info('#### - ' + str(cpp_events_path))

        perl_events_path = self.sync_resource(perl_resource)
        logging.info('#### - ' + str(perl_events_path))

        swift_events_path = self.sync_resource(swift_resource)
        logging.info('#### - ' + str(swift_events_path))

        logging.info('test/gen/untar...')
        self.generate_cross_lang_test(api='python3', url=self.ctx.get(parameters.ArcadiaUrl.name), static_dir='scarab/static')

        logging.debug("Increment version")
        pyscarab_version = None
        with open(self.path() + '/static/python/setup.py') as setup, open(self.path() + "/newsetup", "w") as newsetup:
            for line in setup:
                m = re.match(r'(\s+)version="(\d+)\.(\d+)\.(\d+)"(.*)\n', line)
                if m:
                    indent, major, minor, tiny, rest = m.group(1, 2, 3, 4, 5)
                    tiny = int(tiny)
                    logging.debug("Old version is {}. Insert new version {}".format(tiny, tiny+1))
                    pyscarab_version = "{}.{}.{}".format(major, minor, tiny+1)
                    newsetup.write('{}version="{}"{}\n'.format(indent, pyscarab_version, rest))
                else:
                    newsetup.write(line)

        self.ctx['published_version'] = pyscarab_version

        os.unlink(self.path() + '/static/python/setup.py')
        os.rename(self.path() + "/newsetup", self.path() + '/static/python/setup.py')

        logging.info('pyscarab checkout...')
        pyscarab_svn_url = SandboxArcadiaUrlParameter.default_value + '/scarab/api/python3@HEAD'
        pyscarab_svn_dir = Arcadia.get_arcadia_src_dir(pyscarab_svn_url)

        logging.info('pyscarab_svn_dir - '+pyscarab_svn_dir)

        logging.info("Remove old version")
        shutil.rmtree(pyscarab_svn_dir + '/scarab')
        os.mkdir(pyscarab_svn_dir + '/scarab')
        logging.info("Copy generated files")
        copy_and_add(self.path() + '/generated-sources/python/scarab', pyscarab_svn_dir + '/scarab')
        logging.info("Copy static files")
        copy_and_add(self.path() + '/static/python', pyscarab_svn_dir)
        copy_and_add(self.path() + '/static/cross_lang_test/python', pyscarab_svn_dir)
        copy_and_add(pyscarab_svn_dir, self.path() + '/python_test_util')
        task_source_directory = os.path.dirname(projects.logs.scarab.ScarabCrossLangTest.python.__file__)
        script_path = os.path.join(task_source_directory, 'Swift.txt')
        out_path = os.path.join(task_source_directory, 'Python.txt')
        logging.info('Swift json dir - ' + script_path)
        logging.info('Swift json dir - ' + out_path)
        run_process(["python", "validate_logs.py", str(java_events_path), out_path], work_dir=self.path() + '/python_test_util', log_prefix='RUN autotest')
        run_process(["python", "validate_logs.py", str(cpp_events_path), out_path], work_dir=self.path() + '/python_test_util', log_prefix='RUN autotest')
        run_process(["python", "validate_logs.py", str(perl_events_path), out_path], work_dir=self.path() + '/python_test_util', log_prefix='RUN autotest')
        run_process(["python", "validate_logs.py", str(swift_events_path), out_path], work_dir=self.path() + '/python_test_util', log_prefix='RUN autotest')
        logging.info("Done!")


__Task__ = ScarabCrossLangTest
