import re
import logging
import os.path
from os.path import join as pj

import sandbox.sandboxsdk.paths as paths
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk.environments import SvnEnvironment
import sandbox.sandboxsdk.svn as svn

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import ProcessLog, subprocess as sp

from sandbox.common.errors import TaskFailure
from sandbox.projects.common.build import parameters
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.release_machine.input_params2 as rm_params
from sandbox.projects.logs.common import make_commit

SUPPORTED_APIS = {
    'cpp': 'https://wiki.yandex-team.ru/scarab/api/cpp/',
    'java': 'https://wiki.yandex-team.ru/scarab/api/java/',
    'java-mobile': 'https://wiki.yandex-team.ru/scarab/api/java/',
    'java-mobile-minimal': 'https://wiki.yandex-team.ru/scarab/api/java/',
    'js': 'https://wiki.yandex-team.ru/scarab/api/js/',
    'perl': 'https://wiki.yandex-team.ru/scarab/api/perl/',
    'proto': 'https://wiki.yandex-team.ru/scarab/api/proto/',
    'python3': 'https://wiki.yandex-team.ru/scarab/api/python/',
    'swift': 'https://wiki.yandex-team.ru/scarab/api/swift/',
}


def generate_commit_message(api, version=None, revision=None):
    message = "Scarab {} API".format(api.capitalize())
    if version:
        message += " ver {}".format(version)
    if revision:
        message += " for original commit {}".format(revision)
    message += "\nFYI: API documentation: {}".format(SUPPORTED_APIS[api])
    return message


def commit_api(task, paths, **kwargs):
    if not kwargs.get('author'):
        kwargs['author'] = task.Parameters.commit_author

    return make_commit(task, paths, generate_commit_message(task.api, revision=task.Context.source_revision), **kwargs)


class RunTests(sdk_parameters.SandboxBoolParameter):
    name = 'run tests'
    description = 'run all scarab tests before build'
    required = True
    default_value = True


class DeployScarabBaseV2(sdk2.Task):
    changed_files = []

    class Parameters(rm_params.ComponentName2):
        max_restarts = 0

        arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia url', required=True)
        commit_author = sdk2.parameters.String("Author of new commit", default=None)
        publish_arcadia = sdk2.parameters.Bool('Do commit codegen in Arcadia', default_value=True)
        minor_version = sdk2.parameters.Integer('Set minor version explicitly', default=None)

        force = sdk2.parameters.Bool('Force deploy', default_value=False)
        dry_run = sdk2.parameters.Bool('dry run?', default_value=False)

    class Context(sdk2.Context):
        arcadia_src_dir = None
        arcadia_dst_dir = None
        revision = None
        published_revision = None

    def source_dir_rel(self):
        return 'scarab/api/' + self.api + '_gen'

    def source_dir(self):
        assert self.Context.arcadia_src_dir, "No arcadia_src_dir initialized"
        return pj(self.Context.arcadia_src_dir, self.source_dir_rel())

    def destination_dir_rel(self):
        return 'scarab/api/' + self.api

    def destination_dir(self):
        assert self.Context.arcadia_dst_dir, "No arcadia_dst_dir initialized"
        return pj(self.Context.arcadia_dst_dir, self.destination_dir_rel())

    def destination_arcadia_url(self, revision='HEAD'):
        return self.Parameters.arcadia_url.split('@', 1)[0] + '@' + revision

    def results_dir(self):
        return '{}_generated'.format(self.api)

    def get_ya_token(self):
        return sdk2.Vault.data("USERSESSIONSTOOLS", "robot_ci_sessions_arc_token")

    def get_minor_version(self):
        if self.Parameters.minor_version is not None:
            return self.Parameters.minor_version

        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)

        if parsed_url.trunk:
            return 0
        elif parsed_url.branch:
            return 1
        elif parsed_url.tag:
            return 2
        else:
            return 0

    def PrepareArcadia(self):
        logging.info('PrepareArcadia')

        #  checkout source dir
        src_dir = self.Context.arcadia_src_dir = Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_url)
        arcadia_info = Arcadia.info(src_dir)
        self.Context.source_revision = arcadia_info['entry_revision']

        #  prepare dst dir
        dst_dir = self.Context.arcadia_dst_dir = str(self.path('arcadia_dst'))
        Arcadia.checkout(self.destination_arcadia_url(), dst_dir, depth="empty")
        Arcadia.update(pj(dst_dir, '.arcadia.root'))
        Arcadia.update(pj(dst_dir, 'ya'))
        Arcadia.update(pj(dst_dir, 'build'))
        Arcadia.update(self.destination_dir(), parents=True)

        logging.info("source_dir - {}".format(self.source_dir()))
        logging.info("destination_dir - {}".format(self.destination_dir()))

    def GenerateAPI(self, env=None):
        logging.info('GenerateAPI')

        cmd = ['bash', '-e', '-x',
               pj(self.source_dir(), 'generate.sh'),
               self.source_dir(),
               self.destination_dir()]

        clone_env = os.environ.copy()
        clone_env['YA_TOKEN'] = self.get_ya_token()
        clone_env.update(env or {})

        with ProcessLog(self, logger="generate_sh") as pl:
            proc = sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT, env=clone_env)
            proc.wait()
            if proc.returncode != 0:
                raise TaskFailure("generate.sh exit code {} check generate_sh.out.log for error message".format(proc.returncode))

    def CheckChangedFiles(self, filter_exclude=None):
        logging.info('CheckChangedFiles')
        self.changed_files = []

        destination_dir = self.destination_dir()
        status_raw = Arcadia.status(destination_dir)
        logging.debug("svn status:" + status_raw)

        for line in status_raw.split('\n'):
            if not line:
                continue

            tokens = line.split()
            assert len(tokens) == 2, line

            mark, filename = tokens
            if mark in ['?']:
                continue

            self.changed_files.append(filename)

        if filter_exclude:
            self.changed_files = filter(lambda f: not filter_exclude(f), self.changed_files)

        logging.info("changed files:" + str(self.changed_files))

        if self.changed_files or self.Parameters.force:
            return True

        return False

    def CommitAPI(self, **kwargs):
        if not self.Parameters.publish_arcadia:
            return

        if self.Parameters.dry_run:
            self.set_info("Dry run. Deploy skipped.")
            return

        if not self.changed_files:
            if self.Parameters.force:
                self.set_info("No changes, but forced deploy.")
            else:
                self.set_info("No changes, deploy skipped.")
                return

        result = commit_api(self, self.destination_dir(), ya_token=self.get_ya_token(), **kwargs)

        if result.revision:
            if result.revision == -1:
                self.set_info("No changes")
                self.Context.published_revision = self.Context.source_revision
            else:
                rev = self.Context.published_revision = result.revision
                self.set_info("OK, committed as <a href='https://a.yandex-team.ru/arc/commit/{}'>{}</a>".format(rev, rev), do_escape=False)
        elif result.review_url:
            self.set_info("Created pull-request: <a href='{}'>{}</a>".format(result.review_url, result.review_url), do_escape=False)
        else:
            raise TaskFailure("COMMIT FAILED: {}".format(result.output))

    def build(self, source_root, targets, **kwargs):
        if not isinstance(targets, (list, tuple)):
            targets = [targets]

        results_dir = str(self.path(self.results_dir()))

        sdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            source_root=source_root,
            targets=targets,
            results_dir=results_dir,
            clear_build=False,
            use_dev_version=False,
            sanitize=False,
            checkout=True,
            **kwargs
        )

        #  XXX workaround for local sandbox run
        #  result are unexpectedly built in <execution_dir>/runtime_data/tasks/7/0/7/arcadia_src
        source_dirname = os.path.basename(source_root)
        for path, dirs, _ in os.walk(results_dir):
            for d in dirs:
                if d == source_dirname:
                    return pj(path, d)

        return results_dir

    def untar(self, tarball=None):
        if tarball is None:
            tarball = str(self.path('scarab', 'api', self.api, 'generated.tar'))

        generated = paths.make_folder(str(self.path('generated-sources')))

        run_process(
            ['tar', 'xf', tarball, '-C', generated],
            log_prefix='tar',
            shell=True,
            check=True,
            timeout=10000
        )

        return generated


class DeployScarabBase(SandboxTask):
    environment = (
        SvnEnvironment(),
    )

    input_parameters = [
        parameters.ArcadiaUrl,
        RunTests,
    ]

    def init(self, api, url, static_dir=None):
        self.api = api
        self.url = url
        self.static_dir = static_dir

        self.tarlog_path = pj(paths.get_logs_folder(), 'tar.log')
        self.tarerr_path = pj(paths.get_logs_folder(), 'tar.err')
        self.svnlog_path = pj(paths.get_logs_folder(), 'svn.log')
        self.svnerr_path = pj(paths.get_logs_folder(), 'svn.err')
        self.work_dir = self.path()

    def clone(self):
        self.src_dir = sdk.do_clone(self.url, self)
        self.revision = svn.Arcadia.get_revision(self.src_dir)

    def static(self):
        if self.static_dir:
            path = svn.Arcadia.parse_url(self.url).path + '/' + self.static_dir
            svn.Arcadia.checkout('arcadia:' + path, path=pj(self.path(), 'static'), revision=self.revision)

    def build_cross_lang_test(self):
        if self.api not in SUPPORTED_APIS and 'swift' not in self.api and 'java-mobile-minimal' not in self.api:
            raise Exception("Lang {} is not supported. Supportad apis are: {}".format(self.api, SUPPORTED_APIS.keys()))
        sdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            source_root=self.src_dir,
            targets=['scarab/api/tests/' + self.api + '/cross_lang'],
            build_type=consts.RELEASE_BUILD_TYPE,
            results_dir=self.path(),
            clear_build=False,
            use_dev_version=False,
            sanitize=False,
            checkout=True,
            )

    def test(self):
        sdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            source_root=self.src_dir,
            targets=['scarab/api/tests'],
            build_type=consts.RELEASE_BUILD_TYPE,
            clear_build=True,
            use_dev_version=False,
            sanitize=False,
            checkout=True,
            test=True
            )

    def generate(self):
        if self.api not in SUPPORTED_APIS and 'swift' not in self.api and 'java-mobile-minimal' not in self.api:
            raise Exception("Lang {} is not supported. Supportad apis are: {}".format(self.api, SUPPORTED_APIS.keys()))

        sdk.do_build(
            build_system=consts.YMAKE_BUILD_SYSTEM,
            source_root=self.src_dir,
            targets=['scarab/api/' + self.api],
            build_type=consts.RELEASE_BUILD_TYPE,
            results_dir=self.path(),
            clear_build=True,
            use_dev_version=False,
            sanitize=False,
            checkout=True,
            )

    def untar(self, tarball=None):
        if tarball is None:
            tarball = pj(self.path(), 'scarab', 'api', self.api, 'generated.tar')
        generated = paths.make_folder(pj(self.path(), 'generated-sources'))
        with open(self.tarlog_path, 'w') as tar_log, open(self.tarerr_path, 'w') as tar_err:
            run_process(['tar', 'xf', tarball, '-C', generated], log_prefix='tar', shell=True, check=True, timeout=10000, stdout=tar_log, stderr=tar_err)

    def commit_message(self, version=None):
        if version is None:
            version = self.ctx.get('published_version', 'UNKNOWN')
        return "Scarab {} API version {}\n{} https://sandbox.yandex-team.ru/task/{}/view\nDocumentation: {}".format(self.api, version, self.type, self.id, SUPPORTED_APIS.get(self.api, 'TODO'))

    def set_published_revision(self, svn_stdout):
        if 'published_revisions' not in self.ctx:
            self.ctx['published_revisions'] = []

        m = re.search(r'\s+(\d+)\b', svn_stdout)
        assert m, 'Cant parse committed svn revision from: {}'.format(svn_stdout)

        published_revision = int(m.group(1))
        self.ctx['published_revisions'].append(published_revision)

        return published_revision

    def make_readme(self, publish_dir, text=None):
        if text is None:
            text = self.commit_message()
        with open(publish_dir + '/README', 'w') as f:
            f.write("WARNING! Do not modify this code\n" + text)

    def generate_cross_lang_test(self, api, url, static_dir=None):
        self.init(api, url, static_dir)
        self.clone()
        self.build_cross_lang_test()
        self.untar(tarball=pj(self.path(), 'scarab', 'api', 'tests', self.api, 'cross_lang', 'generated.tar'))
        self.static()

    def test_and_generate(self, api, url, static_dir=None):
        self.init(api, url, static_dir)
        self.clone()
        if self.ctx.get(RunTests.name):
            self.test()
        self.generate()
        self.untar()
        self.static()
