# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging
import os
import random
import string
import sys
import time

from os.path import join as pj

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process  # maybe sdk2.helpers.ProcessLog?
from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.projects.common.arcadia import sdk as arcadia_sdk


def RunProcess(cmd, env, log_prefix=None, exception_if_nonzero_code=True):
    cmd_str = ' '.join([str(cmd_elem) for cmd_elem in cmd])
    process = run_process(
        cmd_str,
        outs_to_pipe=True, check=False, shell=True, wait=True,
        environment=env,
        log_prefix=log_prefix,
    )
    result, error = process.communicate()
    if exception_if_nonzero_code and process.returncode != 0:
        raise Exception(error)
    return result, error


class SendUsStatsToSolomon(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('networkx', version='2.2', use_wheel=True),
        ]

        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        use_prod_revision = sdk2.parameters.Bool("Use production revision", default=False) # change default after fix REACTOR-1366
        history_days = sdk2.parameters.Integer("Days to track back", default=1)

        with sdk2.parameters.Group("Time period") as period_group:
            period_daily = sdk2.parameters.Bool("1d", default=True)
            period_fast = sdk2.parameters.Bool("30min", default=True)

        with sdk2.parameters.Group("Reactor token") as reactor_token_block:
            reactor_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with reactor token", required=True)  # on practice: USERSESSIONSTOOLS. But do not specify default here for safety
            reactor_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with reactor token", required=True)  # on practice: robot-make-sessions-reactor-token (TODO: create secret). But do not specify default here for safety
            # TODO: maybe add defaults at least to description?
        with sdk2.parameters.Group("Statface token") as statface_token_block:
            statface_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with statface token", required=True)  # on practice: USERSESSIONSTOOLS. But do not specify default here for safety
            statface_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with solomon token", required=True)
        with sdk2.parameters.Group("Solomon token") as solomon_token_block:
            solomon_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with solomon token", required=True, default="USERSESSIONSTOOLS")  # remove default after fix REACTOR-1366
            solomon_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with solomon token", required=True, default="robot_make_sessions_solomon_token") # remove default after fix REACTOR-1366

    class Context(sdk2.Context):
        pass

    def GetReactorTokenPath(self):
        secret_content = self.GetReactorToken()
        reactor_token_path = pj(str(self.path()), 'reactor_token_file')
        os.system('echo {} > {}'.format(secret_content, reactor_token_path))

        return reactor_token_path

    def GetStatfaceTokenPath(self):
        secret_content = self.GetStatfaceToken()
        statface_token_path = pj(str(self.path()), 'statface_token_file')
        os.system('echo {} > {}'.format(secret_content, statface_token_path))

        return statface_token_path

    def GetEnv(self, arcadia_user_sessions_local_abs_path):
        env = dict(os.environ)
        paths = env['PYTHONPATH'].split(':') if 'PYTHONPATH' in env else []
        paths.insert(0, arcadia_user_sessions_local_abs_path)
        env['PYTHONPATH'] = ':'.join(paths)

        return env

    def CheckoutArcadiaSubfolder(self, arcadia_subfolder, arcadia_src_dir, svn_url, use_cache=True):
        pos = svn_url.rfind('@')

        if pos != -1:
            dir_url = pj(svn_url[:pos], arcadia_subfolder) + svn_url[pos:]
        else:
            dir_url = pj(svn_url, arcadia_subfolder)

        arcadia_subfolder_local_abs_path = pj(arcadia_src_dir, arcadia_subfolder)

        if use_cache:
            with arcadia_sdk.mount_arc_path(dir_url, use_arc_instead_of_aapi=True) as p:
                sdk2.paths.copy_path(str(p), arcadia_subfolder_local_abs_path)
        else:
            sdk2.svn.Arcadia.checkout(dir_url, arcadia_subfolder_local_abs_path)

        return arcadia_subfolder_local_abs_path

    def PrepareArcadia(self, svn_url):
        if svn_url in self.arcadia_key_to_reactor_dir:
            return

        arcadia_src_dir = pj(str(self.path()), 'local_arcadia_' + \
            ''.join(random.SystemRandom(time.time()).choice(
                string.ascii_lowercase + string.ascii_uppercase + string.digits
            ) for _ in range(20)))

        self.arcadia_key_to_reactor_dir[svn_url] = self.CheckoutArcadiaSubfolder('quality/user_sessions/reactor', arcadia_src_dir, svn_url=svn_url)

    def DoSendToSolomon(self, period, labels, token, statface_token, solomon_token):
        from us_processes import reactor_tasks, create_sessions

        options = reactor_tasks.Options(
            reactor_server="MOCK",
            time_periods=[period],
            is_trouble_mode=False,
            version=None,
            svn_url=None,
            token=token,
            quota_project="fake-quota-project",
            graph_owner="robot-make-sessions",
            prod_version_out_file_path=None,
            is_solomon_mode=True,
            history_days=self.Parameters.history_days,
            statface_token=statface_token,
            solomon_token=solomon_token,
            threads_count=20,
        )

        options.set_labels(labels)

        create_sessions.create_sessions(options)

    def SendToSolomon(self):
        from us_processes.time_periods import Periods
        from us_processes import public_sessions_config as scripts_config

        token = self.GetReactorToken()
        statface_token = self.GetStatfaceToken()
        solomon_token = self.GetSolomonToken()

        for sessions in scripts_config.userSessions:
            scripts_config.userSessions[sessions]["clusters_spec"].calcClusters = [cc for cc in scripts_config.userSessions[sessions]["clusters_spec"].calcClusters if cc not in ["freud", "hume"]]
            scripts_config.userSessions[sessions]["clusters_spec"].syncToClusters = [cc for cc in scripts_config.userSessions[sessions]["clusters_spec"].syncToClusters if cc not in ["freud", "hume"]]

        err_accumulator = []

        time_periods = []
        if self.Parameters.period_daily:
            time_periods.append(Periods.DAILY)
        if self.Parameters.period_fast:
            time_periods.append(Periods.FAST)
        assert time_periods, "Time period is required"

        for period in time_periods:
            labels_lists = [["search"], ["!search"]]
            for labels in labels_lists:
                try:
                    self.DoSendToSolomon(period, labels, token, statface_token, solomon_token)
                except Exception as e:
                    msg = "There were ERRORS on labels: {}; period: {}".format(labels, period)
                    logging.exception(msg + " and now we are moving on")
                    err_accumulator.append(msg)

        if err_accumulator:
            final_msg = ""
            for msg in err_accumulator:
                final_msg += msg + "\n"
            raise Exception(final_msg)


    def GetReactorToken(self):
        secret_owner = self.Parameters.reactor_token_secret_owner
        secret_name = self.Parameters.reactor_token_secret_name
        token = sdk2.Vault.data(secret_owner, secret_name)
        return token

    def GetStatfaceToken(self):
        secret_owner = self.Parameters.statface_token_secret_owner
        secret_name = self.Parameters.statface_token_secret_name
        token = sdk2.Vault.data(secret_owner, secret_name)
        return token

    def GetSolomonToken(self):
        secret_owner = self.Parameters.solomon_token_secret_owner
        secret_name = self.Parameters.solomon_token_secret_name
        token = sdk2.Vault.data(secret_owner, secret_name)
        return token

    def on_execute(self):
        logging.info('SendUsStatsToSolomonTask: Start')

        self.arcadia_key_to_reactor_dir = {}
        self.PrepareArcadia(Arcadia.ARCADIA_TRUNK_URL)

        reactor_token_path = self.GetReactorTokenPath()
        cmd = [
            sys.executable,
            "-m", "us_processes.get_production_svn_url",
            "-t", reactor_token_path
        ]

        env = self.GetEnv(self.arcadia_key_to_reactor_dir[Arcadia.ARCADIA_TRUNK_URL])
        if self.Parameters.use_prod_revision:
            result, error = RunProcess(cmd, env, log_prefix="get_production_svn_url")
            result = result.strip()

            # leave possibility for hotfix merged into release branch
            # but not in prod yet
            prod_svn_url = result.split('@')[0]
        else:
            prod_svn_url = Arcadia.ARCADIA_TRUNK_URL

        self.PrepareArcadia(prod_svn_url)
        sys.path.insert(0, self.arcadia_key_to_reactor_dir[prod_svn_url])

        self.SendToSolomon()
