import logging
import time

from sandbox.common import utils
from sandbox.projects.common import aqua as aqua_launcher
from sandbox.projects.common.aqua import AquaLauncher


class LaunchStatus(utils.Enum):
    class Group(utils.GroupEnum):
        COMPLETE = None
        IN_PROGRESS = None
        ABNORMAL_TERMINATION = None

    with Group.COMPLETE:
        REPORT_STARTED = None
        REPORT_READY = None
        REPORT_FAILED = None
        REPORT_REQUESTED = None
        FINISHED = None
        FAILED = None
        REVOKED = None

    with Group.IN_PROGRESS:
        RUNNING = None
        RUNNABLE = None


LaunchStatus.Group.ABNORMAL_TERMINATION = (LaunchStatus.FAILED, LaunchStatus.REVOKED)


class Aqua(AquaLauncher):
    def __init__(self, pack_name):
        AquaLauncher.__init__(self, pack_name)

    @staticmethod
    def get_result_link(launch_id):
        return aqua_launcher.get_result_link(launch_id)

    @staticmethod
    def get_allure_result_link(launch_id):
        return aqua_launcher.get_allure_result_link(launch_id)

    def revoke(self, launch_id):
        r = self.session.delete(self._get_url("launch", launch_id))
        r.raise_for_status()

    def wait_launch(self, launch_id, poll_interval, timeout):
        start_time = time.time()
        status = None
        launch_info = {}
        while not status or status not in LaunchStatus.Group.COMPLETE:
            logging.info("Launch {} still in progress".format(launch_id))
            logging.debug("Launch has info: '{}'".format(launch_info))
            if time.time() - start_time >= timeout:
                self.revoke(launch_id)
                raise Exception("Timeout exceeded")
            time.sleep(poll_interval)
            launch_info = self.get_info("launch", launch_id)
            status = LaunchStatus[launch_info["launchStatus"]]

        if status in LaunchStatus.Group.ABNORMAL_TERMINATION:
            raise Exception("Tests launch '{}' failed with status '{}'".format(self._launch_id, str(status)))
        else:
            return launch_info
