import json

import time
from sandbox.common import utils
from sandbox.projects.sandbox_ci.qloud.qloud import QloudApi as SandboxQloudApi
from sandbox.projects.sandbox_ci.utils.request import send_request


class EnvStatus(utils.Enum):
    class Group(utils.GroupEnum):
        IDLE = None
        COMPLETE = None
        IN_PROGRESS = None
        ABNORMAL_TERMINATION = None

    with Group.IDLE:
        TEMPLATE = None
        NEW = None

    with Group.COMPLETE:
        CANCELED = None
        REMOVED = None
        FAILED = None
        DEPLOYED = None

    with Group.IN_PROGRESS:
        ALLOCATING = None
        CONFIGURING = None
        PREPARING = None
        DEPLOYING = None
        PENDING_REMOVE = None
        REMOVING = None
        ACTIVATING = None


EnvStatus.Group.ABNORMAL_TERMINATION = (EnvStatus.FAILED, EnvStatus.CANCELED)


def utf_encode(text):
    return text.encode("utf-8") if isinstance(text, unicode) else text


class QloudApi(SandboxQloudApi):
    def __init__(self, url, token, logger, deploy_timeout=1800, delete_timeout=1800):
        self._url = url
        self._token = token
        self._logger = logger
        self._deploy_timeout = deploy_timeout
        self._delete_timeout = delete_timeout
        self._headers = {
            'Authorization': 'OAuth {token}'.format(token=self._token)
        }

    def get_environments(self, project, application):
        url = self.get_full_url('api/v1/project/{project}'.format(
            project=project,
        ))
        res = send_request('get', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('Get project info, code: {} text: {}'.format(res.status_code, utf_encode(res.text)))
        envs = filter(lambda d: d['objectId'] == project+'.'+application, res.json()['applications'])
        return map(lambda a: a['objectId'], envs[0]['environments']) if len(envs) > 0 else []

    def get_environment_status(self, environment_id):
        url = self.get_full_url('api/v1/environment/status/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = send_request('get', url, headers=self._headers, verify=False)
        self._logger.info('Get env status, code: {} text: {}'.format(res.status_code, utf_encode(res.text)))
        return res

    def get_environment_deployed_version(self, environment_id):
        url = self.get_full_url("api/v1/environment/stable/{}".format(environment_id))
        res = send_request('get', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('Get env deployed version, code:{} text:{}'.format(res.status_code, utf_encode(res.text)))
        versions = filter(lambda v: v['status'] == "DEPLOYED", res.json()['versions'])
        if not versions:
            return None
        return str(versions[0]['version'])

    def create_environment(self, environment_name, application_id, engine='platform'):
        status_res = self.get_environment_status(application_id + '.' + environment_name)

        if status_res.ok:
            self._logger.info('env {} already exists, delete it'.format(application_id + '.' + environment_name))
            self.delete_environment(application_id + '.' + environment_name)

        url = self.get_full_url(
            'api/v1/environment/new?environmentName={env_name}&applicationId={app_id}&engine={engine}'.format(
                env_name=environment_name,
                app_id=application_id,
                engine=engine
            ))
        res = send_request('post', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('create env status code:{} text:{}'.format(res.status_code, utf_encode(res.text)))
        return res

    def upload_environment(self, data):
        url = self.get_full_url('api/v1/environment/upload')
        headers = self._headers.copy()
        headers['Content-Type'] = 'application/json'
        res = send_request('post', url, headers=headers, data=data, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('upload new environment status code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        environment_id = json.loads(data)['objectId']
        self.wait_for_environment_deploy(environment_id)
        return res

    def dump_environment(self, environment_id):
        url = self.get_full_url('api/v1/environment/dump/{environment_id}'.format(
            environment_id=environment_id
        ))
        res = send_request('get', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('Dump environment id:{} code: {} text: {}'.format(
            environment_id, res.status_code, utf_encode(res.text)))
        return res

    def fork_environment(self, environment_name, origin_id):
        url = self.get_full_url(
            'api/v1/environment/fork?environmentName={environment_name}&originId={origin_id}'.format(
                environment_name=environment_name,
                origin_id=origin_id
            ))
        res = send_request('post', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('fork environment status code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        return res

    def deploy_environment(self, environment_id, keep_sandbox='true'):
        url = self.get_full_url(
            'api/v1/environment/deploy?environmentId={environment_id}&keepSandbox={keep_sandbox}'.format(
                environment_id=environment_id,
                keep_sandbox=keep_sandbox
            ))
        res = send_request('post', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('deploy environment code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        self.wait_for_environment_deploy(environment_id)
        return res

    def delete_environment(self, environment_id):
        url = self.get_full_url('api/v1/environment/stable/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = send_request('delete', url, headers=self._headers, verify=False)
        self._logger.info('delete env status code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        self._logger.info('Start wait for delete environment')
        status, res = self.wait_environment_status(
            environment_id, expected_statuses=[], timeout=self._delete_timeout)
        self._logger.info('Environment delete, success!')
        return res

    def wait_for_environment_deploy(self, environment_id):
        self._logger.info('Waiting for environment deploy')
        status, res = self.wait_environment_status(environment_id, expected_statuses=EnvStatus.Group.COMPLETE,
                                                   timeout=self._deploy_timeout)
        if status != EnvStatus.DEPLOYED:
            error_text = "Deployment failed with unexpected status '{}'".format(str(status))
            if status == EnvStatus.FAILED:
                error_text += ": {}".format(res['statusMessage'])
            if not res.ok:
                error_text += ": request failed, code: {}, message: {}".format(res.status_code, utf_encode(res.text))
            raise Exception(error_text)

    def wait_environment_status(self, environment_id, expected_statuses, timeout, sleep_time=5):
        start_time = time.time()
        status = None

        while not status or status not in expected_statuses:
            if time.time() - start_time >= timeout:
                raise Exception("Wait timeout exceeded")

            time.sleep(sleep_time)
            res = self.get_environment_status(environment_id)
            if res.ok:
                status = EnvStatus[res.json()['status']]
            else:
                break

        return status, res

    def get_component(self, component_id):
        url = self.get_full_url('api/v1/component/{component_id}'.format(component_id=component_id))
        res = send_request('get', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('get component {}: {}'.format(component_id, str(utf_encode(res.text))))
        return res

    def create_component(self, environment_id, component_name, component_type='standard'):
        req_str = 'api/v1/component?environmentId={environment_id}&componentType={component_type}&componentName={' \
                  'component_name} '
        url = self.get_full_url(req_str.format(environment_id=environment_id,
                                               component_name=component_name,
                                               component_type=component_type))
        res = send_request('put', url, headers=self._headers, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'create shooting ground component code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        return res

    def modify_component(self, component_id, data):
        headers = self._headers.copy()
        headers['Content-Type'] = 'application/json'
        url = self.get_full_url('api/v1/component/{component_id}'.format(
            component_id=component_id
        ))
        res = send_request('post', url, headers=headers, data=data, verify=False)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'modify component code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        return res

    def get_all_running_instances(self, environment_id):
        url = self.get_full_url('api/v1/status/{environment_id}'.format(
            environment_id=environment_id
        ))
        res = send_request('get', url, headers=self._headers, verify=False)
        self._logger.info('get all running instances code:{} text:{}'.format(str(res.status_code), str(utf_encode(res.text))))
        return res

    def deploy_component_image(self, image, image_hash, environment_id, component, version, comment=None):
        url = self.get_full_url("api/v1/component/{}.{}/{}/deploy".format(environment_id, component, version))
        if comment:
            url += "?comment={}".format(comment)
        params = {'repository': image,
                  'hash': image_hash}
        res = send_request('post', url, headers=self._headers, json=params)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'deploy component image code:{} text:{}'.format(res.status_code, utf_encode(res.text)))

    def fast_deploy_environment(self, environment_id, version, params, comment=None):
        url = self.get_full_url('api/v1/environment/fast-deploy/{environment_id}/{version}'.format(
            environment_id=environment_id,
            version=version
        ))
        if comment:
            url += "?comment={}".format(comment)
        res = send_request('post', url, headers=self._headers, json=params)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'deploy environment image code:{} text:{}'.format(res.status_code, utf_encode(res.text)))
        return res
