from ConfigParser import ConfigParser
import subprocess as sp
import logging
import json
import shutil
from dateutil import parser
from datetime import timedelta
from datetime import date, datetime
import os
import glob
import requests
from requests.adapters import HTTPAdapter


class ShootingType:
    LINE = 'line'
    STEP = 'step'
    CONST = 'const'


def json_serial(obj):
    """JSON serializer for objects not serializable by default json code"""
    if isinstance(obj, (datetime, date)):
        return obj.isoformat()
    raise TypeError("Type %s not serializable" % type(obj))


class CommonShooting:
    def __init__(self, shooting_type, dur_time, logger):
        self._type = shooting_type
        self._dur_time = dur_time
        self._shooting_start_time = None
        self._shooting_end_time = None
        self._logger = logger

    @property
    def type(self):
        return self._type

    @property
    def dur_time(self):
        return self._dur_time

    @property
    def shooting_start_time(self):
        return self._shooting_start_time

    @shooting_start_time.setter
    def shooting_start_time(self, val):
        self._shooting_start_time = val

    @property
    def shooting_end_time(self):
        return self._shooting_end_time

    @shooting_end_time.setter
    def shooting_end_time(self, val):
        self._shooting_end_time = val

    def convert_date(self, dt):
        return '{:04d}{:02d}{:02d}{:02d}{:02d}{:02d}'.format(dt.year,
                                                             dt.month, dt.day, dt.hour,
                                                             dt.minute, dt.second)

    def create_shooting_reports_json(self, tank_api):
        common_report = tank_api.get_aggregate_shooting_data()
        monitoring_report = tank_api.get_monitoring_info_aggregate()

        return common_report, monitoring_report


class LineShooting(CommonShooting):
    def __init__(self, dur_time, start_rps, end_rps):
        CommonShooting.__init__(self, ShootingType.LINE, dur_time, logging.getLogger('LineShooting'))
        self._start_rps = start_rps
        self._end_rps = end_rps


class ConstShooting(CommonShooting):
    def __init__(self, dur_time, rps_value):
        CommonShooting.__init__(self, ShootingType.CONST, dur_time, logging.getLogger('ConstShooting'))
        self._rps_value = rps_value


class StepShooting(CommonShooting):
    def __init__(self, step_dur_time, start_rps, end_rps, step_value):
        CommonShooting.__init__(self, ShootingType.STEP, 0, logging.getLogger('StepShooting'))
        self._start_rps = start_rps
        self._end_rps = end_rps
        self._step_value = step_value
        self._step_time = step_dur_time

    def is_timestamp_border(self, timestamp, timestamps_array, is_start=True):
        second = timedelta(seconds=1)
        result = True
        checks = range(1, 4)
        if not is_start:
            checks = map(lambda x: -x, checks)
        for i in checks:
            if timestamp + i * second not in timestamps_array:
                result = False
                break
        return result

    def get_all_steps_times(self, tank_api):
        self.steps_count = (self._end_rps - self._start_rps) / self._step_value + 1
        all_available_rps_values = map(lambda x: self._start_rps + x * self._step_value, range(self.steps_count))
        all_cases = tank_api.get_shooting_cases()
        rps_with_timestamps = {}
        for case in all_cases:
            cnt = case['cnt']
            timestamp = parser.parse(case['dt'])
            nearest_rps = min(all_available_rps_values, key=lambda x: abs(x - cnt))
            if abs(nearest_rps - cnt) < self._step_value / 3.:
                if nearest_rps not in rps_with_timestamps:
                    rps_with_timestamps[nearest_rps] = [timestamp]
                else:
                    rps_with_timestamps[nearest_rps].append(timestamp)
        self._logger.debug('Available rps with timestamps: {}'.format(json.dumps(rps_with_timestamps,
                                                                                 default=json_serial)))
        rps_with_start_end_timestamp = {}
        for rps_value in rps_with_timestamps:
            rps_with_timestamps[rps_value] = sorted(list(set(rps_with_timestamps[rps_value])))
            self._logger.debug('RpsValue:{} Json:{}'.format(rps_value, json.dumps(rps_with_timestamps[rps_value],
                                                                                  default=json_serial)))
            start_step_time, end_step_time = None, None
            cur_ts_arr = rps_with_timestamps[rps_value]
            ts_count = len(cur_ts_arr)
            for num in range(ts_count / 2):
                ts = cur_ts_arr[num]
                reverse_ts = cur_ts_arr[ts_count - num - 1]
                self._logger.debug('Rps:{} ts:{} rev_ts:{}'.format(rps_value, str(ts), str(reverse_ts)))
                if not start_step_time and self.is_timestamp_border(ts, cur_ts_arr):
                    start_step_time = ts
                if not end_step_time and self.is_timestamp_border(reverse_ts, cur_ts_arr, is_start=False):
                    end_step_time = reverse_ts
                self._logger.debug('Rps:{} st:{} end:{}'.format(rps_value, str(start_step_time), str(end_step_time)))
                if start_step_time and end_step_time:
                    rps_with_start_end_timestamp[rps_value] = {'start_ts': start_step_time, 'end_ts': end_step_time}
                    break

        self._logger.debug('Result steps timestamps:{}'.format(json.dumps(rps_with_start_end_timestamp,
                                                                          default=json_serial)))
        return rps_with_start_end_timestamp

    def create_shooting_reports_json(self, tank_api):
        if tank_api.is_regular_performance_test:
            rps_with_start_end_timestamp = self.get_all_steps_times(tank_api)
            common_report = {}
            monitoring_report = {}
            for rps_val in rps_with_start_end_timestamp:
                info = rps_with_start_end_timestamp[rps_val]
                start_ts, end_ts = self.convert_date(info['start_ts']), self.convert_date(info['end_ts'])
                http_codes = tank_api.get_http_codes_distribution_aggregate(ts_from=start_ts, ts_to=end_ts)
                percentiles = tank_api.get_percentiles(ts_from=start_ts, ts_to=end_ts)
                net = tank_api.get_net_codes_distribution_aggregate(ts_from=start_ts, ts_to=end_ts)
                monitoring = tank_api.get_monitoring_info_aggregate(ts_from=start_ts, ts_to=end_ts)
                filter_monitoring = tank_api.filter_monitoring_report(monitoring)

                common_report[rps_val] = {}
                common_report[rps_val]['start_time'] = str(rps_with_start_end_timestamp[rps_val]['start_ts'])
                common_report[rps_val]['end_time'] = str(rps_with_start_end_timestamp[rps_val]['end_ts'])
                common_report[rps_val]['http'] = http_codes
                common_report[rps_val]['percentiles'] = percentiles
                common_report[rps_val]['net'] = net

                monitoring_report[rps_val] = filter_monitoring
            return common_report, monitoring_report
        else:
            return CommonShooting.create_shooting_reports_json(self, tank_api)


def create_shooting_class(shooting_str, logger):
    shooting_str = shooting_str.lower()
    args_start_pos = shooting_str.find('(')
    args_end_pos = shooting_str.rfind(')')
    args = shooting_str[args_start_pos+1:args_end_pos].replace(' ', '').split(',')
    if ShootingType.LINE in shooting_str:
        assert len(args) == 3
        date = parser.parse(args[2])
        dur_time = date.hour * 3600 + date.minute * 60 + date.second
        start_rps, end_rps = int(args[0]), int(args[1])
        logger.debug("Creating line shooting with params dur_time:{} start_rps:{} end_rps:{}".format(
            dur_time, start_rps, end_rps))
        return LineShooting(dur_time, start_rps, end_rps)
    elif ShootingType.STEP in shooting_str:
        assert len(args) == 4
        date = parser.parse(args[3])
        step_dur_time = timedelta(seconds=date.hour * 3600 + date.minute * 60 + date.second)
        start_rps, end_rps, step = int(args[0]), int(args[1]), int(args[2])
        logger.debug("Creating step shooting with params step_dur_time:{} start_rps:{} end_rps:{} step:{}".format(
            step_dur_time, start_rps, end_rps, step))
        return StepShooting(step_dur_time, start_rps, end_rps, step)
    elif ShootingType.CONST in shooting_str:
        assert len(args) == 2
        date = parser.parse(args[1])
        dur_time = date.hour * 3600 + date.minute * 60 + date.second
        rps_val = int(args[0])
        logger.debug("Creating const shooting with params dur_time:{} rps_val:{}".format(
            dur_time, rps_val))
        return ConstShooting(dur_time, rps_val)
    else:
        raise Exception("Unknown shooting type")


class TankApi:
    def __init__(self,
                 target_address,
                 target_host,
                 ammo_file,
                 task,
                 tank_address,
                 logger,
                 tank_port,
                 shooting_type,
                 shooting_component_name,
                 jobno=None,
                 monitoring_script_path='',
                 files_mask_to_copy=['monitoring.log',
                                     'tank*.log', 'api_run.ini',
                                     'saved_conf.ini', 'phantom*.conf',
                                     'lunapark*.lock',
                                     'phantom*.log', 'test_data.log',
                                     'answ*.log', 'load.conf'],
                 job_name='regular loading',
                 operator='',
                 is_regular_performance_test=False,
                 base_service_rps=None):
        self.target_address = target_address
        self.target_host = target_host
        self.ammo_file = ammo_file
        self.task = task
        self.tank_address = tank_address
        self.logger = logger
        self.jobno = jobno
        self.job_name = job_name
        self.operator = operator
        self.tank_address = tank_address
        self.tank_port = tank_port
        self.files_mask_to_copy = files_mask_to_copy
        self.monitoring_script_path = monitoring_script_path
        self.lunapark_api_uri_prefix = 'https://lunapark.yandex-team.ru/api/job/'
        self.shooting_str = shooting_type
        self.shooting_class = create_shooting_class(self.shooting_str, self.logger)
        self.shooting_component_name = shooting_component_name
        self.is_regular_performance_test = is_regular_performance_test
        self.base_service_rps = base_service_rps
        self.monitoring_json = None
        self.aggregate_info_json = None

    def create_tank_config(self):
        self.tank_conf = ConfigParser()

        self.tank_conf.add_section('phantom')
        self.tank_conf.set('phantom', 'rps_schedule', self.shooting_str)
        self.tank_conf.set('phantom', 'address', self.target_address)
        self.tank_conf.set('phantom', 'ammofile', self.ammo_file)

        self.tank_conf.add_section('meta')

        self.tank_conf.set('meta', 'operator', self.operator)
        self.tank_conf.set('meta', 'task', self.task)
        self.tank_conf.set('meta', 'job_name', self.job_name)
        self.tank_conf.set('meta', 'lock_targets', '')

        self.__fill_monitoring_config_section()

        self.logger.info('tank config creating done.')
        return

    def __fill_monitoring_config_section(self):
        startup_cmd = 'python ./yasm_get.py -i 1 --host ' +\
                      self.target_host + ' --pattern ' + self.target_address.split('.')[0] +\
                      ' --prefix ' + self.shooting_component_name +\
                      ' &gt;/tmp/unistat_shooting.log 2&gt;/tmp/unistat_shooting.err'
        self.tank_conf.add_section('telegraf')
        self.tank_conf.set('telegraf', 'disguise_hostnames', '0')
        self.tank_conf.set('telegraf', 'config', '\
        <Monitoring>\n\
        <Host address="localhost">\n\
        <CPU/>\n\
        <Kernel/>\n\
        <Net/>\n\
        <System/>\n\
        <Memory/>\n\
        <Disk/>\n\
        <Netstat/>\n\
        <Nstat/>\n\
        <Startup>' + startup_cmd + '</Startup>\n\
        <Source>/tmp/unistat_shooting.log</Source>\n\
        </Host>\n\
        </Monitoring>')
        return

    def write_tank_config(self, path):
        try:
            self.path_to_config_save = path
            with open(path, 'w') as fd:
                self.tank_conf.write(fd)
            self.logger.info('tank config writing done.')
        except Exception as e:
            self.logger.error('Cant write tank config' + str(e))
            raise

    def start_shooting(self):
        exe_opts = []
        exe_opts.extend(['-t', self.tank_address])
        exe_opts.extend(['-p', self.tank_port])
        exe_opts.extend(['-c', self.path_to_config_save])
        if self.jobno:
            exe_opts.extend(['-j', self.jobno])
        for file_mask in self.files_mask_to_copy:
            exe_opts.extend(['-d', file_mask])

        if self.monitoring_script_path:
            exe_opts.extend(['-f', self.monitoring_script_path])

        cmd = ['tankapi-cmd'] + exe_opts
        try:
            out = sp.check_output(cmd, stderr=sp.STDOUT)
            self.logger.info(out)
        except sp.CalledProcessError as e:
            self.logger.error('CalledProcessError: {} output: {}'.format(str(e), e.output))
            raise
        except ValueError as e:
            self.logger.error('Value error: ' + str(e))
            raise
        except Exception as e:
            self.logger.error('Unknown error: ' + str(e))
            raise
        return 0

    def get_shooting_link(self):
        with open(self.jobno, 'r') as f:
            self.job_number = f.readlines()[-1].strip()
            link = 'https://lunapark.yandex-team.ru/' + self.job_number
            self.logger.debug("Shooting link: " + link)
            return 'https://lunapark.yandex-team.ru/' + self.job_number

    def post_process_logs(self, work_path, src_path):
        if work_path != src_path:
            for file_mask in self.files_mask_to_copy:
                for f in glob.glob(os.path.join(work_path, file_mask)):
                    shutil.move(f, src_path)
        self.logger.info('Post proccess log end.')
        return 0

    def fill_shooting_reports(self):
        self.shooting_link = self.get_shooting_link()
        shooting_class = self.shooting_class
        self.common_info_report_json, self.monitoring_report_json = shooting_class.create_shooting_reports_json(self)

        self.logger.info('All reports was created success.')

    def get_metrics_for_base_rps(self):
        if not self.base_service_rps:
            return {}
        if self.shooting_class.type == ShootingType.LINE:
            return {}
        elif self.shooting_class.type == ShootingType.CONST:
            return {}
        elif self.shooting_class.type == ShootingType.STEP:
            result = {}
            if self.base_service_rps in self.common_info_report_json:
                for metric in self.common_info_report_json[self.base_service_rps]:
                    result[metric] = self.common_info_report_json[self.base_service_rps][metric]
            if self.base_service_rps in self.monitoring_report_json:
                for metric in self.monitoring_report_json[self.base_service_rps]:
                    result[metric] = self.monitoring_report_json[self.base_service_rps][metric]
            return result
        return {}

    def filter_monitoring_report(self, monitoring_json):
        filter_monitoring_report = {}
        for metric in monitoring_json:
            if metric['metric'].startswith('custom') and metric['metric'].find(self.shooting_component_name) != -1:
                for using_metrics_names in ['cpu_guarantee', 'cpu_limit', 'cpu_user', 'mem_guarantee', 'mem_usage', 'disk_write-fs', 'disk_read-fs']:
                    if metric['metric'].find(using_metrics_names) != -1:
                        filter_monitoring_report[using_metrics_names] = metric
        self.logger.info('Monitoring report after filtering:{}'.format(filter_monitoring_report))
        return filter_monitoring_report

    def append_params(self, uri, ts_from=None, ts_to=None):
        params = []
        if ts_from:
            params.append('from={}'.format(ts_from))
        if ts_to:
            params.append('to={}'.format(ts_to))
        if params:
            uri += '?' + '&'.join(params)
        return uri

    def get_time_distribution_aggregate(self):
        return self.__send_get_request('/dist/times.json', 'get_time_distribution_aggregate')

    def get_time_distribution_per_seconds(self):
        return self.__send_get_request('/data/times.json', 'get_time_distribution_per_seconds')

    def get_percentiles(self, ts_from=None, ts_to=None):
        return self.__send_get_request(
            self.append_params('/dist/percentiles.json', ts_from, ts_to), 'get_percentiles')

    def get_http_codes_distribution_aggregate(self, ts_from=None, ts_to=None):
        return self.__send_get_request(
            self.append_params('/dist/http.json', ts_from, ts_to), 'get_http_codes_distribution_aggregate')

    def get_http_codes_distribution_per_seconds(self):
        return self.__send_get_request('/data/http.json', 'get_http_codes_distribution_per_seconds')

    def get_net_codes_distribution_aggregate(self, ts_from=None, ts_to=None):
        return self.__send_get_request(
            self.append_params('/dist/net.json', ts_from, ts_to), 'get_net_codes_distribution_aggregate')

    def get_net_codes_distribution_per_seconds(self):
        return self.__send_get_request('/data/net.json', 'get_net_codes_distribution_per_seconds')

    def get_aggregate_shooting_data(self):
        return self.__send_get_request('/aggregates.json', 'get_aggregate_shooting_data')

    def get_monitoring_info_aggregate(self, ts_from=None, ts_to=None):
        return self.__send_get_request(
            self.append_params('/monitoring.json', ts_from, ts_to), 'get_monitoring_info_aggregate')

    def get_common_shooting_info(self):
        return self.__send_get_request('/summary.json', 'get_common_shooting_info')

    def get_shooting_cases(self):
        return self.__send_get_request('/data/cases.json', 'get_shooting_cases')

    def __send_get_request(self, uri, log_info):
        try:
            s = requests.Session()
            s.mount('https://', HTTPAdapter(max_retries=5))
            r = s.get(self.lunapark_api_uri_prefix + self.job_number + uri)
            self.logger.debug('{} st_code:{} text:{}'.format(log_info, r.status_code, r.text))
            return r.json()
        except requests.ConnectionError as e:
            self.logger.exception('{} connection error: '.format(log_info))
            return {'error': str(e)}
        except Exception:
            self.logger.exception('{} unknown error: '.format(log_info))
            raise
