import datetime
import logging
import string
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskFailure
import sandbox.projects.common.build.YaPackage as YaPackage
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.mail.CommonLib.lib.star_track import StarTrackApi
from sandbox.projects.mail.CommonLib.lib.projects import PROJECTS


JOB_FAIL = 'Fail'
JOB_SUCCESS = 'Success'
KILL_TIMEOUT = 60*60


class TopicString(sdk2.parameters.StrictString):
    regexp = '^[A-Za-z0-9_\-]*$'


class CreateMailRelease(sdk2.Task):
    """ Build and push docker image for mail project"""

    class Context(sdk2.Context):
        registry_tags = []
        build_revision = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        owner = "MAIL"
        description = "Create Mail Release: build and push docker image for mail service and run ya make build and tests"
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        topic = TopicString("Topic for this release (optional, use only ::alnum::, underscore and minus)")
        issues = sdk2.parameters.List("Startrek tasks for notification")
        arcadia_url = sdk2.parameters.ArcadiaUrl("Svn url (you can add '@<commit_number>' or specify branch path)",
                                                 default_value=build_params.ArcadiaUrl.default_value)
        review_id = sdk2.parameters.Integer("Arcanum review number, will apply patch onto arcadia url (optional)",
                                            default_value=None)
        qloud_token_vault_item = sdk2.parameters.String("Vault item containing Qloud OAuth token",
                                                        default_value='robot-gerrit-oauth-platform')
        docker_user = sdk2.parameters.String("Docker registry user",
                                             default_value='robot-gerrit')
        docker_token_vault_item = sdk2.parameters.String("Vault item containing Docker registry token",
                                                         default_value='robot-gerrit-oauth-registry')
        startrack_token_vault_item = sdk2.parameters.String("Vault item containing Startrack oauth token",
                                                            default_value='robot-gerrit-oauth-startrek')
        run_tests = sdk2.parameters.Bool("Run tests", default_value=True)

        tests_retries = sdk2.parameters.Integer("Tests retries", default_value=2)

        with sdk2.parameters.String("Select project to build docker image", required=True,
                                    multiline=True) as mail_project:
            for project in PROJECTS:
                mail_project.values[project.title] = project.title

    def __get_project(self):
        try:
            project = list(filter(lambda x: x.title == self.Parameters.mail_project, PROJECTS))[0]
            return project
        except Exception as e:
            error_message = "Can't find project, error: {}".format(str(e))
            logging.error(error_message)
            raise Exception(error_message)

    def _get_docker_custom_version(self, topic, review_id):
        timestamp = datetime.datetime.now().strftime("%Y%m%d-%H%M")
        revision = str(self.Context.build_revision).replace('/', '_')
        custom_version = "{}.r{}".format(timestamp, revision)
        if review_id:
            custom_version += '-{}'.format(review_id)
        if topic:
            custom_version += ".{}".format(topic)
        return custom_version

    def __get_project_attribute(self, attr_name):
        try:
            return getattr(self.__get_project(), attr_name)
        except Exception as e:
            error_message = "Can't get project attribute with name {} error: {}".format(attr_name, str(e))
            logging.error(error_message)
            raise Exception(error_message)

    def _get_generated_custom_tag(self):
        return self.Context.registry_tags[0]

    def __ya_package_docker(self):
        task_class = sdk2.Task['YA_PACKAGE']
        custom_version = self._get_docker_custom_version(self.Parameters.topic, self.Parameters.review_id)
        self.Context.registry_tags.append("{}:{}".format(self.__get_project().registry_image, custom_version))
        logging.info("The registry tag for updating environment is {}".format(self._get_generated_custom_tag()))

        kwargs = {
            build_params.ArcadiaUrl.name: self.__arcadia_url,
            build_params.ArcadiaPatch.name: self.__arcadia_patch,
            build_params.BuildType.name: 'release',
            build_params.UseArcadiaApiFuse.name: True,
            YaPackage.PackagesParameter.name: self.__get_project_attribute('package_path'),
            YaPackage.PackageTypeParameter.name: YaPackage.DOCKER,
            YaPackage.DockerPushImageParameter.name: True,
            YaPackage.DockerRegistryParameter.name: "registry.yandex.net",
            YaPackage.DockerUserParameter.name: self.Parameters.docker_user,
            YaPackage.DockerTokenVaultName.name: self.Parameters.docker_token_vault_item,
            YaPackage.DockerImageRepositoryParameter.name: self._get_repository(self.__get_project().registry_image),
            YaPackage.CustomVersionParameter.name: custom_version
        }
        sub_task = task_class(
            self,
            kill_timeout=KILL_TIMEOUT,
            description="Subtask for " + self.Parameters.mail_project + ' build docker image',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        return sub_task

    def _get_repository(self, registry_image):
        tags = registry_image.split("/")

        return "/".join(tags[:-1])

    def __ya_make_subtask(self):
        kwargs = {
            build_params.BuildType.name: 'release',
            build_params.ArcadiaUrl.name: self.__arcadia_url,
            build_params.ArcadiaPatch.name: self.__arcadia_patch,
            get_arcadia_project_base_target_params().TargetsParameter.name: self.__get_project_attribute('build_path'),
            build_params.TestParameter.name: self.Parameters.run_tests,
            build_params.TestsRetriesCount.name: self.Parameters.tests_retries,
            build_params.UseArcadiaApiFuse.name: True,
            build_params.EnvironmentVarsParam.name: self.__get_project_attribute('yamake_env_vars'),
            build_params.DefinitionFlags.name: self.__get_project_attribute('yamake_definition_flags')
        }
        task = self.server.task(
            type='YA_MAKE',
            description="Ya Make Subtask for " + self.Parameters.mail_project + ' (run build and tests)',
            owner=self.Parameters.owner,
            custom_fields=[{'name': k, 'value': v} for k, v in kwargs.items()],
            parent={"id": self.id},
            requirements={"platform": 'linux_ubuntu_16.04_xenial'},
            children=True,
            kill_timeout=KILL_TIMEOUT,
            priority=self.Parameters.priority,
            notifications=[
                {"statuses": nt.statuses, "recipients": nt.recipients, "transport": nt.transport}
                for nt in self.Parameters.notifications
            ],
        )
        logging.info('Task states: %s', self.server.batch.tasks.start.update([task["id"]]))
        return task

    def __update_qloud_environment_subtask(self):
        task_class = sdk2.Task['UPDATE_QLOUD_ENVIRONMENT']
        kwargs = {
            'image_url': '/'.join(['registry.yandex.net', self._get_generated_custom_tag()]),
            'components': self.__get_project_attribute('qa_components'),
            'qloud_token_vault_item': self.Parameters.qloud_token_vault_item,
            'st_token_vault_item': self.Parameters.startrack_token_vault_item,
            'deploy_comment': 'Deploy by create mail release sandbox job',
            'platform': 'Stable (platform)',
            'issues': self.Parameters.issues
        }
        sub_task = task_class(
            self,
            kill_timeout=KILL_TIMEOUT,
            description="Deploy qloud env Subtask for " + self.Parameters.mail_project + ' (deploy prestable env)',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        return sub_task

    def __comment_startrek(self):
        tag = self._get_generated_custom_tag()
        if self.__get_job_status() == JOB_SUCCESS:
            comment = "!!(green)**SUCCESS**!!"
        else:
            comment = "!!**ERROR**!!"
        comment += " Create Mail Release for " + self.Parameters.mail_project + '\n'
        comment += "**CommonInfo**:\n"
        comment += "Registry url: %%registry.yandex.net/{tag}%%\n".format(tag=tag)
        comment += "**SubTasksInfo**:\n"
        for sub_task in self.find():
            comment += "**" + str(sub_task.type) + "** "
            if sub_task.status == ctt.Status.SUCCESS:
                comment += "!!(green)**" + str(sub_task.status) + "**!!\n"
            else:
                comment += "!!**" + str(sub_task.status) + "**!!\n"
        logging.info("startrek comment: " + str(comment))

        st_api = StarTrackApi(sdk2.Vault.data(self.Parameters.startrack_token_vault_item))
        errors = st_api.add_comments(issues=self.Parameters.issues, text=comment, task_id=self.id)
        logging.error("Errors while commenting star track: {}".format(str(errors)))

    def __get_job_status(self):
        if not all(map(lambda sub_task: sub_task.status == ctt.Status.SUCCESS, self.find(children=True))):
            return JOB_FAIL
        else:
            return JOB_SUCCESS

    def on_prepare(self):

        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)
        if not parsed_url.revision:
            self.Context.build_revision = Arcadia.get_revision(self.Parameters.arcadia_url)
            logging.info('Revision for build docker image and ya make subtasks: ' + str(self.Context.build_revision))
            self.__arcadia_url = self.Parameters.arcadia_url + '@' + str(self.Context.build_revision)
        else:
            self.Context.build_revision = parsed_url.revision
            logging.info('Revision for build docker image and ya make subtasks: ' + str(parsed_url.revision))
            self.__arcadia_url = self.Parameters.arcadia_url
        self.__arcadia_patch = "arc:{}".format(self.Parameters.review_id) if self.Parameters.review_id else None

    def on_execute(self):
        with self.memoize_stage.run_docker_image_build_and_ya_make_with_tests:
            sub_tasks = []
            ya_make_sub_task = self.__ya_make_subtask()
            sub_tasks.append(ya_make_sub_task["id"])
            ya_package_docker_sub_task = self.__ya_package_docker()
            sub_tasks.append(ya_package_docker_sub_task.id)
            raise sdk2.WaitTask(sub_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.deploy_project:
            if self.__get_job_status() == JOB_SUCCESS and self.__get_project_attribute('qa_components'):
                deploy_project_sub_task = self.__update_qloud_environment_subtask()
                raise sdk2.WaitTask([deploy_project_sub_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.startrek_comment:
            if self.Parameters.issues:
                self.__comment_startrek()

        with self.memoize_stage.finish_job:
            if self.__get_job_status() == JOB_FAIL:
                failed_sub_tasks = filter(lambda sub_task: sub_task.status != ctt.Status.SUCCESS,
                                          self.find(children=True))
                failed_tasks_ids_types = map(lambda sub_task: [sub_task.id, sub_task.type], failed_sub_tasks)
                raise Exception("Following subtasks failed: " + str(failed_tasks_ids_types))
