# -*- coding: utf-8 -*-
import logging
import datetime
import time
import requests
from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
from sandbox.projects.common import link_builder as lb
from sandbox.projects.mail.CommonLib.lib.star_track import StarTrackApi
from sandbox.projects.mail.CommonLib.lib.projects import PROJECTS


JOB_FAIL = 'Fail'
JOB_SUCCESS = 'Success'
KILL_TIMEOUT = 90*60


class DeployMailRelease(sdk2.Task):
    """Deploy new version of mail microservice"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = KILL_TIMEOUT
        owner = "MAIL"
        description = "DeployMailRelease: upgrade mail service production and send notifications"
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        qloud_token_vault_item = sdk2.parameters.String("Vault item containing Qloud OAuth token",
                                                        default_value='robot-gerrit-oauth-platform')
        issues = sdk2.parameters.List("Startrek tasks for notifications")
        emails = sdk2.parameters.List("Emails list for notifications")
        summary = sdk2.parameters.String("Changelog of this release for notifications", required=True, multiline=True)
        version = sdk2.parameters.String('Version to deploy (docker tag)', required=True)

        with sdk2.parameters.String("Select project to deploy", required=True,
                                    multiline=True) as mail_project:
            for project in PROJECTS:
                mail_project.values[project.title] = project.title

    def __image_tag(self):
        return "registry.yandex.net/{}:{}".format(self.__get_project().registry_image, self.Parameters.version)

    def __get_project(self):
        try:
            project = filter(lambda x: x.title == self.Parameters.mail_project, PROJECTS)[0]
            return project
        except Exception as e:
            error_message = "Can't find project, error: {}".format(str(e))
            logging.error(error_message)
            raise Exception(error_message)

    def __get_project_attribute(self, attr_name):
        try:
            return getattr(self.__get_project(), attr_name)
        except Exception as e:
            error_message = "Can't get project attribute with name {} error: {}".format(attr_name, str(e))
            logging.error(error_message)
            raise Exception(error_message)

    def __get_job_status(self):
        if not all(map(lambda sub_task: sub_task.status == ctt.Status.SUCCESS, self.find(children=True))):
            return JOB_FAIL
        else:
            return JOB_SUCCESS

    def __update_qloud_environment_subtask(self, components, components_type):
        task_class = sdk2.Task['UPDATE_QLOUD_ENVIRONMENT']
        kwargs = {
            'image_url': self.__image_tag(),
            'components': components,
            'qloud_token_vault_item': self.Parameters.qloud_token_vault_item,
            'deploy_comment': 'Deployed by DeployMailRelease sandbox job {}'.format(lb.task_link(self.id, plain=True)),
            'platform': 'Stable (platform)',
        }
        sub_task = task_class(
            self,
            kill_timeout=KILL_TIMEOUT,
            description="Subtask for deploying {} to {}".format(self.Parameters.mail_project, components_type),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        return sub_task

    def __comment_startrek(self):
        if self.__get_job_status() == JOB_SUCCESS:
            comment = "!!(green)**SUCCESS**!! DeployMailRelease finished successfully"
        else:
            comment = "!!**ERROR**!! DeployMailRelease failed"
        comment += " for **{prj}** %%{version}%%\n".format(prj=self.__get_project_attribute('title'), version=self.Parameters.version)
        comment += "**Summary:**\n{summary}\n".format(summary=self.Parameters.summary.encode('utf-8'))
        if self.Parameters.issues:
            comment += "**Tasks:**\n"
            for issue in self.Parameters.issues:
                comment += "{}\n".format(issue)
        comment += "**Children jobs:**\n"
        for sub_task in self.find():
            comment += lb.task_link(sub_task.id, plain=True) + "\n"

        st_api = StarTrackApi(sdk2.Vault.data("robot-gerrit-oauth-startrek"))
        errors = st_api.add_comments(issues=self.Parameters.issues, text=comment, task_id=self.id)
        if errors:
            logging.error("Errors while commenting star track: {}".format(str(errors)))

    def __send_email(self):
        if self.__get_job_status() != JOB_SUCCESS:
            return
        fmt_summary = self.Parameters.summary.replace('\n', '<br/>').encode('utf-8')
        body = "<b>Описание:</b><br/>{summary}<br/>".format(summary=fmt_summary)
        if self.Parameters.issues:
            body += "<br/><b>Задачи:</b><br/>"
            for issue in self.Parameters.issues:
                body += '<a href="https://st.yandex-team.ru/{issue}">{issue}</a><br/>'.format(issue=issue)
        body += "<br/>----<br/>Отправлено из sandbox-джобы {}".format(lb.task_link(self.id, plain=False)) if self.id else ""
        self.server.notification(
            subject='Выкатился релиз {prj} {version}'.format(prj=self.__get_project_attribute('title'), version=self.Parameters.version),
            body=body,
            recipients=self.Parameters.emails,
            transport=ctn.Transport.EMAIL,
            type='html'
        )

    def __notify_telegram(self):
        url = "http://tgbot.mail.yandex.net/notify_deploy"
        data = {
            "status": "DEPLOYED",
            "environmentId": self.__get_project_attribute('title'),
            "comment": "Some comment",
            "version": self.Parameters.version,
            "date": datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            "statusChangeTs": int(time.time()),
        }
        requests.post(url, json=data)

    def on_execute(self):
        with self.memoize_stage.deploy_canary:
            canary_components = self.__get_project_attribute('canary_components')
            if canary_components:
                deploy_subtask = self.__update_qloud_environment_subtask(canary_components, 'canary')
                raise sdk2.WaitTask([deploy_subtask.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.notify:
            if self.Parameters.emails:
                self.__send_email()
            self.__notify_telegram()
            if self.Parameters.issues:
                self.__comment_startrek()
