from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.projects.tank.load_resources.resources import AMMO_FILE

import logging
import time
import sys
import os
import os.path
import requests
import random

cfg = [
    {'name': 'blackwhitelist', 'weight': 55, 'params': None},
    {'name': 'multi_list', 'weight': 39, 'params': {0: 25, 1: 70, 5: 4, 20: 1}},
    {'name': 'list.json', 'weight': 4, 'params': None},
    {'name': 'unsubscribe-newsletters', 'weight': 2, 'params': None},
    {'name': 'unsubscribe-filters', 'weight': 0, 'params': None}
]


class FuritaAmmoGen(sdk2.Task):

    class Context(sdk2.Task.Context):
        ammo_resource = ''
        ammo_command = ''

    class Requirements(sdk2.Requirements):
        disk_space = 512    # 500 MiB on disk
        cores = 1           # exactly 1 core
        ram = 1024          # 1GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Ammo parameters') as tankapi_block:
            description = sdk2.parameters.String('Comment for generation')
            users_file = sdk2.parameters.String('Shot for catridges', default='https://proxy.sandbox.yandex-team.ru/1209803758')
            ammo_count = sdk2.parameters.Integer('Capacity of the magazine', default=10000)

        with sdk2.parameters.Output:
            ammo = sdk2.parameters.Resource('Ammo', required=True)

    def make_ammo(self):
        ammo = AMMO_FILE(self, 'My ammo', 'ammo.json', ttl=1)
        ammoData = sdk2.ResourceData(ammo)
        ammoData.path.write_bytes(self.ammo_data())
        ammoData.ready()
        self.Context.ammo_resource = str(ammoData.path)
        return ammo

    def ammo_data(self):
        data = ""
        weighted = []
        for handler in cfg:
            weighted += [(handler['name'], handler['params'])] * handler['weight']
        for i in range(self.Parameters.ammo_count):
            handler, params = random.choice(weighted)
            if data == "":
                data += self.ammo_line(handler, params=params)
            else:
                data += "\n" + self.ammo_line(handler, params=params)
        return data

    def ammo_line(self, handler, params=None):
        uid, white_black_count, rules_count = random.choice(self.users)
        tag = ''
        if handler == 'blackwhitelist':
            uri = '/api/blackwhitelist?uid=%d' % uid
            tag = '%d_wb' % white_black_count
        elif handler == 'list.json':
            uri = '/api/list.json?detailed=1&type=user&uid=%d&db=pg&user=%d&connection_id=LOAD-%05d' % (uid, uid, random.randint(0, 10000))
            tag = '%d_rules' % rules_count
        elif handler == 'unsubscribe-newsletters':
            uri = '/unsubscribe/newsletters?uid=%d&db=pg&user=%d&connection_id=%05d' % (uid, uid, random.randint(0, 10000))
        elif handler == 'unsubscribe-filters':
            uri = '/unsubscribe/filters?uid=%d&db=pg&user=%d&connection_id=LOAD-%05d' % (uid, uid, random.randint(0, 10000))
        elif handler == 'multi_list':
            counts = []
            for c in params:
                counts += [c] * params[c]
            count = random.choice(counts)
            uri, tag = self.gen_multi(count)
        return '%s %s_%s' % (uri, handler, tag)

    def gen_multi(self, count):
        if count > len(self.users):
            raise ce.TaskFailure("Not enough uids for list %d", count - len(self.users))
        uids = [str(i[0]) for i in random.sample(self.users, count)]
        uids_str = ','.join(uids)
        return ('/api/multi_list/?uids=%s' % uids_str,  '%d_users' % count)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start AMMO generator")
        self.users = []
        with open(get_source(self.Parameters.users_file, 'users.source'), 'r') as uf:
            for line in uf.readlines():
                email, uid, white_black_count, rules_count = line.strip().split(',')
                self.users.append((int(uid), int(white_black_count), int(rules_count)))
        self.Parameters.ammo = self.make_ammo()


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'furita_ammo.log',
        maxBytes=(1024 * 1024),
        backupCount=5)
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
