# coding: utf-8
from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.mail.Load.FuritaAmmoGen import FuritaAmmoGen
from sandbox.projects.mail.Load.ShootingComparison import ShootingComparison
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults

import logging
import time
import os
import os.path
import yaml
import requests


descs = {
    "4load": u'\u0428\u0430\u0431\u043b\u043e\u043d ' +
        '\u043a\u043e\u043d\u0444\u0438\u0433\u0443\u0440\u0430\u0446\u0438\u043e\u043d\u043d\u043e\u0433\u043e \u0444\u0430\u0439\u043b\u0430 ' +
        '\u0434\u043b\u044f \u043d\u0430\u0433\u0440\u0443\u0437\u043e\u0447\u043d\u043e\u0433\u043e \u0442\u0435\u0441\u0442\u0430',
    "4users": u'\u0424\u0430\u0439\u043b \u043c\u0435\u0442\u0430\u0434\u0430\u043d\u043d\u044b\u0445 \u0434\u043b\u044f ' +
        '\u0433\u0435\u043d\u0435\u0440\u0430\u0446\u0438\u0438 \u0022\u043f\u0430\u0442\u0440\u043e\u043d\u043e\u0432\u0022 ' +
        '\u0434\u043b\u044f \u043d\u0430\u0433\u0440\u0443\u0437\u043e\u0447\u043d\u043e\u0433\u043e \u0442\u0435\u0441\u0442\u0430'
}


class FuritaShooting(sdk2.Task):

    class Context(sdk2.Task.Context):
        ammo_resource = ""
        ammo_url = ""
        ammo_generator_task_id = ""
        ammo_generator_find_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        comparison_task_id = ""
        send_comment_task_id = ""
        lxc_container = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            host = sdk2.parameters.String('Target host', default='furita.load.furita.mail.stable.qloud-d.yandex.net', required=True)
            scheduler = sdk2.parameters.String('Load profile', default='line(1, 500, 300s)', required=True)
            suffix = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(95,150ms,10s)', 'negative_http(2xx,10%,10s)'])
            rps = sdk2.parameters.List('RPS', default=['RPS_READ_THRPT=1000', 'RPS_WRITE_THRPT=100', 'RPS_READ_RT=500', 'RPS_WRITE_RT=50'])

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            load_file = sdk2.parameters.String('Template shooting config', description=descs["4load"].encode('utf-8'), default='https://proxy.sandbox.yandex-team.ru/1232127913', required=True)
            users_file = sdk2.parameters.String('Resource for ammo generator', description=descs["4users"].encode('utf-8'), default='https://proxy.sandbox.yandex-team.ru/1209803758', required=True)

        with sdk2.parameters.Group('The choice of the tank') as tank_choice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                use_tank.values['mimas.tanks.mail.yandex.net'] = use_tank.Value(value='MIMAS')
                use_tank.values['rhea.tanks.mail.yandex.net'] = use_tank.Value(value='RHEA')
                use_tank.values['iapet.tanks.mail.yandex.net'] = use_tank.Value(value='IAPET', default=True)

        with sdk2.parameters.Group('Parameters for comparing of the shooting') as comparing_block:
            comparing = sdk2.parameters.Bool('To comparing the shooting?', default=True)
            with comparing.value[True]:
                rfile = sdk2.parameters.String('File with the reference shootings', default='https://proxy.sandbox.yandex-team.ru/1275363195')
                threshold = sdk2.parameters.Integer('Permissible deviation in %', default=5)
                ticket = sdk2.parameters.String('Ticket for comment', default='MAILDLV-3170')
                st_token = sdk2.parameters.String('Vault record name with startrek token', default='lunapark-startrek-token')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value="")
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value="")

    def start_shooting(self, desc, ammo_resource, config_content):
        tanks = list()
        tanks.append(self.Parameters.use_tank)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        self.Context.lxc_container = str(container)
        self.Context.save()
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='resource',
            ammo_resource=ammo_resource,
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            container=container
        ).enqueue()

        self.Context.shooting_task_id = str(shooting_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def make_ammo(self):
        self.loger.info('Start ammo generation')
        ammo_gen_task = FuritaAmmoGen(
            self,
            description='ammo generator for task {0}'.format(self.id),
            users_file=self.Parameters.users_file,
            ammo_count=40000).enqueue()
        self.Context.ammo_generator_task_id = str(ammo_gen_task.id)
        self.Context.save()
        self.loger.info('ammo_generator_task_id = %s', ammo_gen_task.id)
        self.loger.info('Starting subtask: ' + ammo_gen_task.Parameters.description)
        raise sdk2.WaitOutput({ammo_gen_task.id: 'ammo'}, wait_all=True, timeout=3600)

    def get_ammo(self):
        ammo_gen_find = self.find(
            FuritaAmmoGen,
            id=self.Context.ammo_generator_task_id).first()
        self.Context.ammo_generator_find_id = str(ammo_gen_find.id)
        self.loger.info('found ammo_generator_task_id = %s', ammo_gen_find.id)
        return ammo_gen_find.Parameters.ammo

    def make_conf(self):
        autostop = []
        config_content = fu.read_file(get_source(self.Parameters.load_file, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)

        yaml_config['phantom']['address'] = self.Parameters.host.encode('utf-8')
        yaml_config['phantom']['headers'] = ['Host: {}'.format(self.Parameters.host).encode('utf-8')]
        yaml_config['phantom']['load_profile']['schedule'] = self.Parameters.scheduler.encode('utf-8')
        yaml_config['yasm']['panels']['tank']['tags'] = "itype=qloud;prj=mail.tanks.mail-tanks;tier={}".format(self.Parameters.use_tank.split('.')[0]).encode('utf-8')
        yaml_config['uploader']['job_name'] = "[furita][{}][{}]".format(self.Parameters.scheduler, self.Parameters.suffix).encode('utf-8')
        yaml_config['uploader']['meta']['use_tank'] = self.Parameters.use_tank.encode('utf-8')
        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def comparing_shooting(self):
        comparison_task = ShootingComparison(
            self,
            sid=self.Parameters.lunapark_job_id,
            rfile=self.Parameters.rfile
        ).enqueue()

        self.Context.comparison_task_id = str(comparison_task.id)
        raise sdk2.WaitTask([comparison_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def comparison_result(self):
        comparison_find = sdk2.Task.find(
            id=self.Context.comparison_task_id,
            children=True
        ).first()
        comparison_result = comparison_find.Parameters.comparison_result
        shooting_type = comparison_find.Parameters.shooting_type
        # Build comment for ticket
        header = "Regression"
        footer = "https://wiki.yandex-team.ru/Load/"
        comment_for_ticket = "((@lix0 Load Support))"
        comparison_status = comparison_result.split("\n")[-1]
        comparison_body = "\n".join(comparison_result.split("\n")[:-1])
        comment_for_ticket += "\n<{%s\n#|\n||%s||\n|#\n%s}>\n%s" % (
            header,
            comparison_body.replace("\n", "||\n||").replace("passed", "!!(green)passed!!").replace("failed", "!!(red)failed!!").replace("improved", "!!(yellow)improved!!"),
            footer,
            comparison_status.replace("passed", "**!!(green)SUCCESS!!**").replace("failed", "**!!(red)FAILED!!**")
        )
        self.Context.comment_for_ticket = comment_for_ticket

        send_comment_task = LoadTestResults(
            self,
            shoot_id=self.Parameters.lunapark_job_id,
            report_type=shooting_type,
            additional_message=comment_for_ticket,
            ticket_id=self.Parameters.ticket,
            st_token_name=self.Parameters.st_token,
            send_comment=True
        ).enqueue()

        self.Context.send_comment_task_id = str(send_comment_task.id)
        self.loger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([send_comment_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start Furita shooting")
        desc = self.Parameters.suffix

        with self.memoize_stage.make_ammo:
            self.make_ammo()

        with self.memoize_stage.make_conf:
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            ammo_resource = self.get_ammo()
            self.Context.ammo_url = str(ammo_resource.http_proxy)
            self.start_shooting(desc, ammo_resource, config_content)

        if self.Context.shooting_task_id != "":
            self.find_shooting()

        # Evaluate shooting and send comment withs the result of evaluation into ticket
        if self.Parameters.comparing and self.Parameters.lunapark_job_id != "":
            if self.Context.comparison_task_id == "":
                self.comparing_shooting()
            elif self.Context.send_comment_task_id == "":
                self.comparison_result()
        #  ========== End Of Class ==========  #


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'furita_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5)
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
