from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import file_utils as fu

from sandbox.projects.mail.Load.WebApiAmmoGen import WebApiAmmoGen
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

import yaml
import logging
import os.path
import requests
import time
import ast
import os

DefaultProfile = {
    "versions": 7700,
    "service-worker-config": 1900,
    "root": 1600,
    "service-worker": 450,
    "print": 14,
    "callback-json": 10,
    "fatal": 2
}


class LizaShooting(sdk2.Task):

    class Context(sdk2.Task.Context):
        ammo_resource = ""
        ammo_url = ""
        ammo_generator_task_id = ""
        ammo_generator_find_id = ""
        shooting_task_id = ""
        shooting_find_id = ""
        lxc_container = ""

    class Requirements(sdk2.Requirements):
        disk_space = 1024   # 1GiB on disk
        cores = 1           # exactly 1 core
        ram = 2048          # 2GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Shooting parameters') as tankapi_block:
            task = sdk2.parameters.String('Lead ticket', default='DARIA-66859')
            comment = sdk2.parameters.String('Comment on the shooting', default='Sandbox shooting')
            target = sdk2.parameters.String('Target ES', default='mail_liza_load.liza', required=True)
            webapi = sdk2.parameters.String('Web-api ES', default='mail_webapi_load.webapi', required=True)
            ssl = sdk2.parameters.Bool('SSL shooting', default=True)
            scheduler = sdk2.parameters.List('Load profile', default=['{"duration": "300s", "type": "const", "ops": 300}'])
            pandora = sdk2.parameters.String(
                'Pandora binary link', default='https://proxy.sandbox.yandex-team.ru/3189577342', required=True)
            profile = sdk2.parameters.JSON('Shooting profile', default=DefaultProfile, required=True)
            autostop = sdk2.parameters.List('Autostop requirements', default=['quantile(90,500ms,20s)', 'negative_http(2xx,20%,20s)'])
            instances = sdk2.parameters.Integer('Pandora instances', default=2000, required=True)

        with sdk2.parameters.Group('Uploaded files') as sources_block:
            load_file = sdk2.parameters.String('Template shooting config', default='https://proxy.sandbox.yandex-team.ru/3192395169')
            users_file = sdk2.parameters.String('Resource for ammo generator', default='https://proxy.sandbox.yandex-team.ru/1210900337')

        with sdk2.parameters.Group('The choice of the tank') as tank_coice_block:
            with sdk2.parameters.RadioGroup('Use tank', required=True) as use_tank:
                use_tank.values['mimas.tanks.mail.yandex.net'] = use_tank.Value(value='MIMAS', default=True)
                use_tank.values['rhea.tanks.mail.yandex.net'] = use_tank.Value(value='RHEA')
                use_tank.values['iapet.tanks.mail.yandex.net'] = use_tank.Value(value='IAPET')

        with sdk2.parameters.Output:
            lunapark_job_id = sdk2.parameters.String('Lunapark job id', default_value='')
            lunapark_link = sdk2.parameters.String('Lunapark link', default_value='')

    def start_shooting(self, desc, config_content):
        tanks = []
        tanks.append(self.Parameters.use_tank)
        container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=ctr.State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
        if container is not None:
            self.Context.lxc_container = str(container)
        shooting_task = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='in_config',
            config_source='file',
            config_content=config_content,
            tanks=tanks,
            container=container
            ).enqueue()
        self.loger.info('Subtask with shooting is started')
        self.Context.shooting_task_id = str(shooting_task.id)
        raise sdk2.WaitTask([shooting_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True, timeout=14400)

    def find_shooting(self):
        shooting_find = sdk2.Task.find(
            id=self.Context.shooting_task_id,
            children=True
        ).first()
        self.Context.shooting_find_id = str(shooting_find.id)
        self.loger.info('found ammo_generator_task_id = %s', shooting_find.id)
        self.Parameters.lunapark_job_id = shooting_find.Parameters.lunapark_job_id
        self.Parameters.lunapark_link = shooting_find.Parameters.lunapark_link

    def make_ammo(self):
        self.loger.info('Start ammo generation')
        ammo_gen_task = WebApiAmmoGen(
            self,
            description='ammo generator for task {0}'.format(self.id),
            source=self.Parameters.users_file,
            profile=self.Parameters.profile,
            wmi=self.es_to_fqdn(self.Parameters.webapi),
        ).enqueue()

        self.Context.ammo_generator_task_id = str(ammo_gen_task.id)
        self.Context.save()
        raise sdk2.WaitOutput({ammo_gen_task.id: 'ammo'}, wait_all=True, timeout=3600)

    def get_ammo(self):
        ammo_gen_find = self.find(id=self.Context.ammo_generator_task_id).first()
        self.Context.ammo_generator_find_id = str(ammo_gen_find.id)
        self.loger.info('found ammo_generator_task_id = %s', ammo_gen_find.id)
        return ammo_gen_find.Parameters.ammo

    def make_conf(self):
        autostop = []
        scheduler = []
        config_content = fu.read_file(get_source(self.Parameters.load_file, 'load.yaml'))
        yaml_config = yaml.safe_load(config_content)
        # Modify config for current task
        yaml_config['pandora']['pandora_cmd'] = self.Parameters.pandora.encode('utf-8')
        yaml_config['pandora']['config_content']['pools'][0]['id'] = "liza"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['type'] = "liza"
        yaml_config['pandora']['config_content']['pools'][0]['gun']['target'] = ':'.join([self.es_to_fqdn(self.Parameters.target).encode('utf-8'), '443' if self.Parameters.ssl else '80'])
        yaml_config['pandora']['config_content']['pools'][0]['gun']['ssl'] = self.Parameters.ssl
        yaml_config['pandora']['config_content']['pools'][0]['startup']['times'] = self.Parameters.instances
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['type'] = "liza_provider"
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['source']['path'] = "./ammo.json"
        yaml_config['pandora']['config_content']['pools'][0]['ammo']['source']['type'] = "file"
        yaml_config['pandora']['resources'][0]['src'] = self.Context.ammo_url
        yaml_config['pandora']['resources'][0]['dst'] = './ammo.json'
        yaml_config['yasm']['panels']['liza']['tags'] = "itype=mail;stage=mail_liza_load;deploy_unit={}".format(self.Parameters.target.split('.')[1]).encode('utf-8')
        yaml_config['yasm']['panels']['tank']['tags'] = "itype=mail_load;stage=mail_load_tanks;deploy_unit={}".format(self.Parameters.use_tank.split('.')[0]).encode('utf-8')
        yaml_config['uploader']['job_name'] = "[LIZA][{}][{}]".format(self.Parameters.scheduler, self.Parameters.comment).encode('utf-8')
        yaml_config['uploader']['task'] = self.Parameters.task.encode('utf-8')

        for condition in self.Parameters.autostop:
            autostop.append(condition.encode('utf-8'))
        yaml_config['autostop']['autostop'] = autostop
        for rps in self.Parameters.scheduler:
            scheduler.append(ast.literal_eval(rps.encode('utf-8')))
        yaml_config['pandora']['config_content']['pools'][0]['rps'] = scheduler
        self.loger.info("scheduler is {}".format(yaml_config['pandora']['config_content']['pools'][0]['rps']))

        return yaml.dump(yaml_config, default_flow_style=False, encoding=None)

    def on_execute(self):
        self.loger = logger()
        self.loger.info("Start liza shooting")

        desc = self.Parameters.comment

        with self.memoize_stage.make_ammo:
            self.make_ammo()

        with self.memoize_stage.make_conf:
            ammo_resource = self.get_ammo()
            self.Context.ammo_resource = str(ammo_resource.id)
            self.Context.ammo_url = str(ammo_resource.http_proxy)
            config_content = self.make_conf()

        with self.memoize_stage.shooting:
            self.start_shooting(desc, config_content)

        self.find_shooting()
        # ==== End Of Class ==== #

    def es_to_fqdn(self, endpoint_set_id):
        dcs = ['iva', 'myt', 'sas', 'vla']
        fqdns = []

        for dc in dcs:
            res = requests.get('http://sd.yandex.net:8080/resolve_endpoints/json', json={
                'cluster_name': dc,
                'endpoint_set_id': endpoint_set_id,
                'client_name': 'mail_load',
            })
            print(res.json())
            if 'endpoint_set' in res.json():
                if 'endpoints' in res.json()['endpoint_set']:
                    for endpoint in res.json()['endpoint_set']['endpoints']:
                        fqdns.append(endpoint['fqdn'])

        if len(fqdns) == 0:
            raise ce.TaskFailure("[ERROR] Cannot get endpoints for endpoint_set {es}".format(es=endpoint_set_id))

        return fqdns[0]


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'liza_shooting.log',
        maxBytes=(1024 * 1024),
        backupCount=5
    )
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def get_source(url, dst):
    session = requests.session()
    try:
        with open(dst, 'wb') as resource:
            resource.write(session.get(url, stream=True).content)
        return os.path.abspath(dst)
    except Exception as ex:
        raise ce.TaskFailure("Can't download resource. {}".format(ex))
    finally:
        session.close()
